/**********************************************************************
 *                                                                    *
 *       (c) Copyright 2003                                           *
 *       All rights reserved                                          *
 *       Programs written by Jianghui Liu (NJIT)                      *
 *                                                                    *
 *       Permission to use, copy, modify, and distribute this         *
 *       software and its documentation for any purpose and without   *
 *       fee is hereby granted, provided that this copyright          *
 *       notice appears in all copies.   Programmer(s) makes no       *
 *       representations about the suitability of this                *
 *       software for any purpose.  It is provided "as is" without    *
 *       express or implied warranty.                                 *
 *                                                                    *
 *       Revised on:            12/08/2003                            *
 *                                                                    *
 *      STABLE VERSION:         10/17/2003                            *
 *      Revised on:             10/16/2003                            *
 *      Revised on:             09/24/2003                            *
 *      Revised on:             09/18/2003                            *
 *      Created on:             08/06/2003                            *
 **********************************************************************/
package RNA;
import java.util.*;
import java.io.*;
import java.math.BigInteger;

public class RNA
{
    private String        RNAName;
    private String        sequence;
    private String        remark;
    private int[]         basePairs;       // this is the matrix bond-representation of the RNA.

    private ArrayList     postOrderPairs;  // bases and base-pairs are sorted by order of 3'-end.
                                           // each base or base-pair is stored as an integer:
                                           // for single base, its sequence index is stored;
                                           // for base-pair like (i, j), the stored integer is
                                           // (i+1) * Length + j.
    private ArrayList     singleCompo, pairCompo;
    //#################################################################
    public int seq_no;
    public int str_no;
    public String conserved;
    public double [] factors_for_conserved= new double[1000];
    public int flag;	//flag = 1 shows tht factor file is provided
    //################################################################
    public RNA(String aSequence)   // this constructor will form a primary structure
    {
        RNAName  = "Unknown";
        sequence = new String(aSequence);
        remark   = "";

        basePairs = new int[sequence.length()];
        for(int i=0; i<basePairs.length; i++)
            basePairs[i] = -1;
    }

    public RNA(String name, String aSequence)  // a primary structure with name
    {
        RNAName  = new String(name);
        sequence = new String(aSequence);
        remark   = "";

        basePairs = new int[aSequence.length()];
        for(int i=0; i<basePairs.length; i++)
            basePairs[i] = -1;
    }

    public RNA(String name, String aSequence, String aRemark)
    {
        RNAName  = new String(name);
        sequence = new String(aSequence);
        remark   = new String(aRemark);

        basePairs = new int[aSequence.length()];
        for(int i=0; i<basePairs.length; i++)
            basePairs[i] = -1;
    }

    public RNA(String name, String aSequence, int[] bonds, String comment)
    {
        remark = new String(comment);
        setup(name, aSequence, bonds);
    }

    public RNA(String name, String aSequence, int[] bonds)
    {
        remark = "";
        setup(name, aSequence, bonds);
    }

    public RNA(String name, String aSequence, String structLine, String aRemark, String aconserved)
    {
        int[] bonds = new int[structLine.length()];
        int cur = -1;
	flag =0; ////###############################Make flag = 0 coz no factors file provided.##########################
        for(int i=0; i<bonds.length; i++) {
           String s = structLine.substring(i, i+1);
           if( s.equals("(") || s.equals("<") ) {
               bonds[i] = cur;
               cur = i;
           } else if( s.equals(")") || s.equals(">") ) {
               if( cur == -1) {
                   System.out.println("Fatal Error of " + name);
                   System.out.println("Premature closing pair!  Check it!");
                   System.exit(1);
               }
               int prev = bonds[cur];
               bonds[cur] = i;
               bonds[i] = cur;
               cur = prev;
           } else {
               bonds[i] = -1;
           }
        }

        if( cur != -1) {
            System.out.println("Processing " + name);
            System.out.println("Fatal Error: some opening pair not closed! Check it now!");
            System.exit(1);
        }

        remark = aRemark;
	conserved = aconserved;
        setup(name, aSequence, bonds);
    }


    public RNA(double [] factors, String name, String aSequence, String structLine, String aRemark, String aconserved)
    {
        int[] bonds = new int[structLine.length()];
        int cur = -1;
	flag =1; //#############################Make flag =1 ############################################
        for(int i=0; i<bonds.length; i++) {
           String s = structLine.substring(i, i+1);
           if( s.equals("(") || s.equals("<") ) {
               bonds[i] = cur;
               cur = i;
           } else if( s.equals(")") || s.equals(">") ) {
               if( cur == -1) {
                   System.out.println("Fatal Error of " + name);
                   System.out.println("Premature closing pair!  Check it!");
                   System.exit(1);
               }
               int prev = bonds[cur];
               bonds[cur] = i;
               bonds[i] = cur;
               cur = prev;
           } else {
               bonds[i] = -1;
           }
        }

        if( cur != -1) {
            System.out.println("Processing " + name);
            System.out.println("Fatal Error: some opening pair not closed! Check it now!");
            System.exit(1);
        }

        remark = aRemark;
	conserved = aconserved;
	factors_for_conserved = factors;
	//System.out.println("IN RNA: " + factors_for_conserved[299]);
        setup(name, aSequence, bonds);
    }
    
    public RNA(String name, String aSequence, String structLine, String aRemark)
    {
        int[] bonds = new int[structLine.length()];
        int cur = -1;
        for(int i=0; i<bonds.length; i++) {
           String s = structLine.substring(i, i+1);
           if( s.equals("(") || s.equals("<") ) {
               bonds[i] = cur;
               cur = i;
           } else if( s.equals(")") || s.equals(">") ) {
               if( cur == -1) {
                   System.out.println("Fatal Error of " + name);
                   System.out.println("Premature closing pair!  Check it!");
                   System.exit(1);
               }
               int prev = bonds[cur];
               bonds[cur] = i;
               bonds[i] = cur;
               cur = prev;
           } else {
               bonds[i] = -1;
           }
        }

        if( cur != -1) {
            System.out.println("Processing " + name);
            System.out.println("Fatal Error: some opening pair not closed! Check it now!");
            System.exit(1);
        }

        remark = aRemark;
        setup(name, aSequence, bonds);
    }
//##############################################################NEW for CSFS##################################################
    public RNA(String name, String aSequence, String structLine, String aRemark,int aseq_no,int astr_no)
    {
        int[] bonds = new int[structLine.length()];
        int cur = -1;
        for(int i=0; i<bonds.length; i++) {
           String s = structLine.substring(i, i+1);
           if( s.equals("(") || s.equals("<") ) {
               bonds[i] = cur;
               cur = i;
           } else if( s.equals(")") || s.equals(">") ) {
               if( cur == -1) {
                   System.out.println("Fatal Error of " + name);
                   System.out.println("Premature closing pair!  Check it!");
                   System.exit(1);
               }
               int prev = bonds[cur];
               bonds[cur] = i;
               bonds[i] = cur;
               cur = prev;
           } else {
               bonds[i] = -1;
           }
        }

        if( cur != -1) {
            System.out.println("Processing " + name);
            System.out.println("Fatal Error: some opening pair not closed! Check it now!");
            System.exit(1);
        }

        remark = aRemark;
        setup(name, aSequence, bonds);
	seq_no = aseq_no;
	str_no = astr_no;
//	System.out.println(seq_no);
//	System.out.println(str_no);
//	System.out.println("Assigned seq no & str no");
}
///############################################################################################################################
    
    //---------------------------------------------------------------------------------------
    // Try to find a region within sequence(parameter 'bSequence') to be as similar as 
    // possible to a given secondary structure(parameter 'aRNA'). The idea is from 
    // K.Zhang's '99 paper.
    // 
    //  Note:
    //---------------------------------------------------------------------------------------
    public static double patternSearcher(RNA aRNA, RNA bSequence, int[] aMap, int[] bMap,
                                Matcher matcher, double penalty)
    {
        if ( aRNA.postOrderPairs == null)
            aRNA._constructPostOrderPairs();

        int seqLen = aRNA.seqLength();

        int varLength = 0;
        for(int v=0; v<seqLen; v++) {
            if( aRNA.getBase(v).equals("n"))
                varLength ++;
        }

        int pairN = aRNA.postOrderPairs.size();
        int len = bSequence.seqLength();
        if( len > (int)(seqLen * 1.5))   // the maximum region is one and half of structure size
            len = (int)(seqLen * 1.5);

        double[][][] dist = new double[pairN][len][len];
        double[][][] encloseDist = new double[pairN][len][len];

        int[][][] distTrace = new int[pairN][len][len];
        int[][][] encloseDistTrace = new int[pairN][len][len];

        int firstBase = 0;

        double result_score = Double.NEGATIVE_INFINITY;
        int startPos = firstBase;
        int endPos = firstBase + len - 1;

        boolean extension = false;   // the first run should be from scratch !!
        String seq = bSequence.getSequence();
        System.out.println("\nScanning RNA " + bSequence.getName() + " now !");
        do {
            if( extension == true) { // shift the 3-D score arrays
                for(int p=0; p<pairN; p++) {
                    for(int i=1; i<len; i++) {
                        for(int j=i; j<len; j++) {
                            dist[p][i-1][j-1] = dist[p][i][j];
                            encloseDist[p][i-1][j-1] = encloseDist[p][i][j];
                        }
                    }
                }
            }

            ComputeInnerCircleDist(aRNA, 0, seqLen-1, seq.substring(firstBase, firstBase + len), 0, len-1,
                     dist, encloseDist, distTrace, encloseDistTrace, matcher, penalty, extension);

            if( firstBase + len >= bSequence.seqLength() ) { // the window reaches the sequence end !!
                int lastBase = bSequence.seqLength() - 1;
                int step = 0;

                while( lastBase - (firstBase+step) + 1 >= seqLen - varLength) {
                    for(int j = (seqLen - varLength - 1) + step; j < len; j++) {
                        if( dist[pairN-1][step][j] > result_score) {
                            result_score = dist[pairN-1][step][j];
                            startPos = firstBase + step;
                            endPos = firstBase + j;
                        }
                    }
                    step ++;
                }

                firstBase ++;
            } else { // the running window will shift right one nucleotide
                for(int j=(seqLen - varLength - 1); j<len; j++) {
                    if( dist[pairN-1][0][j] > result_score) {
                        result_score = dist[pairN-1][0][j];
                        startPos = firstBase;
                        endPos = firstBase + j;
                    }
                }
                extension = true;
                firstBase ++;
            }
        } while ( firstBase + len <= bSequence.seqLength() );

        if( aMap != null && bMap != null) {
            ComputeInnerCircleDist(aRNA, 0, seqLen-1, seq.substring(startPos, endPos+1), 0, endPos-startPos,
                     dist, encloseDist, distTrace, encloseDistTrace, matcher, penalty, false);

            for(int i=0; i<aMap.length; i++)
                aMap[i] = -1;
            for(int i=0; i<bMap.length; i++)
                bMap[i] = -1;

            MatchOneByOne(aRNA, 0, pairN-1, 0, endPos-startPos,
                    distTrace, encloseDistTrace, aMap, bMap);

            for(int i=0; i<bMap.length; i++)
                bMap[i] = -1;
            for(int i=0; i<aMap.length; i++) {
                if( aMap[i] > 0) {
                    aMap[i] = aMap[i] + startPos;
                    bMap[aMap[i]-1] = i + 1;
                }
            }
        }

        return result_score;
    }


    //-----------------------------------------------------------------------------
    // Another version of pattern searching: output top 'nHit' to STDOUT
    //-----------------------------------------------------------------------------
    public static void patternSearcher(RNA aRNA, RNA bSequence, int nHit,
                                Matcher matcher, double penalty)
    {
        if ( aRNA.postOrderPairs == null)
            aRNA._constructPostOrderPairs();

        int aSeqLen = aRNA.seqLength();

        int varLength = 0;
        for(int v=0; v<aSeqLen; v++) {
            if( aRNA.getBase(v).equals("n"))
                varLength ++;
        }

        int pairN = aRNA.postOrderPairs.size();
        int len = bSequence.seqLength();
        if( len > (int)(aSeqLen*1.5))
            len = (int)(aSeqLen*1.5);

        double[][][] dist = new double[pairN][len][len];
        double[][][] encloseDist = new double[pairN][len][len];

        int[][][] distTrace = new int[pairN][len][len];
        int[][][] encloseDistTrace = new int[pairN][len][len];

        int[] aMap = new int[aSeqLen];
        int[] bMap = new int[bSequence.seqLength()];


        StringBuffer[] results = new StringBuffer[nHit];
        double result_scores[] = new double[nHit];
        for(int i=0; i<result_scores.length; i++)
            result_scores[i] = Double.POSITIVE_INFINITY;

        int firstBase = 0;
        int startPos = firstBase;
        int endPos = firstBase + len - 1;

        boolean extension = false;   // the first run should be from scratch !!
        String seq = bSequence.getSequence();
        System.out.println("\nScanning RNA " + bSequence.getName() + " now !");
        do {
            if( extension == true) {  // shift the 3-D score arrays
                for(int p=0; p<pairN; p++) {
                    for(int i=1; i<len; i++) {
                        for(int j=i; j<len; j++) {
                            dist[p][i-1][j-1] = dist[p][i][j];
                            encloseDist[p][i-1][j-1] = encloseDist[p][i][j];
                        }
                    }
                }
            }

            ComputeInnerCircleDist(aRNA, 0, aSeqLen-1, seq.substring(firstBase, firstBase + len), 0, len-1,
                     dist, encloseDist, distTrace, encloseDistTrace, matcher, penalty, extension);

            double curMax = Double.NEGATIVE_INFINITY;
            if( firstBase + len >= bSequence.seqLength() )  // the window reaches the sequence end !!
            {
                int lastBase = bSequence.seqLength() - 1;
                int step = 0;
                while( lastBase - (firstBase+step) + 1 >= aSeqLen - varLength) {
                    for(int j = (aSeqLen - varLength - 1) + step; j < len; j++) {
                        if( dist[pairN-1][step][j] > curMax) {
                            curMax = dist[pairN-1][step][j];
                            startPos = firstBase + step;
                            endPos = firstBase + j;
                        }
                    }
                    step ++;
                }
                firstBase ++;
            } else { // the running window will shift right one nucleotide
                for(int j=(aSeqLen - varLength - 1); j<len; j++) {
                    if( dist[pairN-1][0][j] > curMax) {
                        curMax = dist[pairN-1][0][j];
                        startPos = firstBase;
                        endPos = firstBase + j;
                    }
                }
                extension = true;
                firstBase ++;
            }

            if( curMax > -result_scores[nHit-1]){ // a higher score was found !
                int index = Arrays.binarySearch(result_scores, -curMax);
                if( index < 0)  // a new high score is found
                    index = -(index+1);

                for( int i=nHit-2; i >= index; i--)
                    result_scores[i+1] = result_scores[i];
                result_scores[index] = - curMax;

                ComputeInnerCircleDist(aRNA, 0, aSeqLen-1, seq.substring(startPos, endPos+1), 0, endPos-startPos,
                         dist, encloseDist, distTrace, encloseDistTrace, matcher, penalty, false);

                for(int i=0; i<aMap.length; i++)
                    aMap[i] = -1;
                for(int i=0; i<bMap.length; i++)
                    bMap[i] = -1;

                MatchOneByOne(aRNA, 0, pairN-1, 0, endPos-startPos,
                        distTrace, encloseDistTrace, aMap, bMap);

                for(int i=0; i<bMap.length; i++)
                    bMap[i] = -1;
                for(int i=0; i<aMap.length; i++) {
                    if( aMap[i] > 0) {
                        aMap[i] = aMap[i] + startPos;
                        bMap[aMap[i]-1] = i + 1;
                    }
                }

                for( int i=nHit-2; i>=index; i--)
                    results[i+1] = results[i];

                results[index] = new StringBuffer(RNA.alignMatch(aRNA, aMap, bSequence, bMap));
                results[index].insert(0, "start base: " + (startPos+1) + "\n");
            }
        } while ( firstBase + len <= bSequence.seqLength() );

        for(int i=0; i<result_scores.length; i++) {
            if( result_scores[i] >= Double.POSITIVE_INFINITY)
                break;

            System.out.println("\nScore: " + (- result_scores[i]));
            System.out.println("Alignment: ");
            System.out.println(results[i]);
        }
    }

    //-------------------------------------------------------------------------------------------------
    // Following two will try to compair two RNA secondary structures GLOBALY and find GLOBAL matching
    // component. The optimal result will show all the partial structures which are shared by both RNAs.
    // The last parameter is a object making code translation for sequence annotation. It could be null
    // which means not code translation is needed.
    //-------------------------------------------------------------------------------------------------
    public static int match2D(RNA aRNA, RNA bRNA, int[] aMap, int[] bMap, Matcher matcher)
    {
        if( aRNA.postOrderPairs == null)
            aRNA._constructPostOrderPairs();

        if( bRNA.postOrderPairs == null)
            bRNA._constructPostOrderPairs();

        int compA = aRNA.postOrderPairs.size();
        int compB = bRNA.postOrderPairs.size();

        short[][] score = new short[compA][compB];
        short[][] semiCircleScore1 = new short[compA][compB];
        short[][] semiCircleScore2 = new short[compB][compA];
        short[][] circleScore = new short[compA][compB];

        byte[][] trace = new byte[compA][compB];
        byte[][] circleTrace = new byte[compA][compB];
        byte[][] semiTrace1 = new byte[compA][compB];
        byte[][] semiTrace2 = new byte[compB][compA];

        int result = 0;

        result = globalMatch2D(aRNA, 0, compA-1, bRNA, 0, compB-1, score, semiCircleScore1, 
                               semiCircleScore2, circleScore, trace, circleTrace,
                               semiTrace1, semiTrace2, false, matcher);

        boolean flag1 = (aRNA.beforeAlongCircle(compA-1) != -1);
        boolean flag2 = (bRNA.beforeAlongCircle(compB-1) != -1);

        if( aMap != null && bMap != null) {
            map2D(aRNA, 0, compA-1, aMap, bRNA, 0, compB-1, bMap, trace, circleTrace,
                            semiTrace1, semiTrace2, flag1, flag2, matcher);
        }

        return result;
    }

    public static int match2D(RNA aRNA, RNA bRNA, int[] aMap, int[] bMap)
    {
        return match2D(aRNA, bRNA, aMap, bMap, null);
    }

    //----------------------------------------------------------------------------------------
    // profileMatch2D will try to search a profile within a large RNA secondary structure, the
    // position-dependent scoring matrix is used to measure the alignment quality.
    //----------------------------------------------------------------------------------------
    public static double profileMatch2D(RNA profile, RNA subject, int[] profileMap, int[] subMap, 
                                      ProfileMatcher matcher, double Penalty)
    {
        if( profile.postOrderPairs == null)
            profile._constructPostOrderPairs();
        if( subject.postOrderPairs == null)
            subject._constructPostOrderPairs();

        int compProfile = profile.postOrderPairs.size();
        int compSub   = subject.postOrderPairs.size();

        double[][] score = new double[compProfile][compSub];
        double[][] semiCircleScore1 = new double[compProfile][compSub];
        double[][] semiCircleScore2 = new double[compSub][compProfile];
        double[][] circleScore = new double[compProfile][compSub];

        byte[][] trace = new byte[compProfile][compSub];
        byte[][] circleTrace = new byte[compProfile][compSub];
        byte[][] semiTrace1 = new byte[compProfile][compSub];
        byte[][] semiTrace2 = new byte[compSub][compProfile];

        double penalty = Penalty;

        profileDetector(profile, 0, compProfile-1, subject, 0, compSub-1, score,
                      semiCircleScore1, semiCircleScore2, circleScore, trace, circleTrace,
                      semiTrace1, semiTrace2, false, matcher, penalty);

        // the same tracing back function as that of "motif alignment"!
        double result = motifMap2D(profile, profileMap, subject, subMap, trace, circleTrace, 
                                   semiTrace1, semiTrace2, score, circleScore);
        return result;
    }

    //----------------------------------------------------------------------------------------
    // motifMatch2D will try to search a small motif within a large RNA secondary structure
    //----------------------------------------------------------------------------------------
    public static double motifMatch2D(RNA motif, RNA subject, int[] motifMap, int[] subMap, 
                                      Matcher matcher, double Penalty)
    {
        if( motif.postOrderPairs == null)
            motif._constructPostOrderPairs();
        if( subject.postOrderPairs == null)
            subject._constructPostOrderPairs();

        int compMotif = motif.postOrderPairs.size();
        int compSub   = subject.postOrderPairs.size();

        double[][] score = new double[compMotif][compSub];
        double[][] semiCircleScore1 = new double[compMotif][compSub];
        double[][] semiCircleScore2 = new double[compSub][compMotif];
        double[][] circleScore = new double[compMotif][compSub];

        byte[][] trace = new byte[compMotif][compSub];
        byte[][] circleTrace = new byte[compMotif][compSub];
        byte[][] semiTrace1 = new byte[compMotif][compSub];
        byte[][] semiTrace2 = new byte[compSub][compMotif];

        double penalty = Penalty;

        motifDetector(motif, 0, compMotif-1, subject, 0, compSub-1, score,
                      semiCircleScore1, semiCircleScore2, circleScore, trace, circleTrace,
                      semiTrace1, semiTrace2, false, matcher, penalty);

        double result = motifMap2D(motif, motifMap, subject, subMap, trace, circleTrace, 
                                   semiTrace1, semiTrace2, score, circleScore);
        return result;
    }

    //---------------------------------------------------------------------------------------
    // motifMatch2D will try to search a small motif within a large RNA secondary structure
    //---------------------------------------------------------------------------------------
    public static double motifMatch2D(RNA motif, RNA subject, int[] motifMap, int[] subMap, Matcher matcher)
    {
        return motifMatch2D(motif, subject, motifMap, subMap, matcher, -3); //default penalty: -3
    }


    //------------------------------------------------------------------------------------
    //      LATEST VERSION OF LOCAL ALIGNMENT BETWEEN TWO STRUCTURES
    //------------------------------------------------------------------------------------
    public static double localMatch2D(RNA aRNA, RNA bRNA, int[] aMap, int[] bMap,
                                      Matcher matcher, double Penalty)
    {
        if( aRNA.postOrderPairs == null)
            aRNA._constructPostOrderPairs();
        if( bRNA.postOrderPairs == null)
            bRNA._constructPostOrderPairs();

        int compA = aRNA.postOrderPairs.size();
        int compB = bRNA.postOrderPairs.size();

        double[][] lScore = new double[compA][compB];
        double[][] gScore = new double[compA][compB];
        double[][] semiCircleScore1 = new double[compA][compB];
        double[][] semiCircleScore2 = new double[compB][compA];
        double[][] circleScore = new double[compA][compB];

        byte[][] gTrace = new byte[compA][compB];
        byte[][] lTrace = new byte[compA][compB];
        byte[][] circleTrace = new byte[compA][compB];
        byte[][] lSemiTrace1 = new byte[compA][compB];
        byte[][] gSemiTrace1 = new byte[compA][compB];
        byte[][] lSemiTrace2 = new byte[compB][compA];
        byte[][] gSemiTrace2 = new byte[compB][compA];

        double penalty = Penalty;

        localAligner(aRNA, 0, compA-1, bRNA, 0, compB-1, gScore,
                     semiCircleScore1, semiCircleScore2, lScore, circleScore, gTrace,
                     lTrace, circleTrace, lSemiTrace1, lSemiTrace2,
                     gSemiTrace1, gSemiTrace2, false, matcher, penalty);

        double result = localMap2D(aRNA, aMap, bRNA, bMap, lTrace, gTrace, 
                                   circleTrace, lSemiTrace1, lSemiTrace2,
                                   gSemiTrace1, gSemiTrace2, lScore, circleScore);
        return result;
    }


    //---------------------------------------------------------------------------
    //            !!!! OBSOLETE !!!!
    // A new RNA 2D structure is formed by concatenating two small structures.
    //---------------------------------------------------------------------------
    /*public static RNA concatRNA(RNA aRNA, RNA bRNA)
    {
        String newSeq = new String(aRNA.getSequence() + bRNA.getSequence());
        int[] newBonds = new int[aRNA.seqLength() + bRNA.seqLength()];

        int length = aRNA.seqLength();
        int[] pairs = aRNA.getPairs();
        for(int i=0; i<length; i++)
            newBonds[i] = pairs[i];

        length = bRNA.seqLength();
        pairs = bRNA.getPairs();

        int start = aRNA.seqLength();
        for(int i=start; i<newBonds.length; i++)
        {
            if( pairs[i-start] != -1)
                newBonds[i] = pairs[i-start] + start;
            else
                newBonds[i] = pairs[i-start];
        }

        RNA newRNA = new RNA();
        newRNA.setup(aRNA.getName() + bRNA.getName(), newSeq, newBonds);
        return newRNA;
    }*/

    public String getName()
    {
        return RNAName;
    }

    public String getSequence()
    {
        return sequence;
    }

    //-------------------------------------------------------------
    // display the information in the same form of plain input file
    //-------------------------------------------------------------
    public String getInfo()
    {
        StringBuffer seq = new StringBuffer();
        seq.append(">" + getName() + "\t" + remark + "\n");
        seq.append(getSequence() +  "\n");
        seq.append(getStructLine() + "\n");

        return new String(seq);
    }

    // ----------------------------------------------------------------------
    // Format the RNA secondary structure using nested-parenthesis notation.
    // ----------------------------------------------------------------------
    public String dispRNA()
    {
        StringBuffer ret = new StringBuffer();
        ret.append(">" + getName() + "\t" + remark + "\n");
        StringBuffer lineA = new StringBuffer();
        StringBuffer lineB = new StringBuffer();
        int start = 0;
        int end = seqLength();
        while( start < end )
        {
           int step = start + 50;
           if( step > end)
               step = end;

           lineA.append("    ");
           lineA.append(getStructLine().substring(start, step));
           lineA.append("\n");

           lineB.append(start+1);
           int length = lineB.length();
           for(int j=0; j<4-length; j++)
               lineB.append(" ");
           lineB.append(getSequence().substring(start, step));
           lineB.append(" " + step + "\n");

           ret.append("\n");
           ret.append(lineA);
           ret.append(lineB);

           lineA.delete(0, lineA.length());
           lineB.delete(0, lineB.length());

           start = step;
        }
        ret.append("\n");
 
        return ret.toString();
    }

    public int seqLength()
    {
        if( sequence != null)
            return sequence.length();
        else
            return 0;
    }

    public String getAnnotate()
    {
        return remark;
    }

    public String toString()
    {
        if ( sequence.length() < 10 )
            return "RNA(" + RNAName + ")[(" + seqLength() +")sequence= " + sequence + "]";
        else
            return "RNA(" + RNAName + ")[(" + seqLength() +")sequence= " + sequence.substring(0, 10) + "..."
                    + "]";
    }

    //-----------------------------------------------------------------------------------
    // get all structure components (single bases and base pairs) from the structure !!!!
    //-----------------------------------------------------------------------------------
    private void getComponents()
    {
        if( postOrderPairs == null)
            _constructPostOrderPairs();

        singleCompo = new ArrayList();
        pairCompo = new ArrayList();
        for(int i=0; i<postOrderPairs.size(); i++)
        {
            String bond = getBaseBond(i);

            if( bond.length() == 2)
                pairCompo.add(new String(bond));
            else
                singleCompo.add(new String(bond));
        }
    }

    //---------------------------------------------------------------------------------------
    // Randomize the structure components while keep the overall structure conformation
    //---------------------------------------------------------------------------------------
    public RNA randomize()
    {
        if( singleCompo == null || pairCompo == null )
            getComponents();

        StringBuffer newSeq = new StringBuffer(sequence);
        Random rand = new Random();
        for(int i=0; i<postOrderPairs.size(); i++) {
            int[] pos = getComponentPos(i);
            if( pos.length == 2){ 
               int a = rand.nextInt(pairCompo.size());
               String newBond = (String)pairCompo.get(a);
               newSeq.replace(pos[0], pos[0]+1, newBond.substring(0,1));
               newSeq.replace(pos[1], pos[1]+1, newBond.substring(1,2));
            } else {
               int a = rand.nextInt(singleCompo.size());
               String newSingle = (String)singleCompo.get(a);
               newSeq.replace(pos[0], pos[0] + 1, newSingle.substring(0,1));
            }
        }

        return new RNA(RNAName, newSeq.toString(), getStructLine(), "randomized");
    }


    //---------------------------------------------------------------------------------------------
    // This function will try to extract all single sequence regions and orgnize those in the
    // same loop to one goup, called single-stranded region FEATURE.
    //---------------------------------------------------------------------------------------------
    public static void extractSingleRegions(RNA aRNA, ArrayList singleRegions)
    {
        int start = 0;

        if( aRNA.postOrderPairs == null)
            aRNA._constructPostOrderPairs();

        int end = aRNA.postOrderPairs.size();
        allSingleRegions(aRNA, start, end, singleRegions);
    }

    //---------------------------------------------------------------------------------------------
    // This function will try to extract all loops
    //---------------------------------------------------------------------------------------------
    public static void extractAllLoops(RNA aRNA, ArrayList loops)
    {
        int start = 0;

        if( aRNA.postOrderPairs == null)
            aRNA._constructPostOrderPairs();

        int end = aRNA.postOrderPairs.size();
        allRegions(aRNA, start, end, loops);
    }

    //------------------------------------------------------------------------
    // get the structure representation in the format of nested-parenthesis
    //------------------------------------------------------------------------
    public String getStructLine()
    {
        StringBuffer ret = new StringBuffer();
        for(int i=0; i<basePairs.length; i++)
        {
            if( basePairs[i] == -1)
                ret.append(".");
            else if( basePairs[i] > i)
                ret.append("(");
            else 
                ret.append(")");
        }

        return new String(ret);
    }

    private void setup(String name, String aSequence, int[] bonds)
    {
        RNAName = new String(name);
        sequence = new String(aSequence);

        basePairs = new int[aSequence.length()];
        for(int i=0; i<basePairs.length; i++)
            basePairs[i] = -1;

        int GoodRow = StrictMath.min(bonds.length, sequence.length());
        for (int i=0; i<GoodRow; i++) {
            if ( bonds[i] >= 0 && bonds[i] < sequence.length())
                basePairs[i]= bonds[i];
        }
        postOrderPairs = null;
    }

    /*private void setup(int[] bonds)
    {
        for(int i=0; i<basePairs.length; i++)
            basePairs[i] = -1;

        detStructure(bonds);
    }

    //-----------------------------------------------
    //   determine the matrix representation.
    //-----------------------------------------------
    private void detStructure(int[] pairs)
    {
        int GoodRow = StrictMath.min(pairs.length, sequence.length());

        for (int i=0; i<GoodRow; i++)
        {
            if ( pairs[i] >= 0 && pairs[i] < sequence.length())
                basePairs[i]= pairs[i];
        }

        postOrderPairs = null;
    }*/


    //---------------------------------------------------------------------
    // format the the sequence information in the same form of input file
    //---------------------------------------------------------------------
    private String dispSequence()
    {
        StringBuffer seq = new StringBuffer();

        int rows = seqLength() / 50;
        int tail = seqLength() % 50;

        seq.append(">sequence:\n");
        for(int i=0; i<rows; i++)
            seq.append(""+sequence.substring(i*50, i*50+50) + "\n");

        seq.append(""+sequence.substring(50*rows, seqLength()));
        return new String(seq);
    }

    //------------------------------------------------------
    // format the the secondary structure information
    // in the same form of input file (stem specification)
    //------------------------------------------------------
    private String dispStruct()
    {
        StringBuffer pairs = new StringBuffer();

        if( postOrderPairs == null)
            _constructPostOrderPairs();

        pairs.append(">bonds:\n");
        for(int i=0; i < postOrderPairs.size(); i++) {
            int pairBases = ((Integer)postOrderPairs.get(i)).intValue();

            int n = 0;
            int prevLeft, left;
            if( pairBases >= seqLength()) {
                left = pairBases / seqLength() -  1;
                prevLeft = left;

                do{
                    n ++;

                    if( i+n < postOrderPairs.size())
                        pairBases = ((Integer)postOrderPairs.get(i+n)).intValue();
                    else
                        break;

                    if( pairBases >= seqLength()) {
                        prevLeft = left;
                        left = pairBases / seqLength() - 1;
                    } else
                        break;
                } while( left == (prevLeft - 1));
            }

            if( n > 0) {
                pairBases = ((Integer)postOrderPairs.get(i+n-1)).intValue();
                int Left = pairBases / seqLength() - 1;
                int Right = pairBases % seqLength();

                pairs.append("" + (Left+1) + '\t' + (Right+1) + '\t' + n + "\n");

                i = i+n-1;
            }
        }

        return new String(pairs);
    }

    //--------------------------------------------------------------------------------------------
    // Given a base(by index in the sequence), getPairedBase return its paired base if possible;
    // otherwise, return -1, meaning the given base is unpaired single base or failure happended.
    //--------------------------------------------------------------------------------------------
    public int getPairedBase(int aBaseIndex) // Base index starting from integer 0 !!
    {
        if (aBaseIndex >= seqLength() || aBaseIndex < 0)
            return -1;

        return basePairs[aBaseIndex];
    }

    //-----------------------------------------------------------------
    // Given an index, returns the nucleotide (A, C, G, U).
    //-----------------------------------------------------------------
    public String getBase(int aBaseIndex)
    {
        if (aBaseIndex < 0 || aBaseIndex >= seqLength())
            return null;
        else
            return sequence.substring(aBaseIndex, aBaseIndex+1);
    }

    private int getComponent(int index)
    {
        return ((Integer)postOrderPairs.get(index)).intValue();
    }

    public int[] getComponentPos(int index)
    {
        int component = getComponent(index);
        if( component < sequence.length()) {
            int[] ret = new int[1];
            ret[0] = component;
            return ret;
        } else {
            int left = component / seqLength() - 1;
            int right = component % seqLength();
            int[] ret = new int[2];
            ret[0] = left;
            ret[1] = right;
            return ret;
        }
    }

    public int getNumberOfComponent()
    {
        if( postOrderPairs == null)
            _constructPostOrderPairs();

        return postOrderPairs.size();
    }

    public String getPosBond(int index)
    {
        int component = getComponent(index);
        if( component < sequence.length())
            return ("" + (component+1));
        else{
            int left = component/seqLength() - 1;
            int right = component % seqLength();
            return (String)((left+1) + "-" + (1+right));
        }
    }

    private String getBaseBond(int index)
    {
        int component = getComponent(index);

        if (component < sequence.length())
            return getBase(component);
        else {
            int left = component / seqLength() - 1;
            int right = component % seqLength();
            return getBase(left) + getBase(right);
        }
    }

    //----------------------------------------------------
    // Extract all loops from the secondary structure.
    //----------------------------------------------------
    private static void allRegions(RNA aRNA, int start, int end, ArrayList regions)
    {
        StringBuffer region = new StringBuffer();

        int current = start;
        int prev = current;
        int component = aRNA.getComponent(current);

        int prevBase;
        int leftBond;
        if( component < aRNA.seqLength()) {
            prevBase = component;
            leftBond = component;
        } else {
            prevBase = component % aRNA.seqLength();
            leftBond = component / aRNA.seqLength() - 1;
        }

        do {
            if( component < aRNA.seqLength()){   // a single base
                region.append(aRNA.getBase(component));
                prevBase = component;
            } else{      // base-pair:  two cases as following
                if( leftBond == prevBase + 1){    // a base is encountered along the circumference
                    region.append(aRNA.getBaseBond(current));
                    allRegions(aRNA, prev+1, current, regions);
                } else{     // closing pair
                    region.append(aRNA.getBaseBond(current));
                    regions.add(new String(region));
                    region.delete(0, region.length()-2);
                }
                prevBase = component % aRNA.seqLength();
            }

            prev = current;
            current = aRNA.nextAlongCircle(current);
            if(current != -1) {
                component = aRNA.getComponent(current);
                if( component < aRNA.seqLength())
                    leftBond = component;
                else
                    leftBond = component / aRNA.seqLength() - 1;
            }
        } while( current != -1 && current <= end);

        if( region.length() != 0 && prevBase <= leftBond)
            regions.add(new String(region));
    }


    //--------------------------------------------------------------------------
    // single tranded regions are extracted from the given secondary structure
    //--------------------------------------------------------------------------
    private static void allSingleRegions(RNA aRNA, int start, int end, ArrayList regions)
    {
        StringBuffer region = new StringBuffer();

        int current = start;
        int prev = current;
        int component = aRNA.getComponent(current);

        int prevBase;
        int leftBond;
        if( component < aRNA.seqLength()) {
            prevBase = component;
            leftBond = component;
        } else {
            prevBase = component % aRNA.seqLength();
            leftBond = component / aRNA.seqLength() - 1;
        }

        do {
            if( component < aRNA.seqLength()){    // a single base
                region.append(aRNA.getBase(component));
                prevBase = component;
            } else {     // base-pair:  two cases as following
                if( leftBond == prevBase + 1)  // a base is encountered along the circumference
                    allSingleRegions(aRNA, prev+1, current, regions);
                else{     // closing pair
                    if( region.length() != 0) {
                        regions.add(new String(region));
                        region.delete(0, region.length());
                    }
                }
                prevBase = component % aRNA.seqLength();
            }

            prev = current;
            current = aRNA.nextAlongCircle(current);
            if(current != -1) {
                component = aRNA.getComponent(current);
                if( component < aRNA.seqLength())
                    leftBond = component;
                else
                    leftBond = component / aRNA.seqLength() - 1;
            }
        } while( current != -1 && current <= end);

        if( region.length() != 0)
            regions.add(new String(region));
    }


    //-----------------------------------------------------------------------------------
    // set up the sorted arrary for structural components (base pairs + unpaired bases)
    // according to  3' endpoint
    //-----------------------------------------------------------------------------------
    private void _constructPostOrderPairs()
    {
        postOrderPairs = new ArrayList();

        for (int i=0; i<seqLength(); i++) {
            int pairedBase = getPairedBase(i);

            if ( pairedBase == -1)             // this base is unpaired
                postOrderPairs.add(new Integer(i));
            else if ( pairedBase < i)         // here comes a base pair: (pairedBase, i)
                postOrderPairs.add(new Integer((pairedBase+1) * seqLength() + i));
        }
    }

    //-----------------------------------------------------------
    // it returns the first component staying on the same loop
    // as the "current" component.
    //-----------------------------------------------------------
    private int firstAlongCircle(int current)
    {
        int result;

        result = beforeAlongCircle(current);
        while( result != -1) {
            current = result;
            result = beforeAlongCircle(result);
        }

        return current;
    }


    //-----------------------------------------------------------------------------------
    //  Given a base or base-pair (indexed by parameter "current"), this function returns
    //  the previous base or base-pair (position of sorted arrary) in the same loop as
    //  "current"; otherwise, '-1' will be returned.
    //  Here 'loop' means the same circle in the secondary structural point of view.
    //-----------------------------------------------------------------------------------
    private int beforeAlongCircle(int current)
    {
        if (current < 0 || current >= postOrderPairs.size())  // out of index range
            return -1;

        int basePair = ((Integer)postOrderPairs.get(current)).intValue();
        int beforeBase;

        // If 'basePair' is a single nucleotide, then the before nucleotide SHOULD
        // be at the same loop; otherwise, if 'basePair' is a pair, then the closest
        // nucleotide neighboring to its 5' bonded base SHOULD be at the same loop.
        if (basePair < seqLength())
            beforeBase = basePair - 1;
        else
            beforeBase = (basePair / seqLength() - 1) - 1;

        if (beforeBase < 0)  // we have crossed over the begining.
            return -1;

        int i = current - 1;
        int candidate;
        while ( i >= 0 ) {
            candidate = ((Integer)postOrderPairs.get(i)).intValue();
            if(candidate < seqLength()) {
                if (candidate == beforeBase)   // the before one is a single base
                    return i;
                else
                    i -- ;   // continue searching
            } else { // this is a base pase, it's 3' end must be "beforeBase"
                if ( candidate % seqLength() == beforeBase)
                    return i;
                else
                    i --;
            }
        }

        return -1;  // failure, due to the malformed secondary structure: UNCLOSED CIRCLE
    }


    //-----------------------------------------------------------------------------
    //  Given a base or base-pair (indexed by parameter, this function returns
    //  the next base or base-pair (position of sorted arrary) in the same loop.
    //  Here 'loop' means the same circle of the secondary structure.
    //-----------------------------------------------------------------------------
    private int nextAlongCircle(int current)
    {
        if (current < 0 || current >= postOrderPairs.size())  // out of index range
            return -1;

        int basePair = ((Integer)postOrderPairs.get(current)).intValue();

        int nextBase;   // 'nextBase' is the next ribonucleotide residing in the same
                        // level with 'basePair', which is maybe a single base or base-pair.

        // If 'basePair' is a single ribonucleotide, then the next ribonucleotide
        // SHOULD be at the same loop; otherwise, if 'basePair' is a pair, then the
        // closest ribonucleotide neighboring to its 3' bonded base SHOULD be at the same loop.
        if (basePair < seqLength())
            nextBase = basePair + 1;
        else
            nextBase = basePair % seqLength() + 1;

        if (nextBase >= seqLength())  // we have reached the end.
            return -1;

        int i = current + 1;
        int candidate;
        while ( i < postOrderPairs.size() ) {
            candidate = ((Integer)postOrderPairs.get(i)).intValue();
            if(candidate < seqLength()) {
                if (candidate == nextBase)   // the next one is a single base
                    return i;
                else {
                    i ++;
                    continue;
                }
            } else {
                if( (candidate/seqLength() - 1 == nextBase) || candidate%seqLength() == nextBase){
                // the next one could a bonded base pair, or stack pair, or bulge, or inner loop
                    return i;
                } else if( candidate/seqLength() - 1 > nextBase){  // if this is a secondary structure,
                                                                   // the condition is superfluous.
                    i ++;
                    continue;
                }
            }
        }  // we reached the end of 'while' loop.

        return -1;  // failure, due to the malformed secondary structure: UNCLOSED CIRCLE
    }


    //-----------------------------------------------------------------------------
    // This function tries to make nucleotide to ucleotide matching, or
    // nucleotide-pair to nucleotide-pair matching.
    //
    // Currently, if two single ribonucleotides are equal, we set score as '1';
    // if two ribonucleotide-pairs are matched, we set score as '2'; others, the
    // score will be set to '0'. In fact, we don't match single with pair.
    //-----------------------------------------------------------------------------/
    private static short baseMatch(String base1, String base2)
    {
        int length = base1.length();
        short score = 0;

        if ( length == base2.length()){   // only the equal length cases counted !
            if (base1.equals(base2)) {
                if (length == 1)
                    score = 1;
                else if (length == 2)
                    score = 2;
            } else if (length == 2) {
                String swap = new String(base1.substring(1,2) + base1.substring(0,1));
                if (swap.equals(base2))   // it is possible to exchange bases within a pair.
                    score = 2;
            }
        }

        return score;
    }

    //---------------------------------------------------------------------------
    // This version will be used by LOCAL matching algorithm if no Matcher object
    // is provided during computation.
    //---------------------------------------------------------------------------
    private static short baseMatch(String base1, String base2, boolean ungap)
    {
        short score;
        score = (short)baseMatch(base1, base2);

        if( ungap==true && score<=0)
            score = Short.MIN_VALUE;

        return score;
    }

    private int[] getPairs()
    {
        return basePairs;
    }



    //-----------------------------------------------------------------------------------------------
    // This accomplish the work of computing folding scheme for a sequence RNA with respect to
    // a pattern secondary structure.
    // 
    // The partial secondary structure starts from 'start'th of the sorted base/pair array, until
    // the 'baseIndex'th base is reached within a structrual component;
    // 
    // The sequence segment is confined to between 'startBase'th BASE and 'endBase'th BASE;
    // Parameter 'dist' contains all distances between any circle structure(single base could be
    // treated as a degenerated circle) and any subsegment pieces enclosed by 'startBase'th BASE and
    // 'endBase'th BASE.
    //-----------------------------------------------------------------------------------------------
    private static void ComputeInnerCircleDist(RNA pattern, int start, int baseIndex,
                String bSequence, int startBase, int endBase, double[][][] dist, double[][][] encloseDist,
                int[][][] distTrace, int[][][] encloseDistTrace, Matcher matcher, double penalty,
                boolean extension)
    {
          int component;      // internal representation of the current element along
                                // the circumference of secondary structure 'aRNA'.
          int rightBase;      // right bonded base, if this EQUALS to 'baseIndex', then we are done
          int leftBase;
          int current = start;  // current structure element of 'aRNA'
          int prev = start;       // previous component along the circumference
          int segLength = endBase - startBase + 1;  // sequence segment need to be tested
          int prevBase;         // the nearest 3' nucleotide that visited before current step

          component = pattern.getComponent(current);
          if(component < pattern.seqLength())
          {
              prevBase = component;
              leftBase = component;
              rightBase = component;
          }
          else
          {
              prevBase = component % pattern.seqLength();
              leftBase = component / pattern.seqLength() - 1;
              rightBase = prevBase;
          }

          while( (rightBase <= baseIndex) && (current < pattern.postOrderPairs.size()) )
          {
              int startingPoint = startBase;

              if( extension == true)
                  startingPoint = endBase;

              for(int j=startingPoint; j<=endBase; j++)
              {
                for(int i=0; j-i>=startBase; i++)   // subsequence is (j-i, j)
                {
                    if(component < pattern.seqLength())     // a single base from the structure
                    {
                        double match = 0;
                        if( matcher != null)
                            match = matcher.baseMatch(pattern.getBase(component), bSequence.substring(j, j+1));
                        else
                            match = (double)RNA.baseMatch(pattern.getBase(component), bSequence.substring(j, j+1));

                        double Dcase1 = 0;  // partial circumference case1:  discard element 'current'
                        double Dcase2 = 0;  // partial circumference case2:  discard nucleotide bSquence[j+i];
                        double Dcase3 = 0;  // partial circumference case3:  match 'current' with bSquence[j+i]

                        double Ecase1 = 0;  // 'current'-close loop case1: discard bSquence[j+i]
                        double Ecase2 = 0;  // 'current'-close loop case2: match 'current' with bSquence[j+i]

                        if( (i == 0) && (start == current))  // subsequence has only one(1) nucleotide
                                                             // the partial circumference just begins
                        {
                            if( pattern.getBase(component).equals("n"))
                            {
                                Dcase1 = penalty;
                                Dcase2 = penalty;
                            }
                            else
                            {
                                Dcase1 = 2 * penalty;  // two penalties: one from cut off 'current', another from
                                                       // bSquence[j+i] being matched to NOTHING!
                                Dcase2 = 2 * penalty;  // two penalties: one from cut off bSquence[j+i], another from
                                                       // 'current' element being matched to NOTHING!
                            }

                            Dcase3 = match;       // match 'current' with bSquence[j+i]

                            // 'current' element is treated as an independent entity:
                            if( pattern.getBase(component).equals("n"))
                                Ecase1 = penalty;
                            else
                                Ecase1 = 2 * penalty;   // cut off nucleotide bSquence[j+i]

                            Ecase2 = match;         // or match nucleotide bSquence[j+i]
                        }
                        else if (i==0)    // subsequence still has only one(1) nucleotide while
                                          // the partial circumference has moved forward
                        {
                            // Find out how many optional elements along the partial circumference
                            int element = pattern.firstAlongCircle(current);
                            int comp = pattern.getComponent(element);
                            int firstBase = 0;
                            if( comp < pattern.seqLength())
                                firstBase = comp;
                            else
                                firstBase = comp / pattern.seqLength() - 1;

                            int count = 0;
                            for(int m=firstBase; m<=component; m++)
                            {
                                if( pattern.getBase(m).equals("n") != true)
                                    count ++;
                            }

                            // 'current' element is not counted in matching
                            if( pattern.getBase(component).equals("n") == true)
                                Dcase1 = dist[prev][j-i][j];   // no penalty;
                            else
                                Dcase1 = dist[prev][j-i][j] + penalty;  // gap penalty

                            // Nothing is matched!
                            Dcase2 = count * penalty + penalty;

                            if( pattern.getBase(component).equals("n") != true)
                                count --;

                            Dcase3 = match + count * penalty;

                            // 'current' element is treated as an independent entity:
                            Ecase1 = 2 * penalty;  // Nothing is matched !
                            Ecase2 = match;
                        }
                        else if (start == current)   // subsquence has length larger than one 1
                                                     // while partial circumference just begins
                        {
                            if( pattern.getBase(component).equals("n") == true)
                                Dcase1 = (i+1) * penalty;         // nothing is left, penalty only from subsequence
                            else
                                Dcase1 = penalty + (i+1)*penalty; // nothing is left, penalty from both

                            Dcase2 = dist[current][j-i][j-1] + penalty;   // subsequence's last nucleotide cut off

                            Dcase3 = match + i * penalty; // subsequence's last base is matched with 'current' element
                                                          // while its prefix [j-i, j-1] is cut off, incurring penalty

                            // 'current' element is treated as an independent entity:
                            Ecase1 = encloseDist[current][j-i][j-1] + penalty;  // cut off nucleotide bSquence[j+i]
                            Ecase2 = match + i * penalty;
                        }
                        else
                        {
                            if( pattern.getBase(component).equals("n") == true)
                                Dcase1 = dist[prev][j-i][j];      // 'current' is cut off
                            else
                                Dcase1 = dist[prev][j-i][j] + penalty;      // 'current' is cut off

                            Dcase2 = dist[current][j-i][j-1] + penalty; // bSequence[j+i] is cut off
                            Dcase3 = dist[prev][j-i][j-1] + match;      // 'current' is matched with bSquence[j+i];

                            // 'current' element is treated as an independent entity:
                            Ecase1 = encloseDist[current][j-i][j-1] + penalty; // cut off nucleotide bSquence[j+i]
                            Ecase2 = match + i * penalty;
                        }

                        dist[current][j-i][j] = Math.max(Dcase1, Math.max(Dcase2, Dcase3));
                        encloseDist[current][j-i][j] = Math.max(Ecase1, Ecase2);

                        // Now we trace back the step leading to current inner circle distance
                        if ( Dcase1 >= Dcase2 && Dcase1 >= Dcase3)
                            distTrace[current][j-i][j] = 1;  // we trace back case1: 'current' is thrown away
                        else if (Dcase2 >= Dcase3)
                            distTrace[current][j-i][j] = 2;   // trace back case2: bSequence[j] is thrown away
                        else
                            distTrace[current][j-i][j] = 3;   // trace back case3: 'current' <--> bSequence[j];

                        // We also track back step leading to current enclosing circle distance
                        if ( Ecase1 >= Ecase2)
                            encloseDist[current][j-i][j] = 1;    // trace back case1: bSequence[j] is not used
                        else
                            encloseDist[current][j-i][j] = 2;  // 'current' <--> bSequenc[j].
                                                               // if we trace back here, the prefix
                                                               // subsegment [j...j+i-1] is matched
                                                               // to empty.
                    }
                    else {    // now, it comes the paired ribonucleotides:
                        if (leftBase < prevBase) {   // bond pair:  (leftBase <--> prevBase+1)
                            // a paired base, and we are returning from within nested loop, which means
                            // we can calculate a inner loop's whole enclose distance
                            // we have four cases to get the match between a loop and subsequence
                            // case-1: loop --->  segment [j-i+1 ... j]
                            // case-2: loop --->  segment [j-i ... j-1]
                            // case-3: open loop(cut off pair) --> sequence [j-i ... j]
                            // case-4: loop ---> sequence [j-i ... j] conditioned on the pair
                            //         must be matched with two nucleotides {[j-i], [j]}
                            double case1 = 0;
                            double case2 = 0;
                            double case3 = 0;
                            double case4 = 0;
                            double match = 0;

                            int count = 0;
                            rightBase  = pattern.getPairedBase(leftBase);
                            for(int k=leftBase; k<=rightBase; k++)
                            {
                                if( pattern.getBase(k).equals("n") != true)
                                    count ++;
                            }

                            if (i == 0)  // the subsequence has only one nucleotide
                            {
                                case1 = penalty + count * penalty;  // if nucleotide is cut off, the 
                                                                    // circle will also be cut off
                                case2 = penalty + count * penalty;

                                if( pattern.getBase(leftBase).equals("n") || 
                                          pattern.getBase(rightBase).equals("n"))
                                    case3 = dist[current-1][j-i][j];
                                else
                                    case3 = dist[current-1][j-i][j] + 2 * penalty; // the pair 'current' is cut off

                                case4 = Double.NEGATIVE_INFINITY;
                            }
                            else if(i==1) // the sequence has two bases
                            {
                                case1 = encloseDist[current][j-i+1][j] + penalty;
                                case2 = encloseDist[current][j-i][j-1] + penalty;

                                if( pattern.getBase(leftBase).equals("n") || 
                                            pattern.getBase(rightBase).equals("n"))
                                    case3 = dist[current-1][j-i][j];
                                else
                                    case3 = dist[current-1][j-i][j] + 2 * penalty; // the pair 'current' is cut off

                                case4 = Double.NEGATIVE_INFINITY;
                            }
                            else if (i>1)
                            {
                                case1 = encloseDist[current][j-i+1][j] + penalty;
                                case2 = encloseDist[current][j-i][j-1] + penalty;

                                if( pattern.getBase(leftBase).equals("n") || 
                                             pattern.getBase(rightBase).equals("n"))
                                    case3 = dist[current-1][j-i][j];
                                else
                                    case3 = dist[current-1][j-i][j] + 2 * penalty; // the pair 'current' is cut off

                                if( matcher != null)
                                    match = matcher.baseMatch(pattern.getBase(leftBase) + pattern.getBase(prevBase+1),
                                             bSequence.substring(j-i, j-i+1) + bSequence.substring(j, j+1));
                                else
                                    match = RNA.baseMatch(pattern.getBase(leftBase) + pattern.getBase(prevBase+1),
                                           bSequence.substring(j-i, j-i+1) + bSequence.substring(j, j+1));

                                case4 = dist[current-1][j-i+1][j-1] + match;
                            }

                            encloseDist[current][j-i][j] = Math.max(case1, Math.max(
                                                            case2, Math.max(case3, case4)));

                            // Now we have to make an assumption that the structure is finally ended by
                            // a close loop, in this case, inner loop distance('dist') is equal
                            // to whole loop distance('encloseDist'). Fortunately, if this is not the
                            // case, the following "else" will be excuted to recalculate the distances
                            dist[current][j-i][j] = encloseDist[current][j-i][j];

                            // Now we trace back the step leading to current inner loop distance
                            if ( case1 >= case2 && case1 >= case3 && case1 >= case4)
                                encloseDistTrace[current][j-i][j] = 1;  // we trace back case1: cut off [j-i]
                            else if (case2 >= case3 && case2 >= case4)
                                encloseDistTrace[current][j-i][j] = 2;   // trace back case2: cut off [j]
                            else if (case3 >= case4)
                                encloseDistTrace[current][j-i][j] = 3;   // trace back case3: cut off 'current' pair
                            else
                                encloseDistTrace[current][j-i][j] = 4;   // trace back case4: match 'current' pair
                                                                         // with pair [j-i, j]

                            distTrace[current][j-i][j] = -1 * (j-i);
                        } // end of calcuting whole loop
                        else //if (leftBase > baseIndex)
                        {
                            // we return here from a nested loop
                            // Now we update distance between partial secondary stucture and sequence
                            // we have two cases:
                            // case-1: discard current loop
                            // case-2: match the circle to some subsegment to get optimal solution
                            double case1 = Double.NEGATIVE_INFINITY;
                            double case2 = Double.NEGATIVE_INFINITY;

                            int count1 = 0;
                            rightBase = pattern.getPairedBase(leftBase);
                            for(int k=leftBase; k<=rightBase; k++)
                            {
                                if( pattern.getBase(k).equals("n") != true)
                                    count1 ++;
                            }

                            case1 = dist[prev][j-i][j] + count1 * penalty;

                            // find out how many bases before this loop
                            int element = pattern.firstAlongCircle(current);
                            int comp = pattern.getComponent(element);
                            int firstBase = 0;
                            if( comp < pattern.seqLength())
                                firstBase = comp;
                            else
                                firstBase = comp / pattern.seqLength() - 1;

                            int count2 = 0;
                            for(int m=firstBase; m<leftBase; m++)
                            {
                                if( pattern.getBase(m).equals("n") != true)
                                    count2 ++;
                            }

                            // compute the value of case-2
                            int division = j-i;
                            if( i == 0)     // the subsequence has only one nucleotide
                                case2 = encloseDist[current][j][j] + count2 * penalty;
                            else{  // we choose the optimal matching
                                for(int k=j-i; k<=j; k++)
                                {
                                    double holder;
                                    if (k==j-i)
                                        holder = encloseDist[current][k][j] + count2 * penalty;
                                    else
                                        holder = dist[prev][j-i][k-1] + encloseDist[current][k][j];

                                    if (holder > case2)
                                    {
                                        case2 = holder;
                                        division = k;
                                    }
                                }
                            }

                            dist[current][j-i][j] = Math.max(case1, case2);

                            if (case1 > case2)
                                distTrace[current][j-i][j] = 1;     // here we trace cas 1: the loop is cut off
                            else
                                distTrace[current][j-i][j] = -1 * division;   // here we remember the subsegment

                        } // end of updating inner circle distance
                    }  // finished of paired case: updating partial secondary structure distance
                }  // end for loop
              }  // end for the outer for loop of setting the starting base of segment

              // If we have processed a base-pair, we have to adjuct 'prevBase':
              if (component > pattern.seqLength())
                  prevBase = pattern.getPairedBase(leftBase);
              else
                  prevBase = component;

              prev = current;
              current ++;    // we tentatively go one step forward

              if (current < pattern.postOrderPairs.size())
              {
                  component = pattern.getComponent(current);
                  if (component > pattern.seqLength()){   // here we are finishing a circle
                                                          // from inside of it.
                      leftBase = component / pattern.seqLength() - 1;
                      rightBase = component % pattern.seqLength();
                  }
                  else if ( component > prevBase + 1) {   // here we are entering a new nested loop
                                                          // and finish from outside of it.
                      ComputeInnerCircleDist(pattern, current, pattern.getPairedBase(prevBase+1),
                            bSequence, startBase, endBase, dist, encloseDist, distTrace, encloseDistTrace,
                            matcher, penalty, extension);

                      // unpon return, we have to reposition current as follows:
                      current --;                                 // reset it as its previous value;
                      current = pattern.nextAlongCircle(current);  // this is definitly a base pair
                      component = pattern.getComponent(current);
                      leftBase = component / pattern.seqLength() - 1;
                      rightBase = component % pattern.seqLength();
                  }
                  else   // we stay inside the current loop and move forward: another single ribonucleotide
                  {
                      leftBase = component;
                      rightBase = component;
                  }
              }
          } //end of while loop
    }


    //----------------------------------------------------------------------------------------
    // This function match two RNA sequences, by tracing back calculating steps of "fold"
    // For the first RNA, the corresponding matched ribonucleotides are stored within 'aMap';
    // 'bMap' stores the same matching stuff for the second RNA.
    //---------------------------------------------------------------------------------------- 
    private static void MatchOneByOne(RNA pattern, int start, int end, int startBase,
                                    int endBase, int[][][] distTrace, int[][][] encloseDistTrace,
                                    int[] aMap, int[] bMap)
    {
        int basePair;
        while( start <= end && endBase - startBase >= 0)
        {
            basePair = pattern.getComponent(end);

            int tracebase;
            tracebase = distTrace[end][startBase][endBase];
            if (basePair < pattern.seqLength())       // single base
            {
                if (tracebase == 1)               // trace case1: 'end' element is not used
                    end = pattern.beforeAlongCircle(end);
                else if (tracebase == 2)          // trace case2: nucleotide [endBase] is not used
                    endBase --;
                else if (tracebase == 3)          //trace case3: matched with each other
                {
                    aMap[basePair] = endBase + 1;  //endBase + 1;
                    bMap[endBase] = basePair + 1;

                    end = pattern.beforeAlongCircle(end);
                    endBase --;
                }
            }
            else if (basePair > pattern.seqLength())  // base pair
            {
                int newend = pattern.beforeAlongCircle(end);
                if (newend < start)   // the backtracing is from within a circle
                {
                    tracebase = encloseDistTrace[end][startBase][endBase];
                    if (tracebase == 1)         // case 1
                        startBase ++;
                    else if (tracebase == 2)      // case 2
                        endBase --;
                    else if (tracebase == 3)      // case 3
                        end --;
                    else {
                        int left = basePair / pattern.seqLength() - 1;
                        int right = basePair % pattern.seqLength();
                        aMap[left] = startBase + 1;
                        aMap[right] = endBase + 1;   //endBase + 1;
                        bMap[startBase] = left + 1;
                        bMap[endBase] = right + 1;

                        end --;
                        startBase ++;
                        endBase --;
                    }
                }
                else {    // here we trace back from outer of circle
                    if (tracebase == 1)
                        end = newend;
                    else   // we divide our backtracing here into two parts
                    {
                        int division = -1 * tracebase;
                        MatchOneByOne(pattern, newend+1, end, division, endBase,
                                        distTrace, encloseDistTrace, aMap, bMap);
                        end = newend;
                        endBase = division - 1;
                    }
                }  // end of tracing from outside
            }  // end of base pair handling
        } // end of while
    }


    //---------------------------------------------------------------------------------------------------
    // This accomplishes the work of computing alignment between two RNA secondary structures without
    // gap punishment. That means trying to match as many bases as possible without violating structural
    // restrictions. The computation is global meaning the optimal result is accumulation of all partial
    // structural alignments.
    // 
    // Each RNA is confined by its corresponding starting secondary structural component and
    // ending secondary structural component. i.e. "aRNA" refers to the "partial structure"
    // defined by "start1" and "end1"; "bRNA" refers to the "partial structure" defined by "start2"
    // and "end2";
    //-------------------------------------------------------------------------------------------------- 
     private static short globalMatch2D(RNA aRNA, int start1, int end1,RNA bRNA, int start2, int end2,
                             short[][] score, short[][] semiCircleScore1,short[][] semiCircleScore2,
                             short[][] circleScore, byte[][] trace, byte[][] circleTrace,
                             byte[][] semiTrace1, byte[][] semiTrace2, boolean closeLoop, Matcher matcher)
     {
         int current1 = start1;
         int prev1  = current1;

	//#########################################################################
	int m1=0,m2=0;
	int mcomponent;
	double total_length= aRNA.seqLength();
	double conserved_length = 0;
	String send_conserved = null;

	if((aRNA.conserved != null)&&(aRNA.flag!=1)){
		conserved_length=aRNA.conserved.length();
	
		String c = aRNA.conserved;
		int count=0;
		for(int k = 0;k<conserved_length;k++){
			if(c.substring(k,k+1).equals("*")){
				count++;
			}
		}	
		conserved_length = count;
		send_conserved = aRNA.conserved;		
	}
	else if((bRNA.conserved != null) && (aRNA.flag != 1)){
		
		conserved_length=bRNA.conserved.length();
	
		String c = bRNA.conserved;
		int count=0;
		for(int k = 0;k<conserved_length;k++){
			if(c.substring(k,k+1).equals("*")){
				count++;
			}
		}	
		conserved_length = count;
		send_conserved = bRNA.conserved;
	}

	double factor = conserved_length/total_length;
//	System.out.println("factor in RNA =" + factor); 

	//#########################################################################
         if( start1 == end1 && (aRNA.beforeAlongCircle(start1)!=-1))
             prev1 = aRNA.beforeAlongCircle(start1);

         int component1 = aRNA.getComponent(current1);

         int prevBase1;    // this is the previous visisted BASE ( whether bonded or not).
         int leftBondEnd1; // 5' BASE of current visiting component.

         if( component1 < aRNA.seqLength())     // current component is singlet
         {
             prevBase1 = component1;
             leftBondEnd1 = component1;
         }
         else  // current component is a Bond( BASE pair)
         {
             prevBase1 = component1 % aRNA.seqLength();
             leftBondEnd1 = component1 / aRNA.seqLength() - 1;
         }

         // real computing begins here:
         do {
             if( component1 < aRNA.seqLength())  // component1 is a singlet.
             {
                 int current2 = start2;
                 int prev2 = current2;

                 if( start2 == end2 && bRNA.beforeAlongCircle(start2) != -1)
                      prev2 = bRNA.beforeAlongCircle(start2);

                 int component2 = bRNA.getComponent(current2);
                 int leftBondEnd2, prevBase2;
                 if( component2 < bRNA.seqLength())
                 {
                     prevBase2 = component2;
                     leftBondEnd2 = component2;
                 }
                 else {
                     prevBase2 = component2 % bRNA.seqLength();
                     leftBondEnd2 = component2 / bRNA.seqLength() -1;
                 }
                 int initBase2 = leftBondEnd2;

                 do {
                     if( component2 < bRNA.seqLength()) // case: singlet + singlet
                     {
                         short match = 0;
                         if( matcher == null)
                             match = (short)RNA.baseMatch(aRNA.getBaseBond(current1), bRNA.getBaseBond(current2));
                         else{   ////#############################change##################################
        		     mcomponent = aRNA.getComponent(current1);
        		     if (mcomponent < aRNA.sequence.length()){
            			m1 = mcomponent;
				m2 = -1;
			
        		     }else {
            			m1 = mcomponent / aRNA.seqLength() - 1;
           		        m2 = mcomponent % aRNA.seqLength();
        		     }
			     match = (short)matcher.baseMatch2(aRNA.flag,aRNA.getBaseBond(current1), bRNA.getBaseBond(current2), m1, m2, send_conserved,factor,aRNA.factors_for_conserved);
			 }
                         short case1 = 0;
                         if( (prev1 == start1 && start1 == end1) || (prev2 == start2 && start2 == end2)

                             || (start1 == current1 && start1 != end1) || (start2==current2 && start2 != end2))
                         {
                             case1 = match;
                         }
                         else
                         {
                             case1 = (short)(score[prev1][prev2] + match);
                         }

                         short case2 = 0;
                         if( (prev2 == start2 && start2 == end2) || (start2==current2 && start2 != end2))
                             case2 = 0;
                         else
                             case2 = score[current1][prev2];

                         short case3 = 0;
                         if( (prev1 == start1 && start1 == end1) || (start1==current1 && start1 != end1))
                             case3 = 0;
                         else
                             case3 = score[prev1][current2];

                         score[current1][current2] = (short)Math.max(case1, Math.max(case2,case3));

                         if( case2 >= case1 && case2 >= case3)  // case2
                             trace[current1][current2] = 2;
                         else if ( case3 >= case1)  // case3
                              trace[current1][current2] = 3;
                         else   // case1
                              trace[current1][current2] = 1;

                         prevBase2 = component2;
                     }
                     else  // component2 is a pair
                     {
                         // case of single base + bonded base pair
                         if ( leftBondEnd2 == prevBase2 + 1 || (end2 == start2 && closeLoop == false))
                         // component2 is a pair which is skipping a nested loop
                         {
                             // some preliminary step
                             if( end2 == start2 && closeLoop == false)
                                 prev2 = bRNA.beforeAlongCircle(current2);

                             // Here we consider three cases:
                             // case1: discard the induced loop, including the base pair
                             // case2: discard the single BASE
                             // case3: keep both loop and BASE.
                             // The MOST IMPORTANT thing here is the order of computing them

                             // FIRST, we need to compute CASE3:
                             // Note: score[*][current2] must be used to store inner loop score if
                             // current2 is not a first component of its enclosing loop
                             // Thus, need to backup the status of score[*][current2] and
                             // restore it after the calculation.
                             short case3 = 0;
                             if( end2 == start2 && closeLoop == false)
                             {  // the computing has been done before while circle <--> circle
                                case3 = semiCircleScore2[current2][current1];
                             }
                             else
                             {
                                int first = start1;
                                if( start1 == end1)
                                    first = aRNA.firstAlongCircle(current1);

                                 short[] backup = new short[current1-first+1];

                                 for(int i=0; i<backup.length; i++)
                                 {
                                     backup[i] = score[i+first][current2];
                                     score[i+first][current2] = semiCircleScore2[current2][i+first];
                                 }

                                 // Normally, we extend the computing
                                 // from: (start1, prev1) <--> circle(prev2+1, current2)
                                 // to: (start1, current1) <--> circle(prev2+1, current2);
                                 // otherwise, we have to do it from the scratch.
                                 // here the last parameter is ignored.
                                 case3 = globalMatch2D(aRNA, current1, current1,
                                                 bRNA, prev2+1, current2, score, semiCircleScore1,
                                                 semiCircleScore2, circleScore, trace, circleTrace,
                                                 semiTrace1, semiTrace2, closeLoop, matcher);

                                 for(int i=0; i<backup.length; i++)
                                     score[i+first][current2] = backup[i];
                             }

                             // Then, we are ready to compute CASE-1 and CASE-2
                             short case1 = score[current1][prev2];
                             short case2 = 0;
                             if( (prev1 == start1 && start1 == end1) || (start1 == current1 && start1 != end1))
                                 case2 = 0;
                             else
                                 case2 = score[prev1][current2];

                             short result = (short)Math.max(case1, Math.max(case2, case3));

                             score[current1][current2] = result;

                             if( case2 >= case1 && case2 >= case3)  // case2
                                 trace[current1][current2] = 2;
                             else if( case1 >= case3)  // case1
                                 trace[current1][current2] = 1;
                             else  // case3
                                 trace[current1][current2] = 3;
                         }
                         else {   // component2 is closing a loop
                             // Here two cases are considered:
                             // case1: discard enclosing base pair
                             // case2: discard the single BASE;
                             short case2 = 0;
                             if( end2 == start2 && closeLoop == true)
                             {
                                 prev2 = current2 - 1;
                                 if( current1 == start1)
                                     case2 = 0;
                                 else
                                     case2 = semiCircleScore2[current2][prev1];
                             }
                             else  // Normal case OR derivation of case3 of single + skipping above
                             {
                                 if( (prev1 == start1 && start1 == end1) || (current1==start1 && start1 != end1))
                                     case2 = 0;
                                 else
                                     case2 = semiCircleScore2[current2][prev1];
                             }

                             short case1 = score[current1][prev2];
                             short result = (short)Math.max(case1, case2);
                             score[current1][current2] = result;
                             semiCircleScore2[current2][current1] = result;

                             if( case2 >= case1)  // case2
                                 semiTrace2[current2][current1] = 12;
                             else  // case1
                                 semiTrace2[current2][current1] = 11;
                         }
                         prevBase2 = component2 % bRNA.seqLength();
                     }

                     prev2 = current2;
                     current2 = bRNA.nextAlongCircle(current2);
                     if(current2 != -1)
                     {
                         component2 = bRNA.getComponent(current2);
                         if( component2 < bRNA.seqLength())
                         {
                             leftBondEnd2 = component2;
                         }
                         else {
                             leftBondEnd2 = component2 / bRNA.seqLength() -1;
                         }
                     }
                 } while( current2 != -1 && current2 <= end2) ;
                 prevBase1 = component1;
             }
             else{    // component1 is a base pair
                 int current2 = start2;
                 int prev2 = current2;

                 if( end2==start2 && bRNA.beforeAlongCircle(start2) != -1)
                     prev2 = bRNA.beforeAlongCircle(start2);

                 int component2 = bRNA.getComponent(current2);
                 int leftBondEnd2, prevBase2;
                 if( component2 < bRNA.seqLength())
                 {
                     prevBase2 = component2;
                     leftBondEnd2 = component2;
                 }
                 else {
                     prevBase2 = component2 % bRNA.seqLength();
                     leftBondEnd2 = component2 / bRNA.seqLength() -1;
                 }
                 int initBase2 = leftBondEnd2;

                 do {
                     // case of bonded base pair(component1) + single base(component2)
                     if ( component2 < bRNA.seqLength())
                     {
                         if(leftBondEnd1 == prevBase1 + 1 || (end1 == start1 && closeLoop == false))
                         //component1 is base pair which is skipping the enclosed LOOP
                         {
                             if( end1 == start1 && closeLoop == false)
                                 prev1 = aRNA.beforeAlongCircle(current1);

                             // three cases are considered:
                             // case1: discard the whole enclosed LOOP,inclusing the pair
                             // case2: discard the single BASE
                             // case3: keep both LOOP and the single BASE
                             // The MOST IMPORTANT thing here is the order of computing them

                             // FIRST STEP: computing CASE-3
                             // score[current1][*] must be used to store inner loop score if
                             // current1 is not a first component of its loop
                             // Thus, need to backup the status of score[current1][*] and
                             // restore it after the calculation.
                             short case3 = 0;

                             if( end1 == start1 && closeLoop == false)
                             {
                                 case3 = semiCircleScore1[current1][current2];
                             }
                             else
                             {
                                 int first = start2;
                                 if( start2 == end2)
                                     first = bRNA.firstAlongCircle(first);

                                 short[] backup = new short[current2 - first +1];
                                 for(int i=0; i< backup.length; i++)
                                 {
                                     backup[i] = score[current1][i+first];
                                     score[current1][i+first] = semiCircleScore1[current1][i+first];
                                 }

                                 // Normally, we only need to extend the computing
                                 // from: circle(prev1+1) <--> (start2, prev2)
                                 // to:   circle(prev1+1) <--> (start2, current2)
                                 // Otherwise, we have to compute from the scratch
                                 case3 = globalMatch2D(aRNA,prev1+1, current1,
                                            bRNA, current2, current2, score, semiCircleScore1,
                                            semiCircleScore2, circleScore, trace, circleTrace,
                                            semiTrace1, semiTrace2, closeLoop, matcher);

                                 for(int i=0; i< backup.length; i++)
                                     score[current1][i+first] = backup[i];
                             }


                             // THEN, we are ready to compute CASE-1 and CASE-2
                             short case1 = score[prev1][current2] ;
                             short case2 = 0;
                             if( (prev2 == start2 && start2 == end2) || (current2==start2 && start2!=end2))
                                 case2 = 0;
                             else
                                 case2 = score[current1][prev2];


                             short result = (short)Math.max(case1, Math.max(case2, case3));
                             score[current1][current2] = result;

                             if( case2 >= case3 && case2 >= case1)  // case2
                                 trace[current1][current2] = 2;
                             else if(case1 >= case3)  // case1
                                 trace[current1][current2] = 1;
                             else  // case3
                                 trace[current1][current2] = 3;
                         }
                         else{  //componnet1 is a base pair which is closing the LOOP
                             // two cases are considered:
                             // case1: discard the base pair
                             // case2: discard the single BASE
                             short case2 = 0;
                             if(end1 == start1 && closeLoop == true)
                             {
                                 prev1 = current1 - 1;
                                 if( current2 == start2)
                                     case2 = 0;
                                 else
                                     case2 = semiCircleScore1[current1][prev2];
                             }
                             else    // Normal case OR derivation of case3 of skipping-loop + singlet above
                             {
                                 if( (start2 == prev2 && start2 == end2) || (current2==start2 && start2!=end2))
                                     case2 = 0;
                                 else
                                     case2 = semiCircleScore1[current1][prev2];
                             }

                             short case1 = score[prev1][current2];
                             short result = (short)Math.max(case1, case2);
                             score[current1][current2] = result;
                             semiCircleScore1[current1][current2] = result;

                             if( case2 >= case1)  // case2
                                 semiTrace1[current1][current2] = 12;
                             else  // case1
                                 semiTrace1[current1][current2] = 11;
                         }
                         prevBase2 = component2;
                     }
                     else{ // now, componet2 is also a base pair !!
                         // CASE of bonded base pair(component1->LOOP1) + bonded base pair(component2->LOOP2)
                         if( (leftBondEnd1 == prevBase1 + 1 || (start1==end1 && closeLoop == false)) &&
                             (leftBondEnd2 == prevBase2 + 1 || (start2==end2 && closeLoop == false)) )
                         {   // both are base pairs which are skipping the enclosed LOOPs
                             // five cases are considered:
                             // case1: both LOOPs are kept and matched with each other (2)
                             // case2: discard LOOP1; (0)
                             // case3: discard LOOP2; (1)
                             // case4: both LOOPs are kept where LOOP1 will match to the whole RNA2 (4)
                             // case5: both LOOPs are kets where LOOP2 will match to the whole RNA1 (3)

                             //!!!!!!!!! Here the order of computing is IMPORTANT !!!!!!!!!!
                             //FIRST STEP: We compute CASE-1
                             short temp1 = 0;
                             if( (start1==end1 && closeLoop==false) || (start2==end2 && closeLoop==false))
                             {
                                 temp1 = circleScore[current1][current2];

                                 if( start1 == end1 && closeLoop == false)
                                     prev1 = aRNA.beforeAlongCircle(current1);
                                 if( start2 == end2 && closeLoop == false)
                                     prev2 = bRNA.beforeAlongCircle(current2);
                             }
                             else
                             {
                                 short[] backup1 = null;
                                 short[] backup2 = null;

                                 //some bookkeeping work has to be done
                                 backup1 = new short[current2 - prev2];
                                 for(int i=0; i<backup1.length; i++)
                                 {
                                     backup1[i] = score[current1][i+(prev2+1)];
                                     score[current1][i+(prev2+1)] = semiCircleScore1[current1][i+(prev2+1)];
                                 }
                                 backup2 = new short[current1 - prev1];
                                 for(int i=0; i<backup2.length; i++)
                                 {
                                     backup2[i] = score[i+(prev1+1)][current2];
                                     score[i+(prev1+1)][current2] = semiCircleScore2[current2][i+(prev1+1)];
                                 }

                                 // the last parameter is ignored !!
                                 temp1 = globalMatch2D(aRNA, prev1+1, current1,
                                                 bRNA,prev2+1, current2, score, semiCircleScore1,
                                                 semiCircleScore2, circleScore, trace, circleTrace,
                                                 semiTrace1, semiTrace2, closeLoop, matcher);

                                  circleScore[current1][current2] = temp1;

                                  //restore the initial status.
                                 for(int i=0; i<backup1.length; i++)
                                     score[current1][i+(prev2+1)] = backup1[i];

                                 for(int i=0; i<backup2.length; i++)
                                     score[i+(prev1+1)][current2] = backup2[i];
                             }

                             short case1 = (short)(score[prev1][prev2] + temp1);
                             // CASE-1 is finished here


                             //SECOND STEP: we compute CASE-4
                             short case4 = 0;
                             if( start1==end1 && closeLoop==false)
                             {
                                 case4 = semiCircleScore1[current1][current2];
                             }
                             else
                             {
                                 //some bookkeeping work has to be done
                                 int first = start2;
                                 if( start2 == end2)
                                     first = bRNA.firstAlongCircle(start2);

                                 short[] backup = new short[prev2 - first + 1];
                                 for(int i=0; i<backup.length; i++)
                                 {
                                     backup[i] = score[current1][i+first];
                                     score[current1][i+first] = semiCircleScore1[current1][i+first];
                                 }

                                 // We only need to extend the computing
                                 // from:  circle(prev1+1, current1) <--> (start2, prev2)
                                 // to:    circle(prev1+1, current1) <--> (start2, current2)
                                 // Furthermore, the algorithm has to be notified that the base
                                 // pair, 'current2', is skipping a loop instead of closing loop,
                                 // to accomplish this, the last parameter is set to be 'false'
                                 case4 = globalMatch2D(aRNA, prev1+1, current1,
                                                bRNA, current2, current2, score, semiCircleScore1,
                                                semiCircleScore2, circleScore, trace, circleTrace,
                                                semiTrace1, semiTrace2, false, matcher);

                                 semiCircleScore1[current1][current2] = case4;

                                 //restore the initial status.
                                 for(int i=0; i<backup.length; i++)
                                    score[current1][i+first] = backup[i];
                             }
                             // CASE-4 is finished


                             // THIRD STEP: we compute CASE-5
                             short case5 = 0;
                             if( (start2==end2 && closeLoop==false))
                             {
                                 case5 = semiCircleScore2[current2][current1];
                             }
                             else
                             {
                                 //some bookkeeping work has to be done firstly
                                 int first = start1;
                                 if( start1 == end1)
                                     first = aRNA.firstAlongCircle(start1);
                                 short[] backup = new short[prev1-first+1];

                                 for(int i=0; i<backup.length; i++)
                                 {
                                     backup[i] = score[i+first][current2];
                                     score[i+first][current2] = semiCircleScore2[current2][i+first];
                                 }

                                 // Based on what has obtained from case1, we only need to extend the computing
                                 // from:   (start1, prev1) <--> circle(prev2+1, current2)
                                 // to:     (start1, current1) <--> circle(prev2+1, current2)
                                 // Furthermore, the algorithm has to be notified that the base
                                 // pair, 'current1', is skipping a loop instead of closing loop
                                 // to accomplish this, last parameter is set to be 'false'

                                 case5 = globalMatch2D(aRNA, current1, current1,
                                                bRNA, prev2+1, current2, score, semiCircleScore1,
                                                semiCircleScore2, circleScore, trace, circleTrace,
                                                semiTrace1, semiTrace2, false, matcher);

                                 semiCircleScore2[current2][current1] = case5;

                                 //restore the initial status.
                                 for(int i=0; i<backup.length; i++)
                                    score[i+first][current2] = backup[i];
                             }
                             // CASE-5 is finished !

                             // FOUTH and FIFTH STEPS are to compute CASE-2 and CASE-3
                             short case2 = score[prev1][current2];
                             short case3 = score[current1][prev2];

                             short result = (short)Math.max(case1, Math.max(case2, Math.max(case3,
                                                Math.max(case4, case5))));
                             score[current1][current2] = result;

                             if( case2>=case1 && case2>=case3 && case2 >= case5 && case2 >= case4)
                                 trace[current1][current2] = 2;
                             else if(case3>=case1 && case3 >= case5 && case3 >= case4)
                                 trace[current1][current2] = 3;
                             else if(case4>=case1 && case4>=case5)
                                 trace[current1][current2] = 4;
                             else if(case5>=case1)
                                 trace[current1][current2] = 5;
                             else
                                 trace[current1][current2] = 1;
                         }
                         else if( leftBondEnd1 == prevBase1+1 || (end1 == start1 && closeLoop==false) )
                         // component1 is skipping the enclosed LOOP1 while component2 is closing its LOOP2
                         {
                             // three cases are considered:
                             // case1: both LOOPs are kept and matched with each other (0)
                             // case2: LOOP1 is discarded.(1)
                             // case3: LOOP2 is opened by discarding the enclosing pair (2)

                             if( start2==end2 && closeLoop == true)
                                 prev2 = end2 - 1;

                             short case1 = 0;
                             if( end1==start1 && closeLoop == false)
                             {
                                 prev1 = aRNA.beforeAlongCircle(current1);
                                 case1 = circleScore[current1][current2];
                             }
                             else
                             {
                                 int first = start2;
                                 if( start2 == end2)
                                     first = bRNA.firstAlongCircle(start2-1);

                                 short[] backup = new short[current2 - first + 1];
                                 for(int i=0; i<backup.length;i++)
                                 {
                                     backup[i] = score[current1][i+first];
                                     score[current1][i+first] = semiCircleScore1[current1][i+first];
                                 }

                                 // Normally, we only need to extend computing
                                 // from: circle(prev1+1, current1) <--> (start2, prev2)
                                 // to: circle(prev1+1, current1) <--> circle(start2, current2);
                                 //
                                 // Furthermore, the algorithm has to be notified that pair(current2)
                                 // is closing loop instead of skipping loop
                                 // to acomplish this, the last parameter is set to be 'true'
                                 case1 = globalMatch2D(aRNA, prev1+1, current1,
                                                bRNA, current2, current2, score, semiCircleScore1,
                                                semiCircleScore2, circleScore, trace, circleTrace,
                                                semiTrace1, semiTrace2, true, matcher);
                                 circleScore[current1][current2] = case1;

                                 for(int i=0; i<backup.length; i++)
                                     score[current1][i+first] = backup[i];
                             }

                             short case2 = score[prev1][current2];
                             short case3 = score[current1][prev2];

                             short result = (short)Math.max(case1, Math.max(case2, case3));

                             score[current1][current2] = result;
                             semiCircleScore2[current2][current1] = result;

                             if( case2 >= case1 && case2 >= case3)    //case2
                                 semiTrace2[current2][current1] = 2;
                             else if( case3 >= case1)   // case3
                                 semiTrace2[current2][current1] = 3;
                             else   // case1
                                 semiTrace2[current2][current1] = 1;
                         }
                         else if( leftBondEnd2 == prevBase2+1 || (end2 == start2 && closeLoop == false))
                         // component1 is closing its LOOP1 while component2 is skipping the enclosed LOOP2
                         {
                             // two cases are considered:
                             // case1: both LOOPs are kept and matched with each other(0)
                             // case2: LOOP2 is discarded.(1)
                             // case3: LOOP1 is opened by discarding its enclosing pair(2)

                             if( start1 == end1 && closeLoop == true)
                                 prev1 = end1 - 1;

                             // get the first component of Loop2
                             if( end2 == start2 && closeLoop == false)
                                 prev2 = bRNA.beforeAlongCircle(current2);

                             short case1 = 0;
                             if( end2 == start2 && closeLoop == false)
                             {
                                 case1 = circleScore[current1][current2];
                             }
                             else
                             {
                                 int first = start1;
                                 if( start1 == end1)
                                     first = aRNA.firstAlongCircle(start1-1);

                                 short[] backup = new short[current1 - first + 1];
                                 for(int i=0; i<backup.length; i++)
                                 {
                                     backup[i] = score[i+first][current2];
                                     score[i+first][current2] = semiCircleScore2[current2][i+first];
                                 }

                                 // Normally, we only need to extend the computing
                                 // from: (start1, prev1) <--> circle(prev2+1, current2)
                                 // to:   circle(start1, current1) <--> circle(prev2+1, current2)
                                 //
                                 // Futhermore, the algorithm has to be notified that pair(current1)
                                 // is enclosing a loop instead of skipping a loop
                                 // to accomplish this, last parameter is set to be 'true'

                                 case1 = globalMatch2D(aRNA, current1, current1, bRNA, prev2+1,
                                                 current2, score, semiCircleScore1, semiCircleScore2,
                                                 circleScore, trace, circleTrace,
                                                 semiTrace1, semiTrace2, true, matcher);
                                 circleScore[current1][current2] = case1;

                                 for(int i=0; i<backup.length; i++)
                                     score[i+first][current2] = backup[i];
                             }

                             short case2 = score[current1][prev2];
                             short case3 = score[prev1][current2];

                             short result = (short)Math.max(case1, Math.max(case2, case3));
                             score[current1][current2] = result;
                             semiCircleScore1[current1][current2] = result;

                             if( case2 >= case3 && case2 >= case1)  //case2
                                 semiTrace1[current1][current2] = 2;
                             else if( case3 >= case1)  //case3
                                 semiTrace1[current1][current2] = 3;
                             else  //case1
                                 semiTrace1[current1][current2] = 1;
                         }
                         else // both components are closing the enclosed LOOPs.
                         {
                             // three cases are considered:
                             // case1: both components are kept and matched with each other
                             // case2: component1 is discarded
                             // case3: component2 is discarded
                             short match = 0;
                             if( matcher == null)
                                 match = (short)RNA.baseMatch(aRNA.getBaseBond(current1), bRNA.getBaseBond(current2));
                             else{///############################change#######################
                              
        		     	mcomponent = aRNA.getComponent(current1);
        		     	if (mcomponent < aRNA.sequence.length()){
            				m1 = mcomponent;
					m2 = -1;
			
        		        }else {
            				m1 = mcomponent / aRNA.seqLength() - 1;
           		       	        m2 = mcomponent % aRNA.seqLength();
        		        }	
			         match =(short)matcher.baseMatch2(aRNA.flag,aRNA.getBaseBond(current1), bRNA.getBaseBond(current2),m1,m2,send_conserved,factor,aRNA.factors_for_conserved);
			     }
                             if( end1 == start1 && closeLoop == true)
                                 prev1 = current1 - 1;
                             if( end2 == start2 && closeLoop == true)
                                 prev2 = current2 - 1;

                             short case1 = (short)(score[prev1][prev2] + match);
                             short case2 = score[prev1][current2];
                             short case3 = score[current1][prev2];

                             short result = (short)Math.max(case1, Math.max(case2,case3));

                             score[current1][current2] = result;
                             circleScore[current1][current2] = result;

                             if( aRNA.beforeAlongCircle(current1) == -1)  // first component of loop
                                 semiCircleScore2[current2][current1] = result;
                             if( bRNA.beforeAlongCircle(current2) == -1) // first component of loop
                                 semiCircleScore1[current1][current2] = result;

                             if( case2 >= case1 && case2 >= case3)
                                 circleTrace[current1][current2] = 2;
                             else if( case3 >= case1)
                                 circleTrace[current1][current2] = 3;
                             else 
                                 circleTrace[current1][current2] = 1;
                         }
                         prevBase2 = component2 % bRNA.seqLength();
                     }

                     prev2 = current2;
                     current2 = bRNA.nextAlongCircle(current2);
                     if(current2 != -1)
                     {
                         component2 = bRNA.getComponent(current2);
                         if( component2 < bRNA.seqLength())
                             leftBondEnd2 = component2;
                         else
                             leftBondEnd2 = component2 / bRNA.seqLength() -1;
                     }
                 } while ( current2 != -1 && current2 <= end2);

                 prevBase1 = component1 % aRNA.seqLength();
             }

             prev1 = current1;
             current1 = aRNA.nextAlongCircle(current1);
             if(current1 != -1)
             {
                 component1 = aRNA.getComponent(current1);
                 if( component1 < aRNA.seqLength())
                     leftBondEnd1 = component1;
                 else
                     leftBondEnd1 = component1 / aRNA.seqLength() - 1;
             }
         } while ( current1 != -1 && current1 <= end1);

         return score[end1][end2];
     }


    //--------------------------------------------------------------------------------------
    //  This function determing base-2-base mapping between two RNAs, by tracing back
    // the calculating steps of 'globalMatch2D'; the steps are stored within 'trace'
    // For the first RNA, the corresponding matched nucleotide bases from second RNA are
    // stored within 'aMap'; 'bMap' stores the same matching stuff for the second RNA.
    // 'aflag1' and 'aflag2' will be used when the tracing-back is dealing with loop(pair):
    // 'true' will skip the loop, 'false' will go inside the loop.
    //--------------------------------------------------------------------------------------
     private static void map2D(RNA aRNA, int start1, int end1, int[] aMap,
                             RNA bRNA, int start2, int end2, int[] bMap, byte[][] trace,
                             byte[][] circleTrace, byte[][] semiTrace1,
                             byte[][] semiTrace2, boolean aflag1, boolean aflag2, Matcher matcher)
     {
         boolean flag1 = aflag1;
         if( aRNA.beforeAlongCircle(end1) == -1)   // unique choice
             flag1 = false;

         boolean flag2 = aflag2;
         if( bRNA.beforeAlongCircle(end2) == -1)  // unique choice
             flag2 = false;

         int component1 = 0;
         int component2 = 0;
         boolean single1 = false;
         boolean single2 = false;

         do {
             component1 = aRNA.getComponent(end1);
             component2 = bRNA.getComponent(end2);

             single1 = (component1 < aRNA.seqLength());
             single2 = (component2 < bRNA.seqLength());

             if( single1 == true && single2 == true)
             {
                 int match = 0;
                 if( matcher == null)
                     match = RNA.baseMatch(aRNA.getBaseBond(end1), bRNA.getBaseBond(end2));
                 else{//####################################change###############################
                  
        	//	 mcomponent = aRNA.getComponent(end1);
        	//	 if (m1 < aRNA.sequence.length()){
            	//		m1 = mcomponent;
		//		m2 = -1;
		//	
        	//	  }else {
            	//		m1 = mcomponent / aRNA.seqLength() - 1;
           	//	        m2 = mcomponent % aRNA.seqLength();
        	//	  }
		     match = (int)matcher.baseMatch(aRNA.getBaseBond(end1), bRNA.getBaseBond(end2));
		
		 }
                 if(trace[end1][end2] == 2)
                 {
                     end2 = bRNA.beforeAlongCircle(end2);
                     flag2 = (bRNA.beforeAlongCircle(end2) != -1);
                 }
                 else if(trace[end1][end2] == 3)
                 {
                     end1 = aRNA.beforeAlongCircle(end1);
                     flag1 = (aRNA.beforeAlongCircle(end1) != -1);
                 }
                 else if( trace[end1][end2] == 1 && match > 0)
                 {
                     aMap[component1] = component2 + 1;   // base number is started from "1"
                     bMap[component2] = component1 + 1;
                     end1 = aRNA.beforeAlongCircle(end1);
                     end2 = bRNA.beforeAlongCircle(end2);

                     flag1 = (aRNA.beforeAlongCircle(end1) != -1);
                     flag2 = (bRNA.beforeAlongCircle(end2) != -1);
                 }
                 else if( trace[end1][end2] == 1)
                 {
                     end1 = aRNA.beforeAlongCircle(end1);
                     end2 = bRNA.beforeAlongCircle(end2);

                     flag1 = (aRNA.beforeAlongCircle(end1) != -1);
                     flag2 = (bRNA.beforeAlongCircle(end2) != -1);
                 }
                 else {
                     System.out.println("!!!!!!   ERROR !!!!!!!!");
                     System.out.println("single1 : " + single1 + "  single2:" + single2);
                     System.out.println("flag1 : " + flag1 + "  flag2:" + flag2);
                     System.out.println("trace[" + end1 + "][" + end2 + "]: " + trace[end1][end2]);
                     System.exit(0);
                 }
             }
             else if(single1 == true)
             {
                 if ( flag2 == true)  // component2 is skipping back base pair
                 {
                     if( trace[end1][end2] == 1)  // the loop is discarded
                     {
                         end2 = bRNA.beforeAlongCircle(end2);
                         flag2 = (bRNA.beforeAlongCircle(end2) != -1);
                     }
                     else if( trace[end1][end2] == 2) // the single base is discarded
                     {
                         end1 = aRNA.beforeAlongCircle(end1);
                         flag1 = (aRNA.beforeAlongCircle(end1) != -1);
                     }
                    else if( trace[end1][end2] == 3)
                    {
                        int newstart2 = bRNA.beforeAlongCircle(end2) + 1;
                        map2D(aRNA, start1, end1, aMap, bRNA, newstart2, end2, bMap, trace,
                                    circleTrace, semiTrace1, semiTrace2, flag1, false, matcher);
                        return;
                    }
                    else {
                          System.out.println("!!!!!!   ERROR !!!!!!!!");
                          System.out.println("single1 : " + single1 + "  single2:" + single2);
                          System.out.println("flag1 : " + flag1 + "  flag2:" + flag2);
                          System.out.println("trace[" + end1 + "][" + end2 + "]: " + trace[end1][end2]);
                          System.exit(0);
                     }
                 }
                 else if( flag2 == false) // component2 is enterring a loop
                 {
                     if( semiTrace2[end2][end1] == 11)   // the pair is discarded
                     {
                         end2 --;
                         flag2 = (bRNA.beforeAlongCircle(end2) != -1);
                     }
                     else if( semiTrace2[end2][end1] == 12)  // the single base is discarded
                     {
                         end1 = aRNA.beforeAlongCircle(end1);
                         flag1 = (aRNA.beforeAlongCircle(end1) != -1);
                     }
                     else
                     {
                         System.out.println("!!!!!!   ERROR !!!!!!!!");
                         System.out.println("single1 : " + single1 + "  single2:" + single2);
                         System.out.println("flag1 : " + flag1 + "  flag2:" + flag2);
                         System.out.println("semiTrace2[" + end2 + "][" + end1 + "]: " + semiTrace2[end2][end1]);
                         System.exit(0);
                     }
                 }
             }
             else if( single2 == true)
             {
                 if( flag1 == true)  // component1 is skipping back base pair
                 {
                     if( trace[end1][end2] == 1)  // loop is discarded
                     {
                         end1 = aRNA.beforeAlongCircle(end1);
                         flag1 = (aRNA.beforeAlongCircle(end1) != -1);
                     }
                     else if( trace[end1][end2] == 2) // single base is discarded
                     {
                         end2 = bRNA.beforeAlongCircle(end2);
                         flag2 = (bRNA.beforeAlongCircle(end2) != -1);
                     }
                     else if( trace[end1][end2] == 3)
                     {
                         int newstart1 = aRNA.beforeAlongCircle(end1) + 1;
                         map2D(aRNA, newstart1, end1, aMap, bRNA, start2, end2, bMap,
                                     trace, circleTrace, semiTrace1, semiTrace2, false, flag2, matcher);
                         return;
                     }
                     else {
                          System.out.println("!!!!!!   ERROR !!!!!!!!");
                          System.out.println("single1 : " + single1 + "  single2:" + single2);
                          System.out.println("flag1 : " + flag1 + "  flag2:" + flag2);
                          System.out.println("trace[" + end1 + "][" + end2 + "]: " + trace[end1][end2]);
                          System.exit(0);
                     }
                 }
                 else if( flag1 == false)
                 {
                     if( semiTrace1[end1][end2] == 11)  // discard the base pair
                     {
                         end1 --;
                         flag1 = (aRNA.beforeAlongCircle(end1) != -1);
                     }
                     else if( semiTrace1[end1][end2] == 12) //discard the single base
                     {
                         end2 = bRNA.beforeAlongCircle(end2);
                         flag2 = (bRNA.beforeAlongCircle(end2) != -1);
                     }
                     else {
                          System.out.println("!!!!!!   ERROR !!!!!!!!");
                          System.out.println("single1 : " + single1 + "  single2:" + single2);
                          System.out.println("flag1 : " + flag1 + "  flag2:" + flag2);
                          System.out.println("semiTrace1[" + end1 + "][" + end2 + "]: " + semiTrace1[end1][end2]);
                          System.exit(0);
                     }
                 }
             }
             else  // both are base pairs
             {
                 if( flag1 == true && flag2 == true)  // both components are skipping back pairs
                 {
                     int newstart1 = aRNA.beforeAlongCircle(end1) + 1;
                     int newstart2 = bRNA.beforeAlongCircle(end2) + 1;

                     byte choice = trace[end1][end2];
                     if( choice == 2)  // case2: discard loop1
                     {
                         end1 = aRNA.beforeAlongCircle(end1);
                         flag1 = (aRNA.beforeAlongCircle(end1) !=-1);
                     }
                     else if( choice == 3) // case3: discard loop2
                     {
                         end2 = bRNA.beforeAlongCircle(end2);
                         flag2 = (bRNA.beforeAlongCircle(end2) != -1);
                     }
                     else if( choice == 1)  // case1: loop1 <--> loop2
                     {
                         flag1 = (aRNA.beforeAlongCircle(newstart1-1) != -1);
                         flag2 = (bRNA.beforeAlongCircle(newstart2-1) != -1);

                         map2D(aRNA, start1, newstart1-1, aMap, bRNA, start2, newstart2-1, bMap,
                                         trace, circleTrace, semiTrace1, semiTrace2, flag1, flag2, matcher);

                         map2D(aRNA, newstart1, end1, aMap, bRNA, newstart2, end2, bMap, trace,
                                         circleTrace, semiTrace1, semiTrace2, false, false, matcher);
                         return;
                     }
                     else if( choice == 5) // case5: loop2 <--> whole RNA1
                     {
                         map2D(aRNA, start1, end1, aMap, bRNA, newstart2, end2, bMap, trace,
                                         circleTrace, semiTrace1, semiTrace2, flag1, false, matcher);
                         return;
                     }
                     else if( choice == 4) // case4: loop1 <---> whole RNA2
                     {
                         map2D(aRNA, newstart1, end1, aMap, bRNA, start2, end2, bMap, trace,
                                         circleTrace, semiTrace1, semiTrace2, false, flag1, matcher);
                         return;
                     }
                     else {
                          System.out.println("!!!!!!   ERROR !!!!!!!!");
                          System.out.println("single1 : " + single1 + "  single2:" + single2);
                          System.out.println("flag1 : " + flag1 + "  flag2:" + flag2);
                          System.out.println("trace[" + end1 + "][" + end2 + "]: " + trace[end1][end2]);
                          System.exit(0);
                     }
                 }
                 else if( flag1 == true)  // component1 is skipping and component2 is entering
                 {
                     int newstart1 = aRNA.beforeAlongCircle(end1) + 1;

                     byte choice = semiTrace2[end2][end1];
                     if( choice == 1) // case1: loop1 <--> loop2
                     {
                         map2D(aRNA, newstart1, end1, aMap, bRNA, start2, end2, bMap, trace,
                                         circleTrace, semiTrace1, semiTrace2, false, false, matcher);
                         return;
                     }
                     else if( choice == 2)  // case2: discard loop1
                     {
                         flag1 = (aRNA.beforeAlongCircle(newstart1-1) != -1);

                         map2D(aRNA, start1, newstart1-1, aMap, bRNA, start2, end2, bMap, trace,
                                         circleTrace, semiTrace1, semiTrace2, flag1, false, matcher);
                         return;
                     }
                     else if( choice == 3)  // case3: loop2 is opened
                     {
                         end2 --;
                         flag2 = (bRNA.beforeAlongCircle(end2) != -1);
                     }
                     else {
                          System.out.println("!!!!!!   ERROR !!!!!!!!");
                          System.out.println("single1 : " + single1 + "  single2:" + single2);
                          System.out.println("flag1 : " + flag1 + "  flag2:" + flag2);
                          System.out.println("semiTrace2[" + end2 + "][" + end1 + "]: " + semiTrace1[end2][end1]);
                          System.exit(0);
                     }
                 }
                 else if( flag2 == true)  // component2 is skipping and component1 is entering
                 {
                     int newstart2 = bRNA.beforeAlongCircle(end2) + 1;

                     byte choice = semiTrace1[end1][end2];
                     if( choice == 1)  // case1: loop1 <--> loop2
                     {
                         map2D(aRNA, start1, end1, aMap, bRNA, newstart2, end2, bMap, trace,
                                         circleTrace, semiTrace1, semiTrace2, false, false, matcher);
                         return;
                     }
                     else if( choice == 2)  // case2: loop2 is discarded
                     {
                         flag2 = (bRNA.beforeAlongCircle(newstart2-1) != -1);

                         map2D(aRNA, start1, end1, aMap, bRNA, start2, newstart2-1, bMap, trace,
                                         circleTrace, semiTrace1, semiTrace2, false, flag2, matcher);
                         return;
                     }
                     else if( choice == 3) // case3: loop1 is opened
                     {
                         end1 --;
                         flag1 = (aRNA.beforeAlongCircle(end1) != -1);
                     }
                     else {
                          System.out.println("!!!!!!   ERROR !!!!!!!!");
                          System.out.println("single1 : " + single1 + "  single2:" + single2);
                          System.out.println("flag1 : " + flag1 + "  flag2:" + flag2);
                          System.out.println("semiTrace1[" + end1 + "][" + end2 + "]: " + semiTrace1[end1][end2]);
                          System.exit(0);
                     }
                 }
                 else   if(flag1 == false && flag2 == false)// both are entering loop
                 {
                     int match = 0;
                     if( matcher == null)
                         match = RNA.baseMatch(aRNA.getBaseBond(end1), bRNA.getBaseBond(end2));
                     else{ ///############################CHANGE#########
                     
        	//	 mcomponent = aRNA.getComponent(end1);
        	//	 if (m1 < aRNA.sequence.length()){
            	//		m1 = mcomponent;
		//		m2 = -1;
		//	
        	//	 }else {
            	//		m1 = mcomponent / aRNA.seqLength() - 1;
           	//	        m2 = mcomponent % aRNA.seqLength();
        	//	 }
			 match = (int)matcher.baseMatch(aRNA.getBaseBond(end1), bRNA.getBaseBond(end2));
		     }
                     byte choice = circleTrace[end1][end2];

                     if( choice == 3)  // case3: pair2 is discarded
                     {
                         end2 --;
                         flag2 = (bRNA.beforeAlongCircle(end2) != -1);
                     }
                     else if( choice == 2) // case2: pair1 is discarded
                     {
                         end1 --;
                         flag1 = (aRNA.beforeAlongCircle(end1) != -1);
                     }
                     else if( choice == 1 && match > 0)  // case1: match
                     {
                         int left1 = component1 / aRNA.seqLength() - 1;
                         int right1 = component1 % aRNA.seqLength();

                         int left2 = component2 / bRNA.seqLength() - 1;
                         int right2 = component2 % bRNA.seqLength();

                         aMap[left1] = left2 + 1;    // base number is started from "1"
                         aMap[right1] = right2 + 1;
                         bMap[left2] = left1 + 1;
                         bMap[right2] = right1 + 1;

                         end1 --;
                         end2 --;
                         flag1 = (aRNA.beforeAlongCircle(end1) != -1);
                         flag2 = (bRNA.beforeAlongCircle(end2) != -1);
                     }
                     else if( choice == 1)
                     {
                         end1 --;
                         end2 --;
                         flag1 = (aRNA.beforeAlongCircle(end1) != -1);
                         flag2 = (bRNA.beforeAlongCircle(end2) != -1);
                     }
                     else {
                          System.out.println("!!!!!!   ERROR !!!!!!!!");
                          System.out.println("single1 : " + single1 + "  single2:" + single2);
                          System.out.println("flag1 : " + flag1 + "  flag2:" + flag2);
                          System.out.println("circleTrace[" + end1 + "][" + end2 + "]: " + circleTrace[end1][end2]);
                          System.out.println("choice: " + choice);
                          System.exit(0);
                     }
                 }
             }  // end of both pairs
         } while( start1 <= end1 && start2 <= end2);
     }

     //-------------------------------------------------------------------------------------------------
     //  This accomplishes the work of computing alignment between two RNA secondary structures WITH
     //  gap punishment. The first one is a 'MOTIF' small structure; the second is a text structure
     //  which may be very large.  This is half-global algorithm in that one the motif side it is global
     //  while on the side of text structure it is local.
     //-------------------------------------------------------------------------------------------------
     private static double motifDetector(RNA aRNA, int start1, int end1,RNA bRNA, int start2, int end2,
                             double[][] score, double[][] semiCircleScore1,double[][] semiCircleScore2,
                             double[][] circleScore, byte[][] trace, byte[][] circleTrace,
                             byte[][] semiTrace1, byte[][] semiTrace2, boolean closeLoop, Matcher matcher,
                             double penalty)
     {
         int current1 = start1;
         int prev1  = current1;

         if( start1 == end1 && (aRNA.beforeAlongCircle(start1)!=-1))
             prev1 = aRNA.beforeAlongCircle(start1);

         int component1 = aRNA.getComponent(current1);

         int prevBase1;    // this is the previous visisted BASE ( whether bonded or not).
         int leftBondEnd1; // 5' BASE of current visiting component.

         if( component1 < aRNA.seqLength()){     // current component is singlet
             prevBase1 = component1;
             leftBondEnd1 = component1;
         } else{  // current component is a Bond( BASE pair)
             prevBase1 = component1 % aRNA.seqLength();
             leftBondEnd1 = component1 / aRNA.seqLength() - 1;
         }

         // real computing begins here:
         do {
             if( component1 < aRNA.seqLength()){  // component1 is a singlet.
                 int current2 = start2;
                 int prev2 = current2;

                 if( start2 == end2 && bRNA.beforeAlongCircle(start2) != -1)
                     prev2 = bRNA.beforeAlongCircle(start2);

                 int component2 = bRNA.getComponent(current2);
                 int leftBondEnd2, prevBase2;
                 if( component2 < bRNA.seqLength()) {
                     prevBase2 = component2;
                     leftBondEnd2 = component2;
                 } else {
                     prevBase2 = component2 % bRNA.seqLength();
                     leftBondEnd2 = component2 / bRNA.seqLength() -1;
                 }
                 int initBase2 = leftBondEnd2;

                 do {
                     if( component2 < bRNA.seqLength()){ // case: singlet + singlet
                         double match = 0;
                         if( matcher == null)
                             match = (double)RNA.baseMatch(aRNA.getBaseBond(current1), bRNA.getBaseBond(current2));
                         else{  ////################################change##################
                          
        	//	     mcomponent = aRNA.getComponent(current1);
        	//	     if (m1 < aRNA.sequence.length()){
            	//		m1 = mcomponent;
		//		m2 = -1;
		//	
        	//	     }else {
            	//		m1 = mcomponent / aRNA.seqLength() - 1;
           	//	        m2 = mcomponent % aRNA.seqLength();
        	//	     }
			     match = matcher.baseMatch(aRNA.getBaseBond(current1), bRNA.getBaseBond(current2));
			 }
                         double case1 = 0;
                         if( (prev1 == start1 && start1 == end1) || (prev2 == start2 && start2 == end2)
                             || (start1 == current1 && start1 != end1) || (start2==current2 && start2 != end2))
                         {
                             if( leftBondEnd1 == 0)
                                 case1 = match;
                             else {
                                 int firstBase1=0;
                                 int firstBase2=0;

                                 int element = aRNA.firstAlongCircle(current1);
                                 int comp = aRNA.getComponent(element);
                                 if( comp < aRNA.seqLength())
                                     firstBase1 = comp;
                                 else
                                     firstBase1 = comp / aRNA.seqLength() - 1;

                                 element = bRNA.firstAlongCircle(current2);
                                 comp = bRNA.getComponent(element);
                                 if( comp < bRNA.seqLength())
                                     firstBase2 = comp;
                                 else
                                     firstBase2 = comp / bRNA.seqLength() - 1;

                                 // pick out those "optional" bases:
                                 int n1 = 0;
                                 for(int i=firstBase1; i<leftBondEnd1; i++) {
                                     if( aRNA.getBase(i).equals("n") == true)
                                         n1 ++;
                                 }
                                 case1 = ((leftBondEnd1-firstBase1 - n1
                                             + leftBondEnd2-firstBase2)*penalty + match);
                             }
                         } else
                             case1 = (score[prev1][prev2] + match);

                         double case2 = 0;
                         if( (prev2 == start2 && start2 == end2) || (start2==current2 && start2 != end2))
                         {
                             // motif structure before 'current1' is also cut off now
                             int element = aRNA.firstAlongCircle(current1);
                             int comp = aRNA.getComponent(element);
                             int firstBase = 0;
                             if( comp < aRNA.seqLength())
                                 firstBase = comp;
                             else
                                 firstBase = comp / aRNA.seqLength() - 1;

                             int currentBase1 = aRNA.getComponent(current1);
                             int n1 = 0;
                             for(int i=firstBase; i<=currentBase1; i++) {
                                 if( aRNA.getBase(i).equals("n") == true)
                                     n1 ++;
                             }

                             int currentBase2 = bRNA.getComponent(current2);
                             if( bRNA.getBase(currentBase2).equals("n") == true)
                                 case2 = ( (currentBase1 - firstBase + 1 - n1) * penalty);
                             else
                                 case2 = ( (currentBase1 - firstBase + 1 - n1) * penalty + penalty);
                         } else {
                             int currentBase2 = bRNA.getComponent(current2);
                             if( bRNA.getBase(currentBase2).equals("n") == true)
                                 case2 = (score[current1][prev2]);
                             else
                                 case2 = (score[current1][prev2] + penalty);
                         }

                         double case3 = 0;
                         if( (prev1 == start1 && start1 == end1) || (start1==current1 && start1 != end1))
                         {
                             int currentBase1 = aRNA.getComponent(current1);
                             if( leftBondEnd1 == 0){  // current1 is on the topmost loop
                                 if( aRNA.getBase(currentBase1).equals("n") == true)
                                     case3 = 0;
                                 else
                                     case3 = penalty;
                             } else {
                                 int element = bRNA.firstAlongCircle(current2);
                                 int comp = bRNA.getComponent(element);
                                 int firstBase = 0;
                                 if( comp < bRNA.seqLength())
                                     firstBase = comp;
                                 else
                                     firstBase = comp / bRNA.seqLength() - 1;

                                 int currentBase2 = bRNA.getComponent(current2);
                                 if( aRNA.getBase(currentBase1).equals("n") == true)
                                     case3 = ( (currentBase2 - firstBase + 1) * penalty );
                                 else
                                     case3 = ( (currentBase2 - firstBase + 1)*penalty + penalty);
                             }
                         } else {
                             int currentBase1 = aRNA.getComponent(current1);
                             if( aRNA.getBase(currentBase1).equals("n") == true)
                                 case3 = (score[prev1][current2]);
                             else
                                 case3 = (score[prev1][current2] + penalty);
                         }

                         score[current1][current2] = Math.max(case1, Math.max(case2,case3));

                         if( case1 >= case2 && case1 >= case3)  // case1: match
                             trace[current1][current2] = 1;
                         else if( case2 >= case3)  // case2
                             trace[current1][current2] = 2;
                         else   // case3
                             trace[current1][current2] = 3;

                         prevBase2 = component2;

                     } else{  // component2 is a pair
                         // case of single base + bonded base pair
                         if ( leftBondEnd2 == prevBase2 + 1 || (end2 == start2 && closeLoop == false))
                         // component2 is a pair which is skipping a nested loop
                         {
                             // some preliminary step
                             if( end2 == start2 && closeLoop == false)
                                 prev2 = bRNA.beforeAlongCircle(current2);

                             // Here we consider three cases:
                             // case1: discard the induced loop, including the base pair
                             // case2: discard the single BASE
                             // case3: keep both loop and BASE.
                             // The MOST IMPORTANT thing here is the order of computing them

                             // FIRST, we need to compute CASE3:
                             // Note: score[*][current2] must be used to store inner loop score if
                             // current2 is not a first component of its enclosing loop
                             // Thus, need to backup the status of score[*][current2] and
                             // restore it after the calculation.
                             double case3 = 0;
                             if( end2 == start2 && closeLoop == false)
                             {  // the computing has been done before while circle <--> circle
                                 int element = aRNA.firstAlongCircle(current1);
                                 int comp = aRNA.getComponent(element);
                                 int firstBase = comp;
                                 if( firstBase < aRNA.seqLength())
                                     firstBase = comp;
                                 else
                                     firstBase = comp / aRNA.seqLength() - 1;

                                 if( firstBase == 0)  // 'current1' is ON topmost loop
                                     case3 = semiCircleScore2[current2][current1];
                                 else{
                                     element = bRNA.firstAlongCircle(current2);
                                     comp = bRNA.getComponent(element);
                                     if( comp < bRNA.seqLength())
                                         firstBase = comp;
                                     else
                                         firstBase = comp / bRNA.seqLength() - 1;

                                     case3 = (semiCircleScore2[current2][current1] +
                                                            (leftBondEnd2 - firstBase) * penalty);
                                 }
                             } else {
                                 int first = start1;
                                 if( start1 == end1)
                                     first = aRNA.firstAlongCircle(current1);

                                 double[] backup = new double[current1-first+1];
                                 for(int i=0; i<backup.length; i++) {
                                     backup[i] = score[i+first][current2];
                                     score[i+first][current2] = semiCircleScore2[current2][i+first];
                                 }

                                 // Normally, we extend the computing
                                 // from: (start1, prev1) <--> circle(prev2+1, current2)
                                 // to: (start1, current1) <--> circle(prev2+1, current2);
                                 // otherwise, we have to do it from the scratch.
                                 // here the last parameter is ignored.
                                 case3 = motifDetector(aRNA, current1, current1,
                                                 bRNA, prev2+1, current2, score, semiCircleScore1,
                                                 semiCircleScore2, circleScore, trace, circleTrace,
                                                 semiTrace1, semiTrace2, closeLoop, matcher, penalty);

                                 semiCircleScore2[current2][current1] = case3;

                                 int element = aRNA.firstAlongCircle(current1);
                                 int comp = aRNA.getComponent(element);
                                 int firstBase = comp;
                                 if( firstBase < aRNA.seqLength())
                                     firstBase = comp;
                                 else
                                     firstBase = comp / aRNA.seqLength() - 1;

                                 if( firstBase != 0) {
                                     element = bRNA.firstAlongCircle(current2);
                                     comp = bRNA.getComponent(element);
                                     if( comp < bRNA.seqLength())
                                         firstBase = comp;
                                     else
                                         firstBase = comp / bRNA.seqLength() - 1;

                                     case3 = (case3 + (leftBondEnd2 - firstBase) * penalty);
                                 }

                                 for(int i=0; i<backup.length; i++)
                                     score[i+first][current2] = backup[i];
                             }

                             // Then, we are ready to compute CASE-1 and CASE-2
                             int currentBase2 = bRNA.getPairedBase(leftBondEnd2);
                             double case1 = (score[current1][prev2] +
                                                 (currentBase2 - leftBondEnd2 + 1)*penalty);

                             double case2 = 0;
                             int currentBase1 = aRNA.getComponent(current1);
                             if( (prev1 == start1 && start1 == end1) || (start1 == current1 && start1 != end1))
                             { // current1 is the first single base along loop circumference
                                 if( leftBondEnd1 == 0){  // current1 is on the topmost loop
                                     if( aRNA.getBase(currentBase1).equals("n") == true)
                                         case2 = 0;
                                     else
                                         case2 = penalty;
                                 } else {
                                     int element = bRNA.firstAlongCircle(current2);
                                     int comp = bRNA.getComponent(element);
                                     int firstBase = 0;
                                     if( comp < bRNA.seqLength())
                                         firstBase = comp;
                                     else
                                         firstBase = comp / bRNA.seqLength() - 1;

                                     if( aRNA.getBase(currentBase1).equals("n") == true)
                                         case2 = ((currentBase2 - firstBase + 1)*penalty);
                                     else
                                         case2 = ((currentBase2 - firstBase + 1)*penalty + penalty);
                                 }
                             } else {
                                 if( aRNA.getBase(currentBase1).equals("n") == true)
                                     case2 = (score[prev1][current2]);
                                 else
                                     case2 = (score[prev1][current2] + penalty);
                             }

                             double result = Math.max(case1, Math.max(case2, case3));
                             score[current1][current2] = result;

                             if( case3 >= case1 && case3 >= case2)  // case3
                                 trace[current1][current2] = 3;
                             else if( case2 >= case1)  // case2
                                 trace[current1][current2] = 2;
                             else  // case1
                                 trace[current1][current2] = 1;
                         } else {   // component2 is closing a loop
                             // Here two cases are considered:
                             // case1: discard enclosing base pair
                             // case2: discard the single BASE;
                             double case2 = 0;
                             int currentBase1 = aRNA.getComponent(current1);
                             int currentBase2 = bRNA.getPairedBase(leftBondEnd2);

                             if( end2 == start2 && closeLoop == true) {
                                 prev2 = current2 - 1;

                                 if( current1 == start1){     // 'current1' is the first on loop
                                     if( leftBondEnd1 != 0){  // 'current1' is not on topmost loop
                                         case2 = ( (currentBase2 - leftBondEnd2 + 1) * penalty);
                                         if( aRNA.getBase(currentBase1).equals("n") != true)
                                             case2 += penalty;
                                     } else {
                                         if( aRNA.getBase(currentBase1).equals("n") == true)
                                             case2 = 0;
                                         else
                                             case2 = penalty;
                                     }
                                 } else {
                                     if( aRNA.getBase(currentBase1).equals("n") == true)
                                         case2 = (semiCircleScore2[current2][prev1]);
                                     else
                                         case2 = (semiCircleScore2[current2][prev1] + penalty);
                                 }
                             } else{  // Normal case OR derivation of case3 of single + skipping above
                                 if( (prev1 == start1 && start1 == end1) || (current1==start1 && start1 != end1))
                                 { // 'current1' is the first element on loop
                                     if( leftBondEnd1 != 0){   // 'current1' is not on topmost loop
                                        int n2 = 0;
                                        case2 = ((currentBase2- leftBondEnd2 + 1 - n2) * penalty);
                                        if( aRNA.getBase(currentBase1).equals("n") != true)
                                            case2 += penalty;
                                     } else {
                                         if( aRNA.getBase(currentBase1).equals("n") == true)
                                             case2 = 0;
                                         else
                                             case2 = penalty;
                                     }
                                 } else {
                                     if( aRNA.getBase(currentBase1).equals("n") == true)
                                         case2 = (semiCircleScore2[current2][prev1]);
                                     else
                                         case2 = (semiCircleScore2[current2][prev1] + penalty);
                                 }
                             }

                             double case1 = 0;
                             if( bRNA.getBase(currentBase2).equals("n") == true)
                                 case1 = (score[current1][prev2]);
                             else
                                 case1 = (score[current1][prev2] + 2*penalty);

                             double result = Math.max(case1, case2);
                             score[current1][current2] = result;
                             semiCircleScore2[current2][current1] = result;

                             if( case2 >= case1)  // case2
                                 semiTrace2[current2][current1] = 12;
                             else  // case1
                                 semiTrace2[current2][current1] = 11;
                         }
                         prevBase2 = component2 % bRNA.seqLength();
                     }

                     prev2 = current2;
                     current2 = bRNA.nextAlongCircle(current2);
                     if(current2 != -1) {
                         component2 = bRNA.getComponent(current2);
                         if( component2 < bRNA.seqLength())
                             leftBondEnd2 = component2;
                         else
                             leftBondEnd2 = component2 / bRNA.seqLength() -1;
                     }
                 } while( current2 != -1 && current2 <= end2) ;
                 prevBase1 = component1;
             } else{    // component1 is a base pair
                 int current2 = start2;
                 int prev2 = current2;

                 if( end2==start2 && bRNA.beforeAlongCircle(start2) != -1)
                     prev2 = bRNA.beforeAlongCircle(start2);

                 int component2 = bRNA.getComponent(current2);
                 int leftBondEnd2, prevBase2;
                 if( component2 < bRNA.seqLength()) {
                      prevBase2 = component2;
                      leftBondEnd2 = component2;
                 } else {
                      prevBase2 = component2 % bRNA.seqLength();
                      leftBondEnd2 = component2 / bRNA.seqLength() -1;
                 }
                 int initBase2 = leftBondEnd2;

                 do {
                     // case of bonded base pair(component1) + single base(component2)
                     if ( component2 < bRNA.seqLength()) {
                         if(leftBondEnd1 == prevBase1 + 1 || (end1 == start1 && closeLoop == false))
                         //component1 is base pair which is skipping the enclosed LOOP
                         {
                             if( end1 == start1 && closeLoop == false)
                                 prev1 = aRNA.beforeAlongCircle(current1);

                             // three cases are considered:
                             // case1: discard the whole enclosed LOOP,inclusing the pair
                             // case2: discard the single BASE
                             // case3: keep both LOOP and the single BASE
                             // The MOST IMPORTANT thing here is the order of computing them

                             // FIRST STEP: computing CASE-3
                             // score[current1][*] must be used to store inner loop score if
                             // current1 is not a first component of its loop
                             // Thus, need to backup the status of score[current1][*] and
                             // restore it after the calculation.
                             double case3 = 0;

                             if( end1 == start1 && closeLoop == false) {
                                 int element = aRNA.firstAlongCircle(current1);
                                 int comp = aRNA.getComponent(element);
                                 int firstBase = 0;
                                 if( comp < aRNA.seqLength())
                                     firstBase = comp;
                                 else
                                     firstBase = comp / aRNA.seqLength() - 1;

                                 int n1 = 0;
                                 for(int i=firstBase; i<leftBondEnd1; i++) {
                                     if( aRNA.getBase(i).equals("n") == true)
                                         n1 ++;
                                 }
                                 case3 = (semiCircleScore1[current1][current2]
                                                         + (leftBondEnd1 - firstBase - n1)*penalty);
                             } else {
                                 int first = start2;
                                 if( start2 == end2)
                                    first = bRNA.firstAlongCircle(first);

                                 double[] backup = new double[current2 - first +1];
                                 for(int i=0; i< backup.length; i++) {
                                     backup[i] = score[current1][i+first];
                                     score[current1][i+first] = semiCircleScore1[current1][i+first];
                                 }

                                 // Normally, we only need to extend the computing
                                 // from: circle(prev1+1) <--> (start2, prev2)
                                 // to:   circle(prev1+1) <--> (start2, current2)
                                 // Otherwise, we have to compute from the scratch
                                 case3 = motifDetector(aRNA,prev1+1, current1,
                                            bRNA, current2, current2, score, semiCircleScore1,
                                            semiCircleScore2, circleScore, trace, circleTrace,
                                            semiTrace1, semiTrace2, closeLoop, matcher, penalty);

                                 semiCircleScore1[current1][current2] = case3;

                                 int element = aRNA.firstAlongCircle(current1);
                                 int comp = aRNA.getComponent(element);
                                 int firstBase = 0;
                                 if( comp < aRNA.seqLength())
                                     firstBase = comp;
                                 else
                                     firstBase = comp / aRNA.seqLength() - 1;

                                 int n1 = 0;
                                 for(int i=firstBase; i<leftBondEnd1; i++) {
                                     if( aRNA.getBase(i).equals("n") == true)
                                         n1 ++;
                                 }
                                 case3 = (case3 + ( leftBondEnd1 - firstBase - n1)*penalty);

                                 for(int i=0; i< backup.length; i++)
                                     score[current1][i+first] = backup[i];
                             }


                             // THEN, we are ready to compute CASE-1 and CASE-2
                             int currentBase2 = bRNA.getComponent(current2);
                             int currentBase1 = aRNA.getPairedBase(leftBondEnd1);
                             int n1 = 0;
                             for(int i=leftBondEnd1; i<=currentBase1; i++) {
                                 if( aRNA.getBase(i).equals("n") == true)
                                     n1 ++;
                             }
                             double case1 = (score[prev1][current2] +
                                             (currentBase1 - leftBondEnd1 + 1 - n1)*penalty);

                             double case2 = 0;
                             if( (prev2 == start2 && start2 == end2) || (current2==start2 && start2!=end2))
                             { // current2 is the FIRST single base along loop circumference
                                int element = aRNA.firstAlongCircle(current1);
                                int comp = aRNA.getComponent(element);
                                int firstBase = 0;
                                if( comp < aRNA.seqLength())
                                    firstBase = comp;
                                else
                                    firstBase = comp / aRNA.seqLength() - 1;

                                 n1 = 0;
                                 for(int i=firstBase; i<=currentBase1; i++) {
                                     if( aRNA.getBase(i).equals("n") == true)
                                         n1 ++;
                                 }

                                 case2 = ((currentBase1 - firstBase + 1 - n1)*penalty );

                                 if( bRNA.getBase(currentBase2).equals("n") != true)
                                     case2 += penalty;
                             } else {
                                 case2 = (score[current1][prev2]);

                                 if( bRNA.getBase(currentBase2).equals("n") != true)
                                     case2 = (score[current1][prev2] + penalty);
                             }


                             double result = Math.max(case1, Math.max(case2, case3));

                             score[current1][current2] = result;

                             if( case3 >= case1 && case3 >= case2) // case3
                                 trace[current1][current2] = 3;
                             else if( case2 >= case1)  // case2
                                 trace[current1][current2] = 2;
                             else   // case1
                                 trace[current1][current2] = 1;

                         } else{ //componnet1 is a base pair which is closing the LOOP
                             // two cases are considered:
                             // case1: discard the base pair
                             // case2: discard the single BASE
                             double case2 = 0;
                             int currentBase1= aRNA.getPairedBase(leftBondEnd1);
                             int currentBase2= bRNA.getComponent(current2);

                             prev1 = current1 - 1;
                             int n1 = 0;
                             for(int i=leftBondEnd1; i<=currentBase1; i++) {
                                if( aRNA.getBase(i).equals("n") == true)
                                     n1 ++;
                             }

                             if(end1 == start1 && closeLoop == true) {
                                 if( current2 == start2)
                                     case2 = (penalty + (currentBase1-leftBondEnd1+1-n1)*penalty);
                                 else
                                     case2 = (semiCircleScore1[current1][prev2] + penalty);

                                 if( bRNA.getBase(currentBase2).equals("n") == true)
                                     case2 -= penalty;
                             } else{    // Normal case OR derivation of case3 of skipping-loop + singlet above
                                 if( (start2 == prev2 && start2 == end2) || (current2==start2 && start2!=end2))
                                     case2 = (penalty + (currentBase1-leftBondEnd1+1-n1)*penalty);
                                 else
                                     case2 = (semiCircleScore1[current1][prev2] + penalty);

                                 if( bRNA.getBase(currentBase2).equals("n") == true)
                                     case2 -= penalty;
                             }

                             double case1 = 0;
                             if( aRNA.getBase(currentBase1).equals("n") == true)
                                 case1 = (score[prev1][current2]);
                             else
                                 case1 = (score[prev1][current2] + 2*penalty);

                             double result = Math.max(case1, case2);

                             score[current1][current2] = result;
                             semiCircleScore1[current1][current2] = result;

                             if( case2 >= case1)  // case2
                                 semiTrace1[current1][current2] = 12;
                             else  // case1
                                 semiTrace1[current1][current2] = 11;
                         }

                         prevBase2 = component2;
                     } else { // now, componet2 is also a base pair !!
                         // CASE of bonded base pair(component1->LOOP1) + bonded base pair(component2->LOOP2)
                         if( (leftBondEnd1 == prevBase1 + 1 || (start1==end1 && closeLoop == false)) &&
                             (leftBondEnd2 == prevBase2 + 1 || (start2==end2 && closeLoop == false)) )
                         {   // both are base pairs which are skipping the enclosed LOOPs
                              // five cases are considered:
                              // case1: both LOOPs are kept and matched with each other (2)
                              // case2: discard LOOP1; (0)
                              // case3: discard LOOP2; (1)
                              // case4: both LOOPs are kept where LOOP1 will match to the whole RNA2 (4)
                              // case5: both LOOPs are kets where LOOP2 will match to the whole RNA1 (3)

                             //!!!!!!!!! Here the order of computing is IMPORTANT !!!!!!!!!!
                             //FIRST STEP: We compute CASE-1
                             double temp1 = 0;
                             if( (start1==end1 && closeLoop==false) || (start2==end2 && closeLoop==false))
                             {
                                 temp1 = circleScore[current1][current2];

                                 if( start1 == end1 && closeLoop == false)
                                     prev1 = aRNA.beforeAlongCircle(current1);
                                 if( start2 == end2 && closeLoop == false)
                                     prev2 = bRNA.beforeAlongCircle(current2);
                             } else {
                                 double[] backup1 = null;
                                 double[] backup2 = null;

                                 //some bookkeeping work has to be done
                                 backup1 = new double[current2 - prev2];
                                 for(int i=0; i<backup1.length; i++) {
                                    backup1[i] = score[current1][i+(prev2+1)];
                                    score[current1][i+(prev2+1)] = semiCircleScore1[current1][i+(prev2+1)];
                                 }
                                 backup2 = new double[current1 - prev1];
                                 for(int i=0; i<backup2.length; i++) {
                                    backup2[i] = score[i+(prev1+1)][current2];
                                    score[i+(prev1+1)][current2] = semiCircleScore2[current2][i+(prev1+1)];
                                 }

                                 // the last parameter is ignored !!
                                 temp1 = motifDetector(aRNA, prev1+1, current1,
                                                 bRNA,prev2+1, current2, score, semiCircleScore1,
                                                 semiCircleScore2, circleScore, trace, circleTrace,
                                                 semiTrace1, semiTrace2, closeLoop, matcher, penalty);

                                 circleScore[current1][current2] = temp1;

                                 //restore the initial status.
                                 for(int i=0; i<backup1.length; i++)
                                     score[current1][i+(prev2+1)] = backup1[i];

                                 for(int i=0; i<backup2.length; i++)
                                     score[i+(prev1+1)][current2] = backup2[i];
                             }

                             double case1 = (score[prev1][prev2] + temp1);
                             // CASE-1 is finished here

                             //SECOND STEP: we compute CASE-4
                             double case4 = 0;
                             if( start1==end1 && closeLoop==false) {
                                 int element = aRNA.firstAlongCircle(current1);
                                 int comp = aRNA.getComponent(element);
                                 int firstBase = 0;
                                 if( comp < aRNA.seqLength())
                                     firstBase = comp;
                                 else
                                     firstBase = comp / aRNA.seqLength() - 1;

                                 int n1 = 0;
                                 for(int i=firstBase; i<leftBondEnd1; i++) {
                                     if( aRNA.getBase(i).equals("n") == true)
                                         n1 ++;
                                 }
                                 case4 = (semiCircleScore1[current1][current2]
                                                     + (leftBondEnd1 - firstBase - n1)*penalty);
                             } else {
                                 int first = start2;
                                 if( start2 == end2)
                                     first = bRNA.firstAlongCircle(start2);

                                 double[] backup = new double[prev2 - first + 1];

                                 for(int i=0; i<backup.length; i++) {
                                     backup[i] = score[current1][i+first];
                                     score[current1][i+first] = semiCircleScore1[current1][i+first];
                                 }

                                 // We only need to extend the computing
                                 // from:  circle(prev1+1, current1) <--> (start2, prev2)
                                 // to:    circle(prev1+1, current1) <--> (start2, current2)
                                 // Furthermore, the algorithm has to be notified that the base
                                 // pair, 'current2', is skipping a loop instead of closing loop,
                                 // to accomplish this, the last parameter is set to be 'false'
                                 case4 = motifDetector(aRNA, prev1+1, current1,
                                                 bRNA, current2, current2, score, semiCircleScore1,
                                                 semiCircleScore2, circleScore, trace, circleTrace,
                                                 semiTrace1, semiTrace2, false, matcher, penalty);

                                 semiCircleScore1[current1][current2] = case4;

                                 int element = aRNA.firstAlongCircle(current1);
                                 int comp = aRNA.getComponent(element);
                                 int firstBase = 0;
                                 if( comp < aRNA.seqLength())
                                     firstBase = comp;
                                 else
                                     firstBase = comp / aRNA.seqLength() - 1;

                                 int n1 = 0;
                                 for(int i=firstBase; i<leftBondEnd1; i++) {
                                     if( aRNA.getBase(i).equals("n") == true)
                                         n1 ++;
                                 }
                                 case4 = (case4 + (leftBondEnd1 - firstBase - n1)*penalty);

                                 //restore the initial status.
                                 for(int i=0; i<backup.length; i++)
                                     score[current1][i+first] = backup[i];
                             }
                             // CASE-4 is finished


                             // THIRD STEP: we compute CASE-5
                             double case5 = 0;
                             if( (start2==end2 && closeLoop==false)) {
                                 int element = aRNA.firstAlongCircle(current1);
                                 int comp = aRNA.getComponent(element);
                                 int firstBase = 0;
                                 if( comp < aRNA.seqLength())
                                     firstBase = comp;
                                 else
                                     firstBase = comp / aRNA.seqLength() -  1;

                                 if( firstBase == 0)  // current1 is on topmost loop
                                     case5 = semiCircleScore2[current2][current1];
                                 else {
                                     element = bRNA.firstAlongCircle(current2);
                                     comp = bRNA.getComponent(element);
                                     firstBase = 0;
                                     if( comp < bRNA.seqLength())
                                         firstBase = comp;
                                     else
                                         firstBase = comp / bRNA.seqLength() - 1;

                                     case5 = (semiCircleScore2[current2][current1]
                                                         + (leftBondEnd2 - firstBase)*penalty);
                                 }
                             } else {
                                 //some bookkeeping work has to be done firstly
                                 int first = start1;
                                 if( start1 == end1)
                                     first = aRNA.firstAlongCircle(start1);
                                 double[] backup = new double[prev1-first+1];

                                 for(int i=0; i<backup.length; i++) {
                                     backup[i] = score[i+first][current2];
                                     score[i+first][current2] = semiCircleScore2[current2][i+first];
                                 }

                                 // Based on what has obtained from case1, we only need to extend the computing
                                 // from:   (start1, prev1) <--> circle(prev2+1, current2)
                                 // to:     (start1, current1) <--> circle(prev2+1, current2)
                                 // Furthermore, the algorithm has to be notified that the base
                                 // pair, 'current1', is skipping a loop instead of closing loop
                                 // to accomplish this, last parameter is set to be 'false'

                                 case5 = motifDetector(aRNA, current1, current1,
                                                 bRNA, prev2+1, current2, score, semiCircleScore1,
                                                 semiCircleScore2, circleScore, trace, circleTrace,
                                                 semiTrace1, semiTrace2, false, matcher, penalty);

                                 semiCircleScore2[current2][current1] = case5;

                                 int element = aRNA.firstAlongCircle(current1);
                                 int comp = aRNA.getComponent(element);
                                 int firstBase = 0;
                                 if( comp < aRNA.seqLength())
                                     firstBase = comp;
                                 else
                                     firstBase = comp / aRNA.seqLength() -  1;

                                 if( firstBase != 0){   // current1 is NOT on topmost loop
                                     element = bRNA.firstAlongCircle(current2);
                                     comp = bRNA.getComponent(element);
                                     firstBase = 0;
                                     if( comp < bRNA.seqLength())
                                         firstBase = comp;
                                     else
                                         firstBase = comp / bRNA.seqLength() - 1;

                                     case5 = (case5 + (leftBondEnd2 - firstBase)*penalty);
                                 }

                                 //restore the initial status.
                                 for(int i=0; i<backup.length; i++)
                                     score[i+first][current2] = backup[i];
                             }
                             // CASE-5 is finished !

                             // FOUTH and FIFTH STEPS are to compute CASE-2 and CASE-3
                             int currentBase1 = aRNA.getPairedBase(leftBondEnd1);
                             int currentBase2 = bRNA.getPairedBase(leftBondEnd2);
                             int n1 = 0;
                             int n2 = 0;
                             for(int i=leftBondEnd1; i<=currentBase1; i++) {
                                 if( aRNA.getBase(i).equals("n") == true)
                                     n1 ++;
                             }

                             double case2 = (score[prev1][current2] +
                                             (currentBase1 - leftBondEnd1 + 1 - n1)*penalty);

                             double case3 = (score[current1][prev2] +
                                             (currentBase2 - leftBondEnd2 + 1 - n2)*penalty);

                             double result = Math.max(case1, Math.max(case2, Math.max(case3,
                                                 Math.max(case4, case5))));
                             score[current1][current2] = result;

                             if( case1>=case2 && case1>=case3 && case1>=case4 && case1>=case5)
                                 trace[current1][current2] = 1;
                             else if( case4>=case2 && case4>=case3 && case4>=case5)
                                 trace[current1][current2] = 4;
                             else if( case5>=case2 && case5>=case3)
                                 trace[current1][current2] = 5;
                             else if( case2>=case3)
                                 trace[current1][current2] = 2;
                             else
                                 trace[current1][current2] = 3;
                         } else if( leftBondEnd1 == prevBase1+1 || (end1 == start1 && closeLoop==false) )
                         // component1 is skipping the enclosed LOOP1 while component2 is closing its LOOP2
                         {
                             // three cases are considered:
                             // case1: both LOOPs are kept and matched with each other (0)
                             // case2: LOOP1 is discarded.(1)
                             // case3: LOOP2 is opened by discarding the enclosing pair (2)

                             if( start2==end2 && closeLoop == true)
                                 prev2 = end2 - 1;

                             double case1 = 0;
                             if( end1==start1 && closeLoop == false) {
                                 prev1 = aRNA.beforeAlongCircle(current1);

                                 int element = aRNA.firstAlongCircle(current1);
                                 int comp = aRNA.getComponent(element);
                                 int firstBase = 0;
                                 if( comp < aRNA.seqLength())
                                     firstBase = comp;
                                 else
                                     firstBase = comp / aRNA.seqLength() - 1;

                                 int n1 = 0;
                                 for(int i=firstBase; i<leftBondEnd1; i++) {
                                     if( aRNA.getBase(i).equals("n") == true)
                                         n1 ++;
                                 }
                                 case1 = (circleScore[current1][current2]
                                                             + (leftBondEnd1-firstBase - n1)*penalty);
                             } else {
                                 int first = start2;
                                 if( start2 == end2)
                                     first = bRNA.firstAlongCircle(start2-1);

                                 double[] backup = new double[current2 - first + 1];
                                 for(int i=0; i<backup.length;i++) {
                                     backup[i] = score[current1][i+first];
                                     score[current1][i+first] = semiCircleScore1[current1][i+first];
                                 }

                                 // Normally, we only need to extend computing
                                 // from: circle(prev1+1, current1) <--> (start2, prev2)
                                 // to: circle(prev1+1, current1) <--> circle(start2, current2);
                                 //
                                 // Furthermore, the algorithm has to be notified that pair(current2)
                                 // is closing loop instead of skipping loop
                                 // to acomplish this, the last parameter is set to be 'true'
                                 case1 = motifDetector(aRNA, prev1+1, current1,
                                                 bRNA, current2, current2, score, semiCircleScore1,
                                                 semiCircleScore2, circleScore, trace, circleTrace,
                                                 semiTrace1, semiTrace2, true, matcher, penalty);

                                 circleScore[current1][current2] = case1;

                                 int element = aRNA.firstAlongCircle(current1);
                                 int comp = aRNA.getComponent(element);
                                 int firstBase = 0;
                                 if( comp < aRNA.seqLength())
                                     firstBase = comp;
                                 else
                                     firstBase = comp / aRNA.seqLength() - 1;

                                 int n1 = 0;
                                 for(int i=firstBase; i<leftBondEnd1; i++) {
                                     if( aRNA.getBase(i).equals("n") == true)
                                         n1 ++;
                                 }
                                 case1 = (case1 + (leftBondEnd1-firstBase-n1)*penalty);

                                 for(int i=0; i<backup.length; i++)
                                     score[current1][i+first] = backup[i];
                             }

                             int currentBase1 = aRNA.getPairedBase(leftBondEnd1);
                             int n1 = 0;
                             for(int i=leftBondEnd1; i<=currentBase1; i++) {
                                 if( aRNA.getBase(i).equals("n") == true)
                                     n1 ++;
                             }
                             double case2 = (score[prev1][current2] +
                                                 (currentBase1 - leftBondEnd1 + 1 - n1)*penalty);

                             double case3 = 0;
                             int currentBase2 = bRNA.getPairedBase(leftBondEnd2);

                             if( bRNA.getBase(currentBase2).equals("n") == true)
                                 case3 = (score[current1][prev2]);
                             else
                                 case3 = (score[current1][prev2] + 2*penalty);

                             double result = Math.max(case1, Math.max(case2, case3));

                             score[current1][current2] = result;
                             semiCircleScore2[current2][current1] = result;

                             if( case1 >= case2 && case1 >= case3)
                                 semiTrace2[current2][current1] = 1;
                             else if( case3 >= case2)
                                 semiTrace2[current2][current1] = 3;
                             else    //case2
                                 semiTrace2[current2][current1] = 2;

                         } else if( leftBondEnd2 == prevBase2+1 || (end2 == start2 && closeLoop == false))
                         // component1 is closing its LOOP1 while component2 is skipping the enclosed LOOP2
                         {
                             // two cases are considered:
                             // case1: both LOOPs are kept and matched with each other(0)
                             // case2: LOOP2 is discarded.(1)
                             // case3: LOOP1 is opened by discarding its enclosing pair(2)

                             if( start1 == end1 && closeLoop == true)
                                 prev1 = end1 - 1;

                             // get the first component of Loop2
                             if( end2 == start2 && closeLoop == false)
                                 prev2 = bRNA.beforeAlongCircle(current2);

                             double case1 = 0;
                             if( end2 == start2 && closeLoop == false) {
                                 int element = aRNA.firstAlongCircle(current1);
                                 int comp = aRNA.getComponent(element);
                                 int firstBase = 0;
                                 if( comp < aRNA.seqLength())
                                     firstBase = comp;
                                 else
                                     firstBase = comp / aRNA.seqLength() - 1;

                                 if( firstBase == 0)  // current1 is on topmost loop
                                    case1 = circleScore[current1][current2];
                                 else {
                                    element = bRNA.firstAlongCircle(current2);
                                    comp = bRNA.getComponent(element);
                                    firstBase = 0;
                                    if( comp < bRNA.seqLength())
                                        firstBase = comp;
                                    else
                                        firstBase = comp / bRNA.seqLength() - 1;

                                     case1 = (circleScore[current1][current2]
                                                         + (leftBondEnd2 - firstBase)*penalty);
                                 }
                             } else {
                                 int first = start1;
                                 if( start1 == end1)
                                     first = aRNA.firstAlongCircle(start1-1);

                                 double[] backup = new double[current1 - first + 1];
                                 for(int i=0; i<backup.length; i++) {
                                     backup[i] = score[i+first][current2];
                                     score[i+first][current2] = semiCircleScore2[current2][i+first];
                                 }

                                 // Normally, we only need to extend the computing
                                 // from: (start1, prev1) <--> circle(prev2+1, current2)
                                 // to:   circle(start1, current1) <--> circle(prev2+1, current2)
                                 //
                                 // Futhermore, the algorithm has to be notified that pair(current1)
                                 // is enclosing a loop instead of skipping a loop
                                 // to accomplish this, last parameter is set to be 'true'

                                 case1 = motifDetector(aRNA, current1, current1, bRNA, prev2+1,
                                                 current2, score, semiCircleScore1, semiCircleScore2,
                                                 circleScore, trace, circleTrace,
                                                 semiTrace1, semiTrace2, true, matcher, penalty);

                                 circleScore[current1][current2] = case1;

                                 // test if 'current1' stay in topmost loop
                                 int element = aRNA.firstAlongCircle(current1);
                                 int comp = aRNA.getComponent(element);
                                 int firstBase = 0;
                                 if( comp < aRNA.seqLength())
                                     firstBase = comp;
                                 else
                                     firstBase = comp / aRNA.seqLength() - 1;

                                 if( firstBase != 0){  // 'current1' is NOT on topmost loop
                                     element = bRNA.firstAlongCircle(current2);
                                     comp = bRNA.getComponent(element);
                                     firstBase = 0;
                                     if( comp < bRNA.seqLength())
                                         firstBase = comp;
                                     else
                                         firstBase = comp / bRNA.seqLength() - 1;

                                     case1 = (case1 + (leftBondEnd2-firstBase)*penalty);
                                 }

                                  for(int i=0; i<backup.length; i++)
                                      score[i+first][current2] = backup[i];
                             }

                             int currentBase2 = bRNA.getPairedBase(leftBondEnd2);
                             double case2 = (score[current1][prev2] +
                                            (currentBase2 - leftBondEnd2 + 1)*penalty);

                             int currentBase1 = aRNA.getPairedBase(leftBondEnd1);
                             double case3 = 0;
                             if( aRNA.getBase(currentBase1).equals("n") == true)
                                 case3 = (score[prev1][current2]);
                             else
                                 case3 = (score[prev1][current2] + 2*penalty);

                             double result = Math.max(case1, Math.max(case2, case3));

                             score[current1][current2] = result;
                             semiCircleScore1[current1][current2] = result;

                             if( case1 >= case2 && case1 >= case3)
                                 semiTrace1[current1][current2] = 1;
                             else if( case3 >= case2)
                                 semiTrace1[current1][current2] = 3;
                             else  //case2
                                 semiTrace1[current1][current2] = 2;

                         } else {    // both components are closing the enclosed LOOPs.
                             // three cases are considered:
                             // case1: both components are kept and matched with each other
                             // case2: component1 is discarded
                             // case3: component2 is discarded
                             double match = 0;
                             if( matcher == null)
                                 match = (double)RNA.baseMatch(aRNA.getBaseBond(current1), bRNA.getBaseBond(current2));
                             else{    ///############################change################
                             
        	//	     	mcomponent = aRNA.getComponent(current1);
        	//	     	if (m1 < aRNA.sequence.length()){
            	//			m1 = mcomponent;
		//			m2 = -1;
		//	
        	  //       	}else {
            	//			m1 = mcomponent / aRNA.seqLength() - 1;
           	//	        	m2 = mcomponent % aRNA.seqLength();
        	//	        }
			        match = matcher.baseMatch(aRNA.getBaseBond(current1), bRNA.getBaseBond(current2));
			     }
                             
			     if( end1 == start1 && closeLoop == true)
                                 prev1 = current1 - 1;
                             if( end2 == start2 && closeLoop == true)
                                 prev2 = current2 - 1;

                             int currentBase1 = aRNA.getPairedBase(leftBondEnd1);
                             int currentBase2 = bRNA.getPairedBase(leftBondEnd2);

                             double case1 = (score[prev1][prev2] + match);

                             double case2 = 0;
                             if( aRNA.getBase(currentBase1).equals("n") == true)
                                 case2 = (score[prev1][current2]);
                             else
                                 case2 = (score[prev1][current2] + 2*penalty);

                             double case3 = 0;
                             if( bRNA.getBase(currentBase2).equals("n") == true)
                                 case3 = (score[current1][prev2]);
                             else
                                 case3 = (score[current1][prev2] + 2*penalty);

                             double result = Math.max(case1, Math.max(case2,case3));

                             score[current1][current2] = result;
                             circleScore[current1][current2] = result;

                             if( aRNA.beforeAlongCircle(current1) == -1)  // first component of loop
                                 semiCircleScore2[current2][current1] = result;
                             if( bRNA.beforeAlongCircle(current2) == -1) // first component of loop
                                 semiCircleScore1[current1][current2] = result;

                             if( case1 >= case2 && case1 >= case3)
                                 circleTrace[current1][current2] = 1;
                             else if( case2 >= case3)
                                 circleTrace[current1][current2] = 2;
                             else
                                 circleTrace[current1][current2] = 3;
                         }

                         prevBase2 = component2 % bRNA.seqLength();
                     }

                     prev2 = current2;
                     current2 = bRNA.nextAlongCircle(current2);
                     if(current2 != -1) {
                         component2 = bRNA.getComponent(current2);
                         if( component2 < bRNA.seqLength())
                             leftBondEnd2 = component2;
                         else
                             leftBondEnd2 = component2 / bRNA.seqLength() -1;
                      }
                 } while ( current2 != -1 && current2 <= end2);

                 prevBase1 = component1 % aRNA.seqLength();
             }

             prev1 = current1;
             current1 = aRNA.nextAlongCircle(current1);
             if(current1 != -1) {
                 component1 = aRNA.getComponent(current1);
                 if( component1 < aRNA.seqLength())
                     leftBondEnd1 = component1;
                 else
                     leftBondEnd1 = component1 / aRNA.seqLength() - 1;
             }
         } while ( current1 != -1 && current1 <= end1);

         return score[end1][end2];
     }

     //------------------------------------------------------------------------------------
     // locate the specific structural region which is the optimal candidate of to 
     // fold into the given motif profile structure. And find the detailed alignment by
     // calling 'traceMotifMap2D()'.
     //------------------------------------------------------------------------------------
     private static double motifMap2D(RNA motif, int[] aMap, RNA subject, int[] bMap, 
                             byte[][] trace, byte[][] circleTrace, byte[][] semiTrace1,
                             byte[][] semiTrace2, double[][] score, double[][] circleScore)
     {
         double max = Double.NEGATIVE_INFINITY;
         int row = motif.postOrderPairs.size();
         int col = subject.postOrderPairs.size();

         int position = 0;
         for(int j=col-1; j>=0; j--) {
             if( score[row-1][j] > max) {
                 max = score[row-1][j];
                 position = j;
             }
         }

         boolean flag1 = (motif.beforeAlongCircle(row-1) != -1);
         boolean flag2 = (subject.beforeAlongCircle(position) != -1);

         if( aMap != null && bMap != null) {
             traceMotifMap2D(motif, 0, row-1, aMap, subject, 0, position, bMap,
                        trace, circleTrace, semiTrace1, semiTrace2, flag1, flag2);
         }

         return max;
     }

     //---------------------------------------------------------------------------------------
     // recover the detailed structure alignment information from motif-oriented (half-global)
     // structural alignment.
     //---------------------------------------------------------------------------------------
     private static void traceMotifMap2D(RNA aRNA, int start1, int end1, int[] aMap,
                             RNA bRNA, int start2, int end2, int[] bMap, byte[][] trace,
                             byte[][] circleTrace, byte[][] semiTrace1,
                             byte[][] semiTrace2, boolean aflag1, boolean aflag2)
     {
         boolean flag1 = aflag1;
         if( aRNA.beforeAlongCircle(end1) == -1)   // unique choice
             flag1 = false;

         boolean flag2 = aflag2;
         if( bRNA.beforeAlongCircle(end2) == -1)  // unique choice
             flag2 = false;

         int component1 = 0;
         int component2 = 0;
         boolean single1 = false;
         boolean single2 = false;

         do {
             component1 = aRNA.getComponent(end1);
             component2 = bRNA.getComponent(end2);

             single1 = (component1 < aRNA.seqLength());
             single2 = (component2 < bRNA.seqLength());

             if( single1 == true && single2 == true) {
                 if(trace[end1][end2] == 2) {
                     end2 = bRNA.beforeAlongCircle(end2);
                     flag2 = (bRNA.beforeAlongCircle(end2) != -1);
                 } else if(trace[end1][end2] == 3) {
                     end1 = aRNA.beforeAlongCircle(end1);
                     flag1 = (aRNA.beforeAlongCircle(end1) != -1);
                 } else if( trace[end1][end2] == 1) {
                     aMap[component1] = component2 + 1;   // base number is started from "1"
                     bMap[component2] = component1 + 1;
                     end1 = aRNA.beforeAlongCircle(end1);
                     end2 = bRNA.beforeAlongCircle(end2);

                     flag1 = (aRNA.beforeAlongCircle(end1) != -1);
                     flag2 = (bRNA.beforeAlongCircle(end2) != -1);
                 } else {
                     System.out.println("!!!!!!   ERROR !!!!!!!!");
                     System.out.println("single1 : " + single1 + "  single2:" + single2);
                     System.out.println("flag1 : " + flag1 + "  flag2:" + flag2);
                     System.out.println("trace[" + end1 + "][" + end2 + "]: " + trace[end1][end2]);
                     System.exit(0);
                 }
             } else if(single1 == true) {
                 if ( flag2 == true){                    // component2 is skipping back base pair
                     if( trace[end1][end2] == 1){        // the loop is discarded
                         end2 = bRNA.beforeAlongCircle(end2);
                         flag2 = (bRNA.beforeAlongCircle(end2) != -1);
                     } else if( trace[end1][end2] == 2){ // the single base is discarded
                         end1 = aRNA.beforeAlongCircle(end1);
                         flag1 = (aRNA.beforeAlongCircle(end1) != -1);
                     } else if( trace[end1][end2] == 3) {
                         int newstart2 = bRNA.beforeAlongCircle(end2) + 1;
                         traceMotifMap2D(aRNA, start1, end1, aMap, bRNA, newstart2, end2, bMap, trace,
                                    circleTrace, semiTrace1, semiTrace2, flag1, false);
                         return;
                    } else {
                         System.out.println("!!!!!!   ERROR !!!!!!!!");
                         System.out.println("single1 : " + single1 + "  single2:" + single2);
                         System.out.println("flag1 : " + flag1 + "  flag2:" + flag2);
                         System.out.println("trace[" + end1 + "][" + end2 + "]: " + trace[end1][end2]);
                         System.exit(0);
                    }
                 } else if( flag2 == false) {             // component2 is enterring a loop
                     if( semiTrace2[end2][end1] == 11){   // the pair is discarded
                         end2 --;
                         flag2 = (bRNA.beforeAlongCircle(end2) != -1);
                     } else if( semiTrace2[end2][end1] == 12){  // the single base is discarded
                         end1 = aRNA.beforeAlongCircle(end1);
                         flag1 = (aRNA.beforeAlongCircle(end1) != -1);
                     } else {
                         System.out.println("!!!!!!   ERROR !!!!!!!!");
                         System.out.println("single1 : " + single1 + "  single2:" + single2);
                         System.out.println("flag1 : " + flag1 + "  flag2:" + flag2);
                         System.out.println("semiTrace2[" + end2 + "][" + end1 + "]: " + semiTrace2[end2][end1]);
                         System.exit(0);
                     }
                 }
             } else if( single2 == true) {
                 if( flag1 == true){               // component1 is skipping back base pair
                     if( trace[end1][end2] == 1){  // loop is discarded
                         end1 = aRNA.beforeAlongCircle(end1);
                         flag1 = (aRNA.beforeAlongCircle(end1) != -1);
                     } else if( trace[end1][end2] == 2){ // single base is discarded
                         end2 = bRNA.beforeAlongCircle(end2);
                         flag2 = (bRNA.beforeAlongCircle(end2) != -1);
                     } else if( trace[end1][end2] == 3) {
                         int newstart1 = aRNA.beforeAlongCircle(end1) + 1;
                         traceMotifMap2D(aRNA, newstart1, end1, aMap, bRNA, start2, end2, bMap,
                                     trace, circleTrace, semiTrace1, semiTrace2, false, flag2);
                         return;
                     } else {
                         System.out.println("!!!!!!   ERROR !!!!!!!!");
                         System.out.println("single1 : " + single1 + "  single2:" + single2);
                         System.out.println("flag1 : " + flag1 + "  flag2:" + flag2);
                         System.out.println("trace[" + end1 + "][" + end2 + "]: " + trace[end1][end2]);
                         System.exit(0);
                     }
                 } else if( flag1 == false) {
                     if( semiTrace1[end1][end2] == 11){  // discard the base pair
                         end1 --;
                         flag1 = (aRNA.beforeAlongCircle(end1) != -1);
                     } else if( semiTrace1[end1][end2] == 12){ //discard the single base
                         end2 = bRNA.beforeAlongCircle(end2);
                         flag2 = (bRNA.beforeAlongCircle(end2) != -1);
                     } else {
                          System.out.println("!!!!!!   ERROR !!!!!!!!");
                          System.out.println("single1 : " + single1 + "  single2:" + single2);
                          System.out.println("flag1 : " + flag1 + "  flag2:" + flag2);
                          System.out.println("semiTrace1[" + end1 + "][" + end2 + "]: " + semiTrace1[end1][end2]);
                          System.exit(0);
                     }
                 }
             } else { // both are base pairs
                 if( flag1 == true && flag2 == true){  // both components are skipping back pairs
                     int newstart1 = aRNA.beforeAlongCircle(end1) + 1;
                     int newstart2 = bRNA.beforeAlongCircle(end2) + 1;

                     byte choice = trace[end1][end2];

                     if( choice == 2){  // case2: discard loop1
                         end1 = aRNA.beforeAlongCircle(end1);
                         flag1 = (aRNA.beforeAlongCircle(end1) !=-1);
                     } else if( choice == 3){ // case3: discard loop2
                         end2 = bRNA.beforeAlongCircle(end2);
                         flag2 = (bRNA.beforeAlongCircle(end2) != -1);
                     } else if( choice == 1){  // case1: loop1 <--> loop2
                         flag1 = (aRNA.beforeAlongCircle(newstart1-1) != -1);
                         flag2 = (bRNA.beforeAlongCircle(newstart2-1) != -1);

                         traceMotifMap2D(aRNA, start1, newstart1-1, aMap, bRNA, start2, newstart2-1, bMap,
                                         trace, circleTrace, semiTrace1, semiTrace2, flag1, flag2);

                         traceMotifMap2D(aRNA, newstart1, end1, aMap, bRNA, newstart2, end2, bMap, trace,
                                         circleTrace, semiTrace1, semiTrace2, false, false);
                         return;
                     } else if( choice == 5){ // case5: loop2 <--> whole RNA1
                         traceMotifMap2D(aRNA, start1, end1, aMap, bRNA, newstart2, end2, bMap, trace,
                                         circleTrace, semiTrace1, semiTrace2, flag1, false);
                         return;
                     } else if( choice == 4){ // case4: loop1 <---> whole RNA2
                         traceMotifMap2D(aRNA, newstart1, end1, aMap, bRNA, start2, end2, bMap, trace,
                                         circleTrace, semiTrace1, semiTrace2, false, flag1);
                         return;
                     } else {
                          System.out.println("!!!!!!   ERROR !!!!!!!!");
                          System.out.println("single1 : " + single1 + "  single2:" + single2);
                          System.out.println("flag1 : " + flag1 + "  flag2:" + flag2);
                          System.out.println("trace[" + end1 + "][" + end2 + "]: " + trace[end1][end2]);
                          System.exit(0);
                     }
                 } else if( flag1 == true){  // component1 is skipping and component2 is entering
                     int newstart1 = aRNA.beforeAlongCircle(end1) + 1;

                     byte choice = semiTrace2[end2][end1];
                     if( choice == 1){  // case1: loop1 <--> loop2
                         traceMotifMap2D(aRNA, newstart1, end1, aMap, bRNA, start2, end2, bMap, trace,
                                         circleTrace, semiTrace1, semiTrace2, false, false);
                         return;
                     } else if( choice == 2){  // case2: discard loop1
                         flag1 = (aRNA.beforeAlongCircle(newstart1-1) != -1);

                         traceMotifMap2D(aRNA, start1, newstart1-1, aMap, bRNA, start2, end2, bMap, trace,
                                         circleTrace, semiTrace1, semiTrace2, flag1, false);
                         return;
                     } else if( choice == 3){  // case3: loop2 is opened
                         end2 --;
                         flag2 = (bRNA.beforeAlongCircle(end2) != -1);
                     } else {
                          System.out.println("!!!!!!   ERROR !!!!!!!!");
                          System.out.println("single1 : " + single1 + "  single2:" + single2);
                          System.out.println("flag1 : " + flag1 + "  flag2:" + flag2);
                          System.out.println("semiTrace2[" + end2 + "][" + end1 + "]: " + semiTrace1[end2][end1]);
                          System.exit(0);
                     }
                 } else if( flag2 == true){  // component2 is skipping and component1 is entering
                     int newstart2 = bRNA.beforeAlongCircle(end2) + 1;

                     byte choice = semiTrace1[end1][end2];
                     if( choice == 1){  // case1: loop1 <--> loop2
                         traceMotifMap2D(aRNA, start1, end1, aMap, bRNA, newstart2, end2, bMap, trace,
                                         circleTrace, semiTrace1, semiTrace2, false, false);
                         return;
                     } else if( choice == 2){  // case2: loop2 is discarded
                         flag2 = (bRNA.beforeAlongCircle(newstart2-1) != -1);

                         traceMotifMap2D(aRNA, start1, end1, aMap, bRNA, start2, newstart2-1, bMap, trace,
                                         circleTrace, semiTrace1, semiTrace2, false, flag2);
                         return;
                     } else if( choice == 3){ // case3: loop1 is opened
                         end1 --;
                         flag1 = (aRNA.beforeAlongCircle(end1) != -1);
                     } else {
                          System.out.println("!!!!!!   ERROR !!!!!!!!");
                          System.out.println("single1 : " + single1 + "  single2:" + single2);
                          System.out.println("flag1 : " + flag1 + "  flag2:" + flag2);
                          System.out.println("semiTrace1[" + end1 + "][" + end2 + "]: " + semiTrace1[end1][end2]);
                          System.exit(0);
                     }
                 } else   if(flag1 == false && flag2 == false){  // both are entering loop
                     byte choice = circleTrace[end1][end2];
                     if( choice == 3){  // case3: pair2 is discarded
                         end2 --;
                         flag2 = (bRNA.beforeAlongCircle(end2) != -1);
                     } else if( choice == 2){ // case2: pair1 is discarded
                         end1 --;
                         flag1 = (aRNA.beforeAlongCircle(end1) != -1);
                     } else if( choice == 1){  // case1: match
                         int left1 = component1 / aRNA.seqLength() - 1;
                         int right1 = component1 % aRNA.seqLength();

                         int left2 = component2 / bRNA.seqLength() - 1;
                         int right2 = component2 % bRNA.seqLength();

                         aMap[left1] = left2 + 1;    // base number is started from "1"
                         aMap[right1] = right2 + 1;
                         bMap[left2] = left1 + 1;
                         bMap[right2] = right1 + 1;

                         end1 --;
                         end2 --;
                         flag1 = (aRNA.beforeAlongCircle(end1) != -1);
                         flag2 = (bRNA.beforeAlongCircle(end2) != -1);
                     } else {
                          System.out.println("!!!!!!   ERROR !!!!!!!!");
                          System.out.println("single1 : " + single1 + "  single2:" + single2);
                          System.out.println("flag1 : " + flag1 + "  flag2:" + flag2);
                          System.out.println("circleTrace[" + end1 + "][" + end2 + "]: " + circleTrace[end1][end2]);
                          System.out.println("choice: " + choice);
                          System.exit(0);
                     }
                 }
             }  // end of both pairs
         } while( start1 <= end1 && start2 <= end2);
     }


     //-------------------------------------------------------------------------------------------------
     // This accomplishes the work of aligning a profile to an RNA subject structure.
     // The first one is a multi-alignment-derived 'MOTIF' profile; the second is a text structure
     // which may be very large.  
     //-------------------------------------------------------------------------------------------------
     private static double profileDetector(RNA aRNA, int start1, int end1,RNA bRNA, int start2, int end2,
                             double[][] score, double[][] semiCircleScore1,double[][] semiCircleScore2,
                             double[][] circleScore, byte[][] trace, byte[][] circleTrace,
                             byte[][] semiTrace1, byte[][] semiTrace2, boolean closeLoop, ProfileMatcher matcher,
                             double penalty)
     {
         int current1 = start1;
         int prev1  = current1;

         if( start1 == end1 && (aRNA.beforeAlongCircle(start1)!=-1))
             prev1 = aRNA.beforeAlongCircle(start1);

         int component1 = aRNA.getComponent(current1);

         int prevBase1;    // this is the previous visisted BASE ( whether bonded or not).
         int leftBondEnd1; // 5' BASE of current visiting component.

         if( component1 < aRNA.seqLength()){     // current component is singlet
             prevBase1 = component1;
             leftBondEnd1 = component1;
         } else {       // current component is a Bond( BASE pair)
             prevBase1 = component1 % aRNA.seqLength();
             leftBondEnd1 = component1 / aRNA.seqLength() - 1;
         }


         // real computing begins here:
         do {
             if( component1 < aRNA.seqLength()){  // component1 is a singlet.
                 int current2 = start2;
                 int prev2 = current2;

                 if( start2 == end2 && bRNA.beforeAlongCircle(start2) != -1)
                     prev2 = bRNA.beforeAlongCircle(start2);

                 int component2 = bRNA.getComponent(current2);
                 int leftBondEnd2, prevBase2;
                 if( component2 < bRNA.seqLength()) {
                     prevBase2 = component2;
                     leftBondEnd2 = component2;
                 } else {
                     prevBase2 = component2 % bRNA.seqLength();
                     leftBondEnd2 = component2 / bRNA.seqLength() -1;
                 }
                 int initBase2 = leftBondEnd2;

                 do {
                     if( component2 < bRNA.seqLength()){ // case: singlet + singlet
                         double match = 0;
                         match = matcher.getScore(bRNA.getBaseBond(current2), current1);

                         double case1 = 0;
                         if( (prev1 == start1 && start1 == end1) || (prev2 == start2 && start2 == end2)
                             || (start1 == current1 && start1 != end1) || (start2==current2 && start2 != end2))
                         {
                             if( leftBondEnd1 == 0)
                                 case1 = match;
                             else {
                                 int first = aRNA.firstAlongCircle(current1);
                                 while( aRNA.getComponent(first) > aRNA.seqLength() )
                                     first = aRNA.firstAlongCircle(first - 1);

                                 for(int k=first; k<current1; k++)
                                     case1 += matcher.getScore("-", k);


                                 int firstBase2 = 0;
                                 int element = bRNA.firstAlongCircle(current2);
                                 int comp = bRNA.getComponent(element);
                                 if( comp < bRNA.seqLength())
                                     firstBase2 = comp;
                                 else
                                     firstBase2 = comp / bRNA.seqLength() - 1;

                                 case1 += (leftBondEnd2-firstBase2)*penalty + match;
                             }
                         } else
                             case1 = (score[prev1][prev2] + match);

                         double case2 = 0;
                         if( (prev2 == start2 && start2 == end2) || (start2==current2 && start2 != end2))
                         {
                             // motif structure before 'current1' is also cut off now
                             case2 = penalty;
                             int first = aRNA.firstAlongCircle(current1);
                             while( aRNA.getComponent(first) > aRNA.seqLength() )
                                 first = aRNA.firstAlongCircle(first - 1);
                             for(int k=first; k<=current1; k++)
                                 case2 += matcher.getScore("-", k);
                         } else {
                             case2 = score[current1][prev2] + penalty;
                         }

                         double case3 = 0;
                         if( (prev1 == start1 && start1 == end1) || (start1==current1 && start1 != end1))
                         {
                             if( leftBondEnd1 == 0){  // current1 is on the topmost loop
                                 int element = bRNA.firstAlongCircle(current2);
                                 int comp = bRNA.getComponent(element);
                                 int firstBase = 0;
                                 if( comp < bRNA.seqLength())
                                     firstBase = comp;
                                 else
                                     firstBase = comp / bRNA.seqLength() - 1;

                                 int currentBase2 = bRNA.getComponent(current2);

                                 //case3 = matcher.getScore("-", current1);
                                 case3 = ( (currentBase2 - firstBase + 1)*penalty + matcher.getScore("-", current1));
                             } else {
                                 int element = bRNA.firstAlongCircle(current2);
                                 int comp = bRNA.getComponent(element);
                                 int firstBase = 0;
                                 if( comp < bRNA.seqLength())
                                     firstBase = comp;
                                 else
                                     firstBase = comp / bRNA.seqLength() - 1;

                                 int currentBase2 = bRNA.getComponent(current2);
                                 case3 = ( (currentBase2 - firstBase + 1)*penalty + matcher.getScore("-", current1));
                             }
                         } else {
                             case3 = (score[prev1][current2] + matcher.getScore("-", current1));
                         }

                         score[current1][current2] = Math.max(case1, Math.max(case2,case3));

                         if( case1 >= case2 && case1 >= case3)  // case1: match
                             trace[current1][current2] = 1;
                         else if( case2 >= case3)  // case2
                             trace[current1][current2] = 2;
                         else   // case3
                             trace[current1][current2] = 3;

                         prevBase2 = component2;
                     } else {   // component2 is a pair
                         // case of single base + bonded base pair
                         if ( leftBondEnd2 == prevBase2 + 1 || (end2 == start2 && closeLoop == false))
                         // component2 is a pair which is skipping a nested loop
                         {
                             // some preliminary step
                             if( end2 == start2 && closeLoop == false)
                                 prev2 = bRNA.beforeAlongCircle(current2);

                             // Here we consider three cases:
                             // case1: discard the induced loop, including the base pair
                             // case2: discard the single BASE
                             // case3: keep both loop and BASE.
                             // The MOST IMPORTANT thing here is the order of computing them

                             // FIRST, we need to compute CASE3:
                             // Note: score[*][current2] must be used to store inner loop score if
                             // current2 is not a first component of its enclosing loop
                             // Thus, need to backup the status of score[*][current2] and
                             // restore it after the calculation.
                             double case3 = 0;
                             if( end2 == start2 && closeLoop == false)
                             {  // the computing has been done before while circle <--> circle
                                 int element = aRNA.firstAlongCircle(current1);
                                 int comp = aRNA.getComponent(element);
                                 int firstBase = comp;
                                 if( firstBase < aRNA.seqLength())
                                     firstBase = comp;
                                 else
                                     firstBase = comp / aRNA.seqLength() - 1;

                                 if( firstBase == 0)  // 'current1' is ON topmost loop
                                     case3 = semiCircleScore2[current2][current1];
                                 else{
                                     element = bRNA.firstAlongCircle(current2);
                                     comp = bRNA.getComponent(element);
                                     if( comp < bRNA.seqLength())
                                         firstBase = comp;
                                     else
                                         firstBase = comp / bRNA.seqLength() - 1;

                                     case3 = (semiCircleScore2[current2][current1] +
                                                            (leftBondEnd2 - firstBase) * penalty);
                                 }
                             } else {
                                 int first = start1;
                                 if( start1 == end1)
                                     first = aRNA.firstAlongCircle(current1);

                                 double[] backup = new double[current1-first+1];
                                 for(int i=0; i<backup.length; i++) {
                                     backup[i] = score[i+first][current2];
                                     score[i+first][current2] = semiCircleScore2[current2][i+first];
                                 }

                                 // Normally, we extend the computing
                                 // from: (start1, prev1) <--> circle(prev2+1, current2)
                                 // to: (start1, current1) <--> circle(prev2+1, current2);
                                 // otherwise, we have to do it from the scratch.
                                 // here the last parameter is ignored.
                                 case3 = profileDetector(aRNA, current1, current1,
                                                 bRNA, prev2+1, current2, score, semiCircleScore1,
                                                 semiCircleScore2, circleScore, trace, circleTrace,
                                                 semiTrace1, semiTrace2, closeLoop, matcher, penalty);

                                 semiCircleScore2[current2][current1] = case3;

                                 int element = aRNA.firstAlongCircle(current1);
                                 int comp = aRNA.getComponent(element);
                                 int firstBase = comp;
                                 if( firstBase < aRNA.seqLength())
                                     firstBase = comp;
                                 else
                                     firstBase = comp / aRNA.seqLength() - 1;

                                 if( firstBase != 0) {
                                     element = bRNA.firstAlongCircle(current2);
                                     comp = bRNA.getComponent(element);
                                     if( comp < bRNA.seqLength())
                                         firstBase = comp;
                                     else
                                         firstBase = comp / bRNA.seqLength() - 1;

                                     case3 = (case3 + (leftBondEnd2 - firstBase) * penalty);
                                 }

                                 for(int i=0; i<backup.length; i++)
                                     score[i+first][current2] = backup[i];
                             }

                             // Then, we are ready to compute CASE-1 and CASE-2
                             int currentBase2 = bRNA.getPairedBase(leftBondEnd2);
                             double case1 = (score[current1][prev2] +
                                                 (currentBase2 - leftBondEnd2 + 1)*penalty);

                             double case2 = 0;
                             int currentBase1 = aRNA.getComponent(current1);
                             if( (prev1 == start1 && start1 == end1) || (start1 == current1 && start1 != end1))
                             { // current1 is the first single base along loop circumference
                                 if( leftBondEnd1 == 0)  // current1 is on the topmost loop
                                     case2 = matcher.getScore("-", current1);
                                 else {
                                     int element = bRNA.firstAlongCircle(current2);
                                     int comp = bRNA.getComponent(element);
                                     int firstBase = 0;
                                     if( comp < bRNA.seqLength())
                                         firstBase = comp;
                                     else
                                         firstBase = comp / bRNA.seqLength() - 1;

                                     case2 = (currentBase2 - firstBase + 1)*penalty + matcher.getScore("-", current1);
                                 }
                             } else {
                                 case2 = score[prev1][current2] + matcher.getScore("-", current1);
                             }

                             double result = Math.max(case1, Math.max(case2, case3));
                             score[current1][current2] = result;

                             if( case3 >= case1 && case3 >= case2)  // case3
                                 trace[current1][current2] = 3;
                             else if( case2 >= case1)  // case2
                                 trace[current1][current2] = 2;
                             else  // case1
                                 trace[current1][current2] = 1;
                         } else {   // component2 is closing a loop
                             // Here two cases are considered:
                             // case1: discard enclosing base pair
                             // case2: discard the single BASE;
                             double case2 = 0;
                             int currentBase2 = bRNA.getPairedBase(leftBondEnd2);
                             if( end2 == start2 && closeLoop == true) {
                                 prev2 = current2 - 1;

                                 if( current1 == start1){   // 'current1' is the first on loop
                                     if( leftBondEnd1 != 0) // 'current1' is not on topmost loop
                                         case2 = (currentBase2 - leftBondEnd2 + 1) * penalty + matcher.getScore("-", current1);
                                     else
                                         case2 = matcher.getScore("-", current1);
                                 } else {
                                     case2 = semiCircleScore2[current2][prev1] + matcher.getScore("-", current1);
                                 }
                             } else {   // Normal case OR derivation of case3 of single + skipping above
                                 if( (prev1 == start1 && start1 == end1) || (current1==start1 && start1 != end1))
                                 { // 'current1' is the first element on loop
                                     if( leftBondEnd1 != 0)  // 'current1' is not on topmost loop
                                         case2 = (currentBase2- leftBondEnd2 + 1 ) * penalty + matcher.getScore("-", current1);
                                     else
                                         case2 = matcher.getScore("-", current1);
                                 } else {
                                     case2 = semiCircleScore2[current2][prev1] + matcher.getScore("-", current1);
                                 }
                             }

                             double case1 = score[current1][prev2] + 2*penalty;

                             double result = Math.max(case1, case2);
                             score[current1][current2] = result;
                             semiCircleScore2[current2][current1] = result;

                             if( case2 >= case1)  // case2
                                 semiTrace2[current2][current1] = 12;
                             else  // case1
                                 semiTrace2[current2][current1] = 11;
                         }
                         prevBase2 = component2 % bRNA.seqLength();
                     }

                     prev2 = current2;
                     current2 = bRNA.nextAlongCircle(current2);
                     if(current2 != -1) {
                         component2 = bRNA.getComponent(current2);
                         if( component2 < bRNA.seqLength())
                             leftBondEnd2 = component2;
                         else
                             leftBondEnd2 = component2 / bRNA.seqLength() -1;
                     }
                 } while( current2 != -1 && current2 <= end2) ;
                 prevBase1 = component1;
             } else {    // component1 is a base pair
                 int current2 = start2;
                 int prev2 = current2;

                 if( end2==start2 && bRNA.beforeAlongCircle(start2) != -1)
                     prev2 = bRNA.beforeAlongCircle(start2);

                 int component2 = bRNA.getComponent(current2);
                 int leftBondEnd2, prevBase2;
                 if( component2 < bRNA.seqLength()) {
                      prevBase2 = component2;
                      leftBondEnd2 = component2;
                 } else {
                      prevBase2 = component2 % bRNA.seqLength();
                      leftBondEnd2 = component2 / bRNA.seqLength() -1;
                 }
                 int initBase2 = leftBondEnd2;

                 do {
                     // case of bonded base pair(component1) + single base(component2)
                     if ( component2 < bRNA.seqLength()) {
                         if(leftBondEnd1 == prevBase1 + 1 || (end1 == start1 && closeLoop == false))
                         //component1 is base pair which is skipping the enclosed LOOP
                         {
                             if( end1 == start1 && closeLoop == false)
                                 prev1 = aRNA.beforeAlongCircle(current1);

                             // three cases are considered:
                             // case1: discard the whole enclosed LOOP,inclusing the pair
                             // case2: discard the single BASE
                             // case3: keep both LOOP and the single BASE
                             // The MOST IMPORTANT thing here is the order of computing them

                             // FIRST STEP: computing CASE-3
                             // score[current1][*] must be used to store inner loop score if
                             // current1 is not a first component of its loop
                             // Thus, need to backup the status of score[current1][*] and
                             // restore it after the calculation.
                             double case3 = 0;

                             if( end1 == start1 && closeLoop == false) {
                                 int element = aRNA.firstAlongCircle(current1);
                                 while( aRNA.getComponent(element) > aRNA.seqLength())
                                     element = aRNA.firstAlongCircle(element-1);
                                 for(int k=element; k<=prev1; k++)
                                     case3 += matcher.getScore("-", k);

                                 case3 += semiCircleScore1[current1][current2];
                             } else {
                                 int first = start2;
                                 if( start2 == end2)
                                    first = bRNA.firstAlongCircle(first);

                                 double[] backup = new double[current2 - first +1];
                                 for(int i=0; i< backup.length; i++) {
                                     backup[i] = score[current1][i+first];
                                     score[current1][i+first] = semiCircleScore1[current1][i+first];
                                 }

                                 // Normally, we only need to extend the computing
                                 // from: circle(prev1+1) <--> (start2, prev2)
                                 // to:   circle(prev1+1) <--> (start2, current2)
                                 // Otherwise, we have to compute from the scratch
                                 case3 = profileDetector(aRNA,prev1+1, current1,
                                            bRNA, current2, current2, score, semiCircleScore1,
                                            semiCircleScore2, circleScore, trace, circleTrace,
                                            semiTrace1, semiTrace2, closeLoop, matcher, penalty);

                                 semiCircleScore1[current1][current2] = case3;

                                 int element = aRNA.firstAlongCircle(current1);
                                 while( aRNA.getComponent(element) > aRNA.seqLength())
                                     element = aRNA.firstAlongCircle(element-1);
                                 for(int k=element; k<=prev1; k++)
                                     case3 += matcher.getScore("-", k);

                                 for(int i=0; i< backup.length; i++)
                                     score[current1][i+first] = backup[i];
                             }


                             // THEN, we are ready to compute CASE-1 and CASE-2
                             double case1 = score[prev1][current2];
                             for(int k=prev1+1; k<=current1; k++)
                                 case1 += matcher.getScore("-", k);

                             double case2 = 0;
                             if( (prev2 == start2 && start2 == end2) || (current2 == start2 && start2 != end2))
                             {   // current2 is the FIRST single base along loop circumference
                                 int element = aRNA.firstAlongCircle(current1);
                                 while( aRNA.getComponent(element) > aRNA.seqLength())
                                     element = aRNA.firstAlongCircle(element-1);
                                 for(int k=element; k<=current1; k++)
                                     case2 += matcher.getScore("-", k);
                             } else {
                                 case2 = score[current1][prev2] + penalty;
                             }


                             double result = Math.max(case1, Math.max(case2, case3));

                             score[current1][current2] = result;

                             if( case3 >= case1 && case3 >= case2) // case3
                                 trace[current1][current2] = 3;
                             else if( case2 >= case1)  // case2
                                 trace[current1][current2] = 2;
                             else   // case1
                                 trace[current1][current2] = 1;

                         } else {    //componnet1 is a base pair which is closing the LOOP
                             // two cases are considered:
                             // case1: discard the base pair
                             // case2: discard the single BASE
                             double case2 = 0;
                             int first = aRNA.firstAlongCircle(current1-1);
                             while( aRNA.getComponent(first) > aRNA.seqLength())
                                 first = aRNA.firstAlongCircle(first-1);

                             if(end1 == start1 && closeLoop == true) {
                                 case2 = 0;
                                 if( current2 == start2) {
                                     for(int k = first; k <= current1; k++)
                                         case2 += matcher.getScore("-", k);
                                 } else
                                     case2 = semiCircleScore1[current1][prev2] + penalty;
                             } else {   // Normal case OR derivation of case3 of skipping-loop + singlet above
                                 case2 = 0;
                                 if( (start2 == prev2 && start2 == end2) || (current2==start2 && start2!=end2))
                                 {
                                     for(int k = first; k <= current1; k++)
                                         case2 += matcher.getScore("-", k);
                                 } else
                                     case2 = semiCircleScore1[current1][prev2] + penalty;
                             }

                             double case1 = 0;
                             case1 = score[current1-1][current2] + matcher.getScore("-", current1);

                             double result = Math.max(case1, case2);

                             score[current1][current2] = result;
                             semiCircleScore1[current1][current2] = result;

                             if( case2 >= case1)  // case2
                                 semiTrace1[current1][current2] = 12;
                             else  // case1
                                 semiTrace1[current1][current2] = 11;
                         }

                         prevBase2 = component2;
                     } else { // now, componet2 is also a base pair !!
                         // CASE of bonded base pair(component1->LOOP1) + bonded base pair(component2->LOOP2)
                         if( (leftBondEnd1 == prevBase1 + 1 || (start1==end1 && closeLoop == false)) &&
                             (leftBondEnd2 == prevBase2 + 1 || (start2==end2 && closeLoop == false)) )
                         {   // both are base pairs which are skipping the enclosed LOOPs
                              // five cases are considered:
                              // case1: both LOOPs are kept and matched with each other (2)
                              // case2: discard LOOP1; (0)
                              // case3: discard LOOP2; (1)
                              // case4: both LOOPs are kept where LOOP1 will match to the whole RNA2 (4)
                              // case5: both LOOPs are kets where LOOP2 will match to the whole RNA1 (3)

                             //!!!!!!!!! Here the order of computing is IMPORTANT !!!!!!!!!!
                             //FIRST STEP: We compute CASE-1
                             double temp1 = 0;
                             if( (start1==end1 && closeLoop==false) || (start2==end2 && closeLoop==false))
                             {
                                 temp1 = circleScore[current1][current2];

                                 if( start1 == end1 && closeLoop == false)
                                     prev1 = aRNA.beforeAlongCircle(current1);
                                 if( start2 == end2 && closeLoop == false)
                                     prev2 = bRNA.beforeAlongCircle(current2);
                             } else {
                                 double[] backup1 = null;
                                 double[] backup2 = null;

                                 //some bookkeeping work has to be done
                                 backup1 = new double[current2 - prev2];
                                 for(int i=0; i<backup1.length; i++) {
                                    backup1[i] = score[current1][i+(prev2+1)];
                                    score[current1][i+(prev2+1)] = semiCircleScore1[current1][i+(prev2+1)];
                                 }
                                 backup2 = new double[current1 - prev1];
                                 for(int i=0; i<backup2.length; i++) {
                                    backup2[i] = score[i+(prev1+1)][current2];
                                    score[i+(prev1+1)][current2] = semiCircleScore2[current2][i+(prev1+1)];
                                 }

                                 // the last parameter is ignored !!
                                 temp1 = profileDetector(aRNA, prev1+1, current1,
                                                 bRNA,prev2+1, current2, score, semiCircleScore1,
                                                 semiCircleScore2, circleScore, trace, circleTrace,
                                                 semiTrace1, semiTrace2, closeLoop, matcher, penalty);

                                 circleScore[current1][current2] = temp1;

                                 //restore the initial status.
                                 for(int i=0; i<backup1.length; i++)
                                     score[current1][i+(prev2+1)] = backup1[i];

                                 for(int i=0; i<backup2.length; i++)
                                     score[i+(prev1+1)][current2] = backup2[i];
                             }

                             double case1 = (score[prev1][prev2] + temp1);
                             // CASE-1 is finished here

                             //SECOND STEP: we compute CASE-4
                             double case4 = 0;
                             if( start1==end1 && closeLoop==false) {
                                 int element = aRNA.firstAlongCircle(current1);
                                 while( aRNA.getComponent(element) > aRNA.seqLength())
                                     element = aRNA.firstAlongCircle(element-1);
                                 for(int k=element; k<=prev1; k++)
                                     case4 += matcher.getScore("-", k);

                                 case4 += semiCircleScore1[current1][current2];
                             } else {
                                 int first = start2;
                                 if( start2 == end2)
                                     first = bRNA.firstAlongCircle(start2);

                                 double[] backup = new double[prev2 - first + 1];

                                 for(int i=0; i<backup.length; i++) {
                                     backup[i] = score[current1][i+first];
                                     score[current1][i+first] = semiCircleScore1[current1][i+first];
                                 }

                                 // We only need to extend the computing
                                 // from:  circle(prev1+1, current1) <--> (start2, prev2)
                                 // to:    circle(prev1+1, current1) <--> (start2, current2)
                                 // Furthermore, the algorithm has to be notified that the base
                                 // pair, 'current2', is skipping a loop instead of closing loop,
                                 // to accomplish this, the last parameter is set to be 'false'
                                 case4 = profileDetector(aRNA, prev1+1, current1,
                                                 bRNA, current2, current2, score, semiCircleScore1,
                                                 semiCircleScore2, circleScore, trace, circleTrace,
                                                 semiTrace1, semiTrace2, false, matcher, penalty);

                                 semiCircleScore1[current1][current2] = case4;

                                 int element = aRNA.firstAlongCircle(current1);
                                 while( aRNA.getComponent(element) > aRNA.seqLength())
                                     element = aRNA.firstAlongCircle(element-1);
                                 for(int k=element; k<=prev1;  k++)
                                     case4 += matcher.getScore("-", k);


                                 //restore the initial status.
                                 for(int i=0; i<backup.length; i++)
                                     score[current1][i+first] = backup[i];
                             }
                             // CASE-4 is finished


                             // THIRD STEP: we compute CASE-5
                             double case5 = 0;
                             if( (start2==end2 && closeLoop==false)) {
                                 int element = aRNA.firstAlongCircle(current1);
                                 int comp = aRNA.getComponent(element);
                                 int firstBase = 0;
                                 if( comp < aRNA.seqLength())
                                     firstBase = comp;
                                 else
                                     firstBase = comp / aRNA.seqLength() -  1;

                                 if( firstBase == 0)  // current1 is on topmost loop
                                     case5 = semiCircleScore2[current2][current1];
                                 else {
                                     element = bRNA.firstAlongCircle(current2);
                                     comp = bRNA.getComponent(element);
                                     firstBase = 0;
                                     if( comp < bRNA.seqLength())
                                         firstBase = comp;
                                     else
                                         firstBase = comp / bRNA.seqLength() - 1;

                                     case5 = (semiCircleScore2[current2][current1]
                                                         + (leftBondEnd2 - firstBase)*penalty);
                                 }
                             } else {
                                 //some bookkeeping work has to be done firstly
                                 int first = start1;
                                 if( start1 == end1)
                                     first = aRNA.firstAlongCircle(start1);
                                 double[] backup = new double[prev1-first+1];

                                 for(int i=0; i<backup.length; i++) {
                                     backup[i] = score[i+first][current2];
                                     score[i+first][current2] = semiCircleScore2[current2][i+first];
                                 }

                                 // Based on what has obtained from case1, we only need to extend the computing
                                 // from:   (start1, prev1) <--> circle(prev2+1, current2)
                                 // to:     (start1, current1) <--> circle(prev2+1, current2)
                                 // Furthermore, the algorithm has to be notified that the base
                                 // pair, 'current1', is skipping a loop instead of closing loop
                                 // to accomplish this, last parameter is set to be 'false'

                                 case5 = profileDetector(aRNA, current1, current1,
                                                 bRNA, prev2+1, current2, score, semiCircleScore1,
                                                 semiCircleScore2, circleScore, trace, circleTrace,
                                                 semiTrace1, semiTrace2, false, matcher, penalty);

                                 semiCircleScore2[current2][current1] = case5;

                                 int element = aRNA.firstAlongCircle(current1);
                                 int comp = aRNA.getComponent(element);
                                 int firstBase = 0;
                                 if( comp < aRNA.seqLength())
                                     firstBase = comp;
                                 else
                                     firstBase = comp / aRNA.seqLength() -  1;

                                 if( firstBase != 0){  // current1 is NOT on topmost loop
                                     element = bRNA.firstAlongCircle(current2);
                                     comp = bRNA.getComponent(element);
                                     firstBase = 0;
                                     if( comp < bRNA.seqLength())
                                         firstBase = comp;
                                     else
                                         firstBase = comp / bRNA.seqLength() - 1;

                                     case5 += (leftBondEnd2 - firstBase)*penalty;
                                 }

                                 //restore the initial status.
                                 for(int i=0; i<backup.length; i++)
                                     score[i+first][current2] = backup[i];
                             }
                             // CASE-5 is finished !

                             // FOUTH and FIFTH STEPS are to compute CASE-2 and CASE-3
                             double case2 = 0;
                             for(int k=prev1+1; k<=current1; k++)
                                 case2 += matcher.getScore("-", k);

                             case2 += score[prev1][current2];

                             int currentBase2 = bRNA.getPairedBase(leftBondEnd2);
                             double case3 = (score[current1][prev2] +
                                             (currentBase2 - leftBondEnd2 + 1)*penalty);

                             double result = Math.max(case1, Math.max(case2, Math.max(case3,
                                                 Math.max(case4, case5))));
                             score[current1][current2] = result;

                             if( case1>=case2 && case1>=case3 && case1>=case4 && case1>=case5)
                                 trace[current1][current2] = 1;
                             else if( case4>=case2 && case4>=case3 && case4>=case5)
                                 trace[current1][current2] = 4;
                             else if( case5>=case2 && case5>=case3)
                                 trace[current1][current2] = 5;
                             else if( case2>=case3)
                                 trace[current1][current2] = 2;
                             else
                                 trace[current1][current2] = 3;
                         } else if( leftBondEnd1 == prevBase1+1 || (end1 == start1 && closeLoop==false) )
                         // component1 is skipping the enclosed LOOP1 while component2 is closing its LOOP2
                         {
                             // three cases are considered:
                             // case1: both LOOPs are kept and matched with each other (0)
                             // case2: LOOP1 is discarded.(1)
                             // case3: LOOP2 is opened by discarding the enclosing pair (2)

                             if( start2==end2 && closeLoop == true)
                                 prev2 = end2 - 1;

                             double case1 = 0;
                             if( end1==start1 && closeLoop == false) {
                                 int element = aRNA.firstAlongCircle(current1);
                                 while( aRNA.getComponent(element) > aRNA.seqLength())
                                     element = aRNA.firstAlongCircle(element-1);
                                 for(int k=element; k<=prev1; k++)
                                     case1 += matcher.getScore("-", k);

                                 case1 += circleScore[current1][current2];
                             } else {
                                 int first = start2;
                                 if( start2 == end2)
                                     first = bRNA.firstAlongCircle(start2-1);

                                 double[] backup = new double[current2 - first + 1];
                                 for(int i=0; i<backup.length;i++) {
                                     backup[i] = score[current1][i+first];
                                     score[current1][i+first] = semiCircleScore1[current1][i+first];
                                 }

                                 // Normally, we only need to extend computing
                                 // from: circle(prev1+1, current1) <--> (start2, prev2)
                                 // to: circle(prev1+1, current1) <--> circle(start2, current2);
                                 //
                                 // Furthermore, the algorithm has to be notified that pair(current2)
                                 // is closing loop instead of skipping loop
                                 // to acomplish this, the last parameter is set to be 'true'
                                 case1 = profileDetector(aRNA, prev1+1, current1,
                                                 bRNA, current2, current2, score, semiCircleScore1,
                                                 semiCircleScore2, circleScore, trace, circleTrace,
                                                 semiTrace1, semiTrace2, true, matcher, penalty);

                                 circleScore[current1][current2] = case1;

                                 int element = aRNA.firstAlongCircle(current1);
                                 while( aRNA.getComponent(element) > aRNA.seqLength())
                                     element = aRNA.firstAlongCircle(element-1);
                                 for(int k=element; k<=prev1; k++)
                                     case1 += matcher.getScore("-", k);

                                 for(int i=0; i<backup.length; i++)
                                     score[current1][i+first] = backup[i];
                             }

                             int currentBase1 = aRNA.getPairedBase(leftBondEnd1);

                             double case2 = 0;
                             int first = aRNA.beforeAlongCircle(current1) + 1;
                             for(int k=first; k<=current1; k++)
                                 case2 += matcher.getScore("-", k);

                             case2 += score[prev1][current2];

                             double case3 = (score[current1][prev2] + 2*penalty);

                             double result = Math.max(case1, Math.max(case2, case3));

                             score[current1][current2] = result;
                             semiCircleScore2[current2][current1] = result;

                             if( case1 >= case2 && case1 >= case3)
                                 semiTrace2[current2][current1] = 1;
                             else if( case3 >= case2)
                                 semiTrace2[current2][current1] = 3;
                             else    //case2
                                 semiTrace2[current2][current1] = 2;

                         } else if( leftBondEnd2 == prevBase2+1 || (end2 == start2 && closeLoop == false))
                         // component1 is closing its LOOP1 while component2 is skipping the enclosed LOOP2
                         {
                             // two cases are considered:
                             // case1: both LOOPs are kept and matched with each other(0)
                             // case2: LOOP2 is discarded.(1)
                             // case3: LOOP1 is opened by discarding its enclosing pair(2)

                             if( start1 == end1 && closeLoop == true)
                                 prev1 = end1 - 1;

                             // get the first component of Loop2
                             if( end2 == start2 && closeLoop == false)
                                 prev2 = bRNA.beforeAlongCircle(current2);

                             double case1 = 0;
                             if( end2 == start2 && closeLoop == false) {
                                 int element = aRNA.firstAlongCircle(current1);
                                 int comp = aRNA.getComponent(element);
                                 int firstBase = 0;
                                 if( comp < aRNA.seqLength())
                                     firstBase = comp;
                                 else
                                     firstBase = comp / aRNA.seqLength() - 1;

                                 if( firstBase == 0)  // current1 is on topmost loop
                                    case1 = circleScore[current1][current2];
                                 else {
                                    element = bRNA.firstAlongCircle(current2);
                                    comp = bRNA.getComponent(element);
                                    firstBase = 0;
                                    if( comp < bRNA.seqLength())
                                        firstBase = comp;
                                    else
                                        firstBase = comp / bRNA.seqLength() - 1;

                                     case1 = (circleScore[current1][current2]
                                                         + (leftBondEnd2 - firstBase)*penalty);
                                 }
                             } else {
                                 int first = start1;
                                 if( start1 == end1)
                                     first = aRNA.firstAlongCircle(start1-1);

                                 double[] backup = new double[current1 - first + 1];
                                 for(int i=0; i<backup.length; i++) {
                                     backup[i] = score[i+first][current2];
                                     score[i+first][current2] = semiCircleScore2[current2][i+first];
                                 }

                                 // Normally, we only need to extend the computing
                                 // from: (start1, prev1) <--> circle(prev2+1, current2)
                                 // to:   circle(start1, current1) <--> circle(prev2+1, current2)
                                 //
                                 // Futhermore, the algorithm has to be notified that pair(current1)
                                 // is enclosing a loop instead of skipping a loop
                                 // to accomplish this, last parameter is set to be 'true'

                                 case1 = profileDetector(aRNA, current1, current1, bRNA, prev2+1,
                                                 current2, score, semiCircleScore1, semiCircleScore2,
                                                 circleScore, trace, circleTrace,
                                                 semiTrace1, semiTrace2, true, matcher, penalty);

                                 circleScore[current1][current2] = case1;

                                 // test if 'current1' stay in topmost loop
                                 int element = aRNA.firstAlongCircle(current1);
                                 int comp = aRNA.getComponent(element);
                                 int firstBase = 0;
                                 if( comp < aRNA.seqLength())
                                     firstBase = comp;
                                 else
                                     firstBase = comp / aRNA.seqLength() - 1;

                                 if( firstBase != 0){   // 'current1' is NOT on topmost loop
                                     element = bRNA.firstAlongCircle(current2);
                                     comp = bRNA.getComponent(element);
                                     firstBase = 0;
                                     if( comp < bRNA.seqLength())
                                         firstBase = comp;
                                     else
                                         firstBase = comp / bRNA.seqLength() - 1;

                                     case1 = (case1 + (leftBondEnd2-firstBase)*penalty);
                                 }

                                 for(int i=0; i<backup.length; i++)
                                     score[i+first][current2] = backup[i];
                             }

                             int currentBase2 = bRNA.getPairedBase(leftBondEnd2);
                             double case2 = (score[current1][prev2] +
                                            (currentBase2 - leftBondEnd2 + 1)*penalty);

                             double case3 = score[prev1][current2] + matcher.getScore("-", current1);

                             double result = Math.max(case1, Math.max(case2, case3));

                             score[current1][current2] = result;
                             semiCircleScore1[current1][current2] = result;

                             if( case1 >= case2 && case1 >= case3)
                                 semiTrace1[current1][current2] = 1;
                             else if( case3 >= case2)
                                 semiTrace1[current1][current2] = 3;
                             else  //case2
                                 semiTrace1[current1][current2] = 2;
                         } else {  // both components are closing the enclosed LOOPs.
                             // three cases are considered:
                             // case1: both components are kept and matched with each other
                             // case2: component1 is discarded
                             // case3: component2 is discarded
                             double match = matcher.getScore(bRNA.getBaseBond(current2), current1);

                             if( end1 == start1 && closeLoop == true)
                                 prev1 = current1 - 1;
                             if( end2 == start2 && closeLoop == true)
                                 prev2 = current2 - 1;

                             double case1 = score[prev1][prev2] + match;
                             double case2 = score[prev1][current2] + matcher.getScore("-", current1);
                             double case3 = score[current1][prev2] + 2*penalty;

                             double result = Math.max(case1, Math.max(case2,case3));
                             score[current1][current2] = result;
                             circleScore[current1][current2] = result;

                             if( aRNA.beforeAlongCircle(current1) == -1)  // first component of loop
                                 semiCircleScore2[current2][current1] = result;
                             if( bRNA.beforeAlongCircle(current2) == -1) // first component of loop
                                 semiCircleScore1[current1][current2] = result;

                             if( case1 >= case2 && case1 >= case3)
                                 circleTrace[current1][current2] = 1;
                             else if( case2 >= case3)
                                 circleTrace[current1][current2] = 2;
                             else
                                 circleTrace[current1][current2] = 3;
                         }

                         prevBase2 = component2 % bRNA.seqLength();
                     }

                     prev2 = current2;
                     current2 = bRNA.nextAlongCircle(current2);
                     if(current2 != -1) {
                         component2 = bRNA.getComponent(current2);
                         if( component2 < bRNA.seqLength())
                             leftBondEnd2 = component2;
                         else
                             leftBondEnd2 = component2 / bRNA.seqLength() -1;
                     }
                 } while ( current2 != -1 && current2 <= end2);

                 prevBase1 = component1 % aRNA.seqLength();
             }

             prev1 = current1;
             current1 = aRNA.nextAlongCircle(current1);
             if(current1 != -1) {
                 component1 = aRNA.getComponent(current1);
                 if( component1 < aRNA.seqLength())
                     leftBondEnd1 = component1;
                 else
                     leftBondEnd1 = component1 / aRNA.seqLength() - 1;
             }
         } while ( current1 != -1 && current1 <= end1);

         return score[end1][end2];
     }

     //-----------------------------------------------------------------------------------------------
     // This is the latest implementation of LOCAL structure alignment algorithm: see the publications
     //-----------------------------------------------------------------------------------------------
     private static double localAligner(RNA aRNA, int start1, int end1,RNA bRNA, int start2, int end2,
                             double[][] gScore, double[][] semiCircleScore1, double[][] semiCircleScore2,
                             double[][] lScore, double[][] circleScore, byte[][] gTrace, byte[][] lTrace,
                             byte[][] circleTrace,byte[][] lSemiTrace1, byte[][] lSemiTrace2,
                             byte[][] gSemiTrace1, byte[][] gSemiTrace2, boolean closeLoop,
                             Matcher matcher, double penalty)
     {
         int current1 = start1;
         int prev1  = current1;
//##################################################################################
	// int mcomponent;
	// int m1=-1,m2=-1;
//##################################################################################


	//#########################################################################
	int m1=0,m2=0;
	int mcomponent;
	double total_length= aRNA.seqLength();
	double conserved_length = 0;
	String send_conserved = null;
	if((aRNA.conserved != null)&&(aRNA.flag!=1)){
		conserved_length=aRNA.conserved.length();
	String c = aRNA.conserved;
	int count=0;
	for(int k = 0;k<conserved_length;k++){
		if(c.substring(k,k+1).equals("*")){
			count++;
		}
	}	
	conserved_length = count;
	send_conserved  = aRNA.conserved;
	}
	else if((bRNA.conserved != null) && (aRNA.flag != 1)){

                conserved_length=bRNA.conserved.length();

                String c = bRNA.conserved;
                int count=0;
                for(int k = 0;k<conserved_length;k++){
                        if(c.substring(k,k+1).equals("*")){
                                count++;
                        }
                }
                conserved_length = count;
                send_conserved = bRNA.conserved;
        }


	double factor = conserved_length/total_length;

	//#########################################################################

         if( start1 == end1 && (aRNA.beforeAlongCircle(start1)!=-1))
             prev1 = aRNA.beforeAlongCircle(start1);

         int component1 = aRNA.getComponent(current1);

         int prevBase1;    // this is the previous visisted BASE ( whether bonded or not).
         int leftBondEnd1; // 5' BASE of current visiting component.

         if( component1 < aRNA.seqLength()){     // current component is singlet
             prevBase1 = component1;
             leftBondEnd1 = component1;
         } else{  // current component is a Bond( BASE pair)
             prevBase1 = component1 % aRNA.seqLength();
             leftBondEnd1 = component1 / aRNA.seqLength() - 1;
         }

         // real computing begins here:
         do {
             if( component1 < aRNA.seqLength()){  // component1 is a singlet.
                 int current2 = start2;
                 int prev2 = current2;

                 if( start2 == end2 && bRNA.beforeAlongCircle(start2) != -1)
                      prev2 = bRNA.beforeAlongCircle(start2);

                 int component2 = bRNA.getComponent(current2);
                 int leftBondEnd2, prevBase2;
                 if( component2 < bRNA.seqLength()) {
                     prevBase2 = component2;
                     leftBondEnd2 = component2;
                 } else {
                     prevBase2 = component2 % bRNA.seqLength();
                     leftBondEnd2 = component2 / bRNA.seqLength() -1;
                 }
                 int initBase2 = leftBondEnd2;

                 do {
                     if( component2 < bRNA.seqLength()){  // case: singlet + singlet 
                         double match = 0;
                         if( matcher == null){
                             match = (double)RNA.baseMatch(aRNA.getBaseBond(current1), 
                                                           bRNA.getBaseBond(current2));
			System.out.println("match is null");
                         }else{ ///##################change###########################
                          
   	     		     mcomponent = aRNA.getComponent(current1);
        		     if (mcomponent < aRNA.sequence.length()){
            			m1 = mcomponent;
				m2 = -1;
			
        		     }else {
            			m1 = mcomponent / aRNA.seqLength() - 1;
           		        m2 = mcomponent % aRNA.seqLength();
        		     }
			     match = matcher.baseMatch2(aRNA.flag,aRNA.getBaseBond(current1), bRNA.getBaseBond(current2),m1,m2,send_conserved,factor,aRNA.factors_for_conserved);
			 }
                         double localCase1 = 0;     // local case
                         double globalCase1 = 0;    // global case

                         // at least one of the components is the 'first' one of the resident loop
                         if( (prev1 == start1 && start1 == end1) || (prev2 == start2 && start2 == end2)
                             || (start1 == current1 && start1 != end1) || (start2==current2 && start2 != end2))
                         {
                             localCase1 = match;

                             int firstBase1=0;    // first base within current resident loop
                             int firstBase2=0;

                             int element = aRNA.firstAlongCircle(current1);
                             int comp = aRNA.getComponent(element);
                             if( comp < aRNA.seqLength())
                                 firstBase1 = comp;
                             else
                                 firstBase1 = comp / aRNA.seqLength() - 1;

                             element = bRNA.firstAlongCircle(current2);
                             comp = bRNA.getComponent(element);
                             if( comp < bRNA.seqLength())
                                 firstBase2 = comp;
                             else
                                 firstBase2 = comp / bRNA.seqLength() - 1;

                             globalCase1 = ((leftBondEnd1-firstBase1)*penalty
                                         + (leftBondEnd2-firstBase2)*penalty + match);
                         } else {
                             localCase1 = (lScore[prev1][prev2] + match);
                             globalCase1 = (gScore[prev1][prev2] + match);
                         }

                         // second case: 'current2' matched with gap:
                         double localCase2 = 0;
                         double globalCase2 = 0;
                         if( (prev2 == start2 && start2 == end2) || (start2==current2 && start2 != end2))
                         {
                            localCase2 = 0;

                            // small structure before 'current1' is also cut off now
                            int element = aRNA.firstAlongCircle(current1);
                            int comp = aRNA.getComponent(element);
                            int firstBase = 0;
                            if( comp < aRNA.seqLength())
                                firstBase = comp;
                            else
                                firstBase = comp / aRNA.seqLength() - 1;

                            int currentBase1 = aRNA.getComponent(current1);
                            globalCase2 = ( (currentBase1 - firstBase + 1) * penalty + penalty);
                         } else {
                            localCase2 = lScore[current1][prev2] + penalty;
                            globalCase2 = gScore[current1][prev2] + penalty;
                         }

                         // third case: 'current1' matched with gap:
                         double localCase3 = 0;
                         double globalCase3 = 0;
                         if( (prev1 == start1 && start1 == end1) || (start1==current1 && start1 != end1))
                         {
                            localCase3 = 0;

                            int element = bRNA.firstAlongCircle(current2);
                            int comp = bRNA.getComponent(element);
                            int firstBase = 0;
                            if( comp < bRNA.seqLength())
                                firstBase = comp;
                            else
                                firstBase = comp / bRNA.seqLength() - 1;

                            int currentBase2 = bRNA.getComponent(current2);
                            globalCase3 = (currentBase2 - firstBase + 1) * penalty + penalty;
                         } else {
                             localCase3 = lScore[prev1][current2] + penalty;
                             globalCase3 = gScore[prev1][current2] + penalty;
                         }

                         lScore[current1][current2] = Math.max(0, Math.max(localCase1,
                                                             Math.max(localCase2,localCase3)));
                         gScore[current1][current2] = Math.max(globalCase1,
                                                            Math.max(globalCase2, globalCase3));

                         // fill in the global traceback table
                         if( globalCase1 >= globalCase2 && globalCase1 >= globalCase3)  // case1: match
                             gTrace[current1][current2] = 1;
                         else if( globalCase2 >= globalCase3)  // case2
                             gTrace[current1][current2] = 2;
                         else   // case3
                              gTrace[current1][current2] = 3;

                         // fill in the local traceback table
                         if( localCase1 >= localCase2 && localCase1 >= localCase3)
                             lTrace[current1][current2] = 1;
                         else if( localCase2 >= localCase3)
                             lTrace[current1][current2] = 2;
                         else
                             lTrace[current1][current2] = 3;

                         prevBase2 = component2;
                     } else {    // component2 is a pair, component1 is a single
                         // case of single base + bonded base pair
                         if ( leftBondEnd2 == prevBase2 + 1 || (end2 == start2 && closeLoop == false))
                         // component2 is a pair which is skipping a nested loop
                         {
                             // some preliminary step
                             if( end2 == start2 && closeLoop == false)
                                 prev2 = bRNA.beforeAlongCircle(current2);

                             // Here we consider three cases:
                             // case1: discard the induced loop, including the base pair
                             // case2: discard the single BASE
                             // case3: keep both loop and BASE.
                             // The MOST IMPORTANT thing here is the order of computing them

                             // FIRST, we need to compute CASE3:
                             // Note: score[*][current2] must be used to store inner loop score if
                             // current2 is not a first component of its enclosing loop
                             // Thus, need to backup the status of score[*][current2] and
                             // restore it after the calculation.
                             double localCase3 = 0;
                             double globalCase3 = 0;
                             if( end2 == start2 && closeLoop == false)
                             {  // the computing has been done before while circle <--> circle
                                 localCase3 = semiCircleScore2[current2][current1];
                                 if( localCase3 < 0)
                                     localCase3 = 0;

                                 int element = bRNA.firstAlongCircle(current2);
                                 int comp = bRNA.getComponent(element);
                                 int firstBase = 0;
                                 if( comp < bRNA.seqLength())
                                     firstBase = comp;
                                 else
                                     firstBase = comp / bRNA.seqLength() - 1;

                                 globalCase3 = semiCircleScore2[current2][current1] +
                                                  (leftBondEnd2 - firstBase) * penalty;
                             } else {
                                int first = start1;
                                if( start1 == end1)
                                    first = aRNA.firstAlongCircle(current1);

                                 double[] localBackup = new double[current1-first+1];
                                 double[] globalBackup = new double[current1-first+1];

                                 for(int i=0; i<localBackup.length; i++) {
                                     localBackup[i] = lScore[i+first][current2];
                                     lScore[i+first][current2] = semiCircleScore2[current2][i+first];

                                     globalBackup[i] = gScore[i+first][current2];
                                     gScore[i+first][current2] = semiCircleScore2[current2][i+first];
                                 }

                                 // Normally, we extend the computing
                                 // from: (start1, prev1) <--> circle(prev2+1, current2)
                                 // to: (start1, current1) <--> circle(prev2+1, current2);
                                 // otherwise, we have to do it from the scratch.
                                 // here the parameter 'closLoop' is ignored.
                                 localCase3 = localAligner(aRNA, current1, current1,
                                                 bRNA, prev2+1, current2, gScore, semiCircleScore1,
                                                 semiCircleScore2, lScore,
                                                 circleScore, gTrace, lTrace, circleTrace,
                                                 lSemiTrace1, lSemiTrace2, gSemiTrace1, gSemiTrace2, closeLoop,
                                                 matcher, penalty);

                                 int element = bRNA.firstAlongCircle(current2);
                                 int comp = bRNA.getComponent(element);
                                 int firstBase;
                                 if( comp < bRNA.seqLength())
                                     firstBase = comp;
                                 else
                                     firstBase = comp / bRNA.seqLength() - 1;

                                 globalCase3 = semiCircleScore2[current2][current1] +
                                                                   (leftBondEnd2-firstBase)*penalty;

                                 for(int i=0; i<localBackup.length; i++) {
                                     lScore[i+first][current2] = localBackup[i];
                                     gScore[i+first][current2] = globalBackup[i];
                                 }
                             } // case3 is finished

                             // Then, we are ready to compute CASE-1 and CASE-2
                             int currentBase2 = bRNA.getPairedBase(leftBondEnd2);
                             double localCase1 = lScore[current1][prev2] +
                                                 (currentBase2 - leftBondEnd2 + 1)*penalty;

                             double globalCase1 = gScore[current1][prev2] +
                                                 (currentBase2 - leftBondEnd2 + 1)*penalty;

                             double localCase2 = 0;
                             double globalCase2 = 0;
                             int currentBase1 = aRNA.getComponent(current1);
                             if( (prev1 == start1 && start1 == end1) || (start1 == current1 && start1 != end1))
                             { // current1 is the first single base along loop circumference
                                 localCase2 = 0;

                                 int element = bRNA.firstAlongCircle(current2);
                                 int comp = bRNA.getComponent(element);
                                 int firstBase = 0;
                                 if( comp < bRNA.seqLength())
                                     firstBase = comp;
                                 else
                                     firstBase = comp / bRNA.seqLength() - 1;

                                 globalCase2 = (currentBase2-firstBase+1)*penalty + penalty;
                             } else {
                                localCase2 = lScore[prev1][current2] + penalty;
                                globalCase2 = gScore[prev1][current2] + penalty;
                             }


                             double gResult = Math.max(globalCase1, Math.max(globalCase2, globalCase3));
                             double lResult = Math.max(0, Math.max(localCase1,
                                                          Math.max(localCase2, localCase3)));

                             gScore[current1][current2] = gResult;
                             lScore[current1][current2] = lResult;

                             // traceback for global alignment
                             if( globalCase3 >= globalCase1 && globalCase3 >= globalCase2)  // case3
                                 gTrace[current1][current2] = 3;
                             else if( globalCase2 >= globalCase1)  // case2
                                 gTrace[current1][current2] = 2;
                             else  // case1
                                 gTrace[current1][current2] = 1;

                             // traceback for local alignment
                             if( localCase3 >= localCase1 && localCase3 >= localCase2)
                                 lTrace[current1][current2] = 3;
                             else if( localCase2 >= localCase1)
                                 lTrace[current1][current2] = 2;
                             else
                                 lTrace[current1][current2] = 1;
                         } else {     // component2 is closing a loop, component1 is a single base
                             // Here two cases are considered:
                             // case1: discard enclosing base pair
                             // case2: discard the single BASE;
                             double case2 = 0;

                             int currentBase1 = aRNA.getComponent(current1);
                             int currentBase2 = bRNA.getPairedBase(leftBondEnd2);

                             if( end2 == start2 && closeLoop == true) {
                                 prev2 = current2 - 1;
                                 if( current1 == start1) // 'current1' is the first on loop
                                     case2 = (currentBase2-leftBondEnd2+1)*penalty + penalty;
                                 else
                                     case2 = semiCircleScore2[current2][prev1] + penalty;
                             } else{  // Normal case OR derivation of case3 of single + skipping above
                                 if( (prev1 == start1 && start1 == end1) || (current1==start1 && start1 != end1))
                                     // 'current1' is the first element on loop
                                     case2 = (currentBase2-leftBondEnd2+2)*penalty;
                                 else
                                    case2 = semiCircleScore2[current2][prev1] + penalty;
                             }

                             double case1 = gScore[current1][prev2] + 2*penalty;

                             gScore[current1][current2] = Math.max(case1, case2);
                             semiCircleScore2[current2][current1] = gScore[current1][current2];

                             lScore[current1][current2] = Math.max(0, gScore[current1][current2]);

                             // global alignment traceback information
                             if( case2 >= case1)  // case2
                                 gSemiTrace2[current2][current1] = 12;
                             else  // case1
                                 gSemiTrace2[current2][current1] = 11;

                             // local alignment traceback information:
                             if( case2 >= case1)
                                 lSemiTrace2[current2][current1] = 12;
                             else
                                 lSemiTrace2[current2][current1] = 11;
                         }
                         prevBase2 = component2 % bRNA.seqLength();
                     }

                     prev2 = current2;
                     current2 = bRNA.nextAlongCircle(current2);
                     if(current2 != -1) {
                         component2 = bRNA.getComponent(current2);
                         if( component2 < bRNA.seqLength())
                             leftBondEnd2 = component2;
                         else
                             leftBondEnd2 = component2 / bRNA.seqLength() -1;
                     }
                 } while( current2 != -1 && current2 <= end2) ;
                 prevBase1 = component1;
             } else {   // component1 is a base pair
                 int current2 = start2;
                 int prev2 = current2;

                 if( end2==start2 && bRNA.beforeAlongCircle(start2) != -1)
                     prev2 = bRNA.beforeAlongCircle(start2);

                 int component2 = bRNA.getComponent(current2);
                 int leftBondEnd2, prevBase2;
                 if( component2 < bRNA.seqLength()) {
                     prevBase2 = component2;
                     leftBondEnd2 = component2;
                 } else {
                     prevBase2 = component2 % bRNA.seqLength();
                     leftBondEnd2 = component2 / bRNA.seqLength() -1;
                 }
                 int initBase2 = leftBondEnd2;

                 do {
                     // case of bonded base pair(component1) + single base(component2)
                     if ( component2 < bRNA.seqLength()) {
                         // component1 is a base pair which is skipping the enclosed LOOP
                         // component2 is a single base
                         if(leftBondEnd1 == prevBase1 + 1 || (end1 == start1 && closeLoop == false))
                         {
                             if( end1 == start1 && closeLoop == false)
                                 prev1 = aRNA.beforeAlongCircle(current1);

                             // three cases are considered:
                             // case1: discard the whole enclosed LOOP,inclusing the pair
                             // case2: discard the single BASE
                             // case3: keep both LOOP and the single BASE
                             // The MOST IMPORTANT thing here is the order of computing them

                             // FIRST STEP: computing CASE-3
                             // score[current1][*] must be used to store inner loop score if
                             // current1 is not a first component of its loop
                             // Thus, need to backup the status of score[current1][*] and
                             // restore it after the calculation.
                             double localCase3 = 0;
                             double globalCase3= 0;

                             if( end1 == start1 && closeLoop == false) {
                                 localCase3 = semiCircleScore1[current1][current2];
                                 if( localCase3 < 0)
                                     localCase3 = 0;

                                 int element = aRNA.firstAlongCircle(current1);
                                 int comp = aRNA.getComponent(element);
                                 int firstBase = 0;
                                 if( comp < aRNA.seqLength())
                                     firstBase = comp;
                                 else
                                     firstBase = comp / aRNA.seqLength() - 1;

                                 globalCase3 = (semiCircleScore1[current1][current2]
                                                         + (leftBondEnd1 - firstBase)*penalty);
                             } else {
                                 int first = start2;
                                 if( start2 == end2)
                                     first = bRNA.firstAlongCircle(first);

                                 double[] localBackup = new double[current2 - first +1];
                                 double[] globalBackup = new double[current2-first+1];

                                 for(int i=0; i< localBackup.length; i++) {
                                     localBackup[i] = lScore[current1][i+first];
                                     lScore[current1][i+first] = semiCircleScore1[current1][i+first];

                                     globalBackup[i] = gScore[current1][i+first];
                                     gScore[current1][i+first] = semiCircleScore1[current1][i+first];
                                 }

                                 // Normally, we only need to extend the computing
                                 // from: circle(prev1+1) <--> (start2, prev2)
                                 // to:   circle(prev1+1) <--> (start2, current2)
                                 // Otherwise, we have to compute from the scratch
                                 localCase3 = localAligner(aRNA,prev1+1, current1,
                                            bRNA, current2, current2, gScore, semiCircleScore1,
                                            semiCircleScore2, lScore, circleScore, gTrace, lTrace, circleTrace,
                                            lSemiTrace1, lSemiTrace2, gSemiTrace1, gSemiTrace2, closeLoop,
                                            matcher, penalty);

                                 int element = aRNA.firstAlongCircle(current1);
                                 int comp = aRNA.getComponent(element);
                                 int firstBase = 0;
                                 if( comp < aRNA.seqLength())
                                    firstBase = comp;
                                 else
                                     firstBase = comp / aRNA.seqLength() - 1;

                                 globalCase3 = (semiCircleScore1[current1][current2] +
                                                       ( leftBondEnd1 - firstBase)*penalty);

                                 for(int i=0; i< localBackup.length; i++) {
                                     lScore[current1][i+first] = localBackup[i];
                                     gScore[current1][i+first] = globalBackup[i];
                                 }
                             }


                             // THEN, we are ready to compute CASE-1 and CASE-2
                             int currentBase2 = bRNA.getComponent(current2);
                             int currentBase1 = aRNA.getPairedBase(leftBondEnd1);

                             double localCase1 = (lScore[prev1][current2] +
                                             (currentBase1 - leftBondEnd1 + 1)*penalty);
                             double globalCase1 = gScore[prev1][current2] +
                                            (currentBase1 - leftBondEnd1 + 1)*penalty;

                             double localCase2 = 0;
                             double globalCase2 = 0;
                             if( (prev2 == start2 && start2 == end2) || (current2==start2 && start2!=end2))
                             { // current2 is the FIRST single base along loop circumference
                                 localCase2 = 0;

                                 int element = aRNA.firstAlongCircle(current1);
                                 int comp = aRNA.getComponent(element);
                                 int firstBase = 0;
                                 if( comp < aRNA.seqLength())
                                     firstBase = comp;
                                 else
                                     firstBase = comp / aRNA.seqLength() - 1;

                                 globalCase2 = (currentBase1-firstBase+2)*penalty;
                             } else {
                                 localCase2 = lScore[current1][prev2] + penalty;
                                 globalCase2 = gScore[current1][prev2] + penalty;
                             }


                             double gResult = Math.max(globalCase1, Math.max(globalCase2, globalCase3));
                             double lResult = Math.max(0, Math.max(localCase1, Math.max(localCase2, localCase3)));

                             gScore[current1][current2] = gResult;
                             lScore[current1][current2] = lResult;

                             // global alignment traceback
                             if( globalCase3 >= globalCase1 && globalCase3 >= globalCase2) // case3
                                 gTrace[current1][current2] = 3;
                             else if( globalCase2 >= globalCase1)  // case2
                                 gTrace[current1][current2] = 2;
                             else   // case1
                                 gTrace[current1][current2] = 1;

                             // local alignment tracebase:
                             if( localCase3 >= localCase1 && localCase3 >= localCase2)
                                 lTrace[current1][current2] = 3;
                             else if( localCase2 >= localCase1)
                                 lTrace[current1][current2] = 2;
                             else
                                 lTrace[current1][current2] = 1;
                         } else { // componnet1 is a base pair which is closing the LOOP
                                  // component2 is a single base
                             // two cases are considered:
                             // case1: discard the base pair
                             // case2: discard the single BASE
                             double case2 = 0;

                             int currentBase1= aRNA.getPairedBase(leftBondEnd1);
                             int currentBase2= bRNA.getComponent(current2);
                             prev1 = current1 - 1;

                             if(end1 == start1 && closeLoop == true) {
                                 if( current2 == start2)
                                     case2 = (currentBase1-leftBondEnd1+2)*penalty;
                                 else
                                     case2 = semiCircleScore1[current1][prev2] + penalty;
                             } else {   // Normal case OR derivation of case3 of skipping-loop + singlet above
                                 if( (start2 == prev2 && start2 == end2) || (current2==start2 && start2!=end2))
                                     case2 = (currentBase1-leftBondEnd1+2)*penalty;
                                 else
                                     case2 = semiCircleScore1[current1][prev2] + penalty;
                             }

                             double case1 = gScore[prev1][current2] + 2*penalty;

                             double gResult = Math.max(case1, case2);
                             double lResult = Math.max(0, gResult);

                             lScore[current1][current2] = lResult;

                             semiCircleScore1[current1][current2] = gResult;
                             gScore[current1][current2] = gResult;

                             // global alignment traceback information!
                             if( case2 >= case1)  // case2
                                 gSemiTrace1[current1][current2] = 12;
                             else  // case1
                                 gSemiTrace1[current1][current2] = 11;

                             // local alignment traceback information
                             if( case2 >= case1)
                                 lSemiTrace1[current1][current2] = 12;
                             else
                                 lSemiTrace1[current1][current2] = 11;
                         }
                         prevBase2 = component2;
                     } else { // TWO BASE PAIRS !!
                              // now, componet2 is also a base pair !!
                         // CASE of bonded base pair(component1->LOOP1) + bonded base pair(component2->LOOP2)
                         if( (leftBondEnd1 == prevBase1 + 1 || (start1==end1 && closeLoop == false)) &&
                             (leftBondEnd2 == prevBase2 + 1 || (start2==end2 && closeLoop == false)) )
                         {   // both are base pairs which are skipping the enclosed LOOPs
                             // five cases are considered:
                             // case1: both LOOPs are kept and matched with each other (2)
                             // case2: discard LOOP1; (0)
                             // case3: discard LOOP2; (1)
                             // case4: both LOOPs are kept where LOOP1 will match to the whole RNA2 (4)
                             // case5: both LOOPs are kets where LOOP2 will match to the whole RNA1 (3)

                             //!!!!!!!!! Here the order of computing is IMPORTANT !!!!!!!!!!
                             //FIRST STEP: We compute CASE-1
                             double temp1 = 0;
                             if( (start1==end1 && closeLoop==false) || (start2==end2 && closeLoop==false))
                             {
                                 temp1 = circleScore[current1][current2];

                                 if( start1 == end1 && closeLoop == false)
                                     prev1 = aRNA.beforeAlongCircle(current1);

                                 if( start2 == end2 && closeLoop == false)
                                     prev2 = bRNA.beforeAlongCircle(current2);
                             } else {
                                 double[] localBackup1 = null;
                                 double[] globalBackup1= null;
                                 double[] localBackup2 = null;
                                 double[] globalBackup2= null;

                                 //some bookkeeping work has to be done
                                 localBackup1 = new double[current2 - prev2];
                                 globalBackup1= new double[current2 - prev2];
                                 for(int i=0; i<localBackup1.length; i++) {
                                     localBackup1[i] = lScore[current1][i+(prev2+1)];
                                     lScore[current1][i+(prev2+1)] = semiCircleScore1[current1][i+(prev2+1)];

                                     globalBackup1[i]= gScore[current1][i+(prev2+1)];
                                     gScore[current1][i+(prev2+1)] = semiCircleScore1[current1][i+(prev2+1)];
                                 }

                                 localBackup2 = new double[current1 - prev1];
                                 globalBackup2= new double[current1 - prev1];
                                 for(int i=0; i<localBackup2.length; i++) {
                                     localBackup2[i] = lScore[i+(prev1+1)][current2];
                                     lScore[i+(prev1+1)][current2] = semiCircleScore2[current2][i+(prev1+1)];

                                     globalBackup2[i]= gScore[i+(prev1+1)][current2];
                                     gScore[i+(prev1+1)][current2] = semiCircleScore2[current2][i+(prev1+1)];
                                 }

                                 // the last parameter is ignored !!
                                 localAligner(aRNA, prev1+1, current1,
                                             bRNA,prev2+1, current2, gScore, semiCircleScore1, semiCircleScore2,
                                             lScore, circleScore, gTrace, lTrace, circleTrace,
                                             lSemiTrace1, lSemiTrace2, gSemiTrace1, gSemiTrace2,
                                             closeLoop, matcher, penalty);

                                 temp1 = circleScore[current1][current2];

                                 //restore the initial status.
                                 for(int i=0; i<localBackup1.length; i++) {
                                    lScore[current1][i+(prev2+1)] = localBackup1[i];
                                    gScore[current1][i+(prev2+1)] = globalBackup1[i];
                                 }

                                 for(int i=0; i<localBackup2.length; i++) {
                                     lScore[i+(prev1+1)][current2] = localBackup2[i];
                                     gScore[i+(prev1+1)][current2] = globalBackup2[i];
                                 }
                             }

                             double localCase1 = (lScore[prev1][prev2] + temp1);
                             double globalCase1= (gScore[prev1][prev2] + temp1);
                             // CASE-1 is finished here


                             //SECOND STEP: we compute CASE-4
                             double localCase4 = 0;
                             double globalCase4= 0;
                             //int currentBase1= aRNA.getPairedBase(leftBondEnd1);
                             if( start1==end1 && closeLoop==false) {
                                 localCase4 = semiCircleScore1[current1][current2];
                                 if( localCase4 < 0)
                                     localCase4 = 0;

                                 int element = aRNA.firstAlongCircle(current1);
                                 int comp = aRNA.getComponent(element);
                                 int firstBase = 0;
                                 if( comp < aRNA.seqLength())
                                     firstBase = comp;
                                 else
                                     firstBase = comp / aRNA.seqLength() - 1;

                                 globalCase4 = (semiCircleScore1[current1][current2]
                                                     + (leftBondEnd1 - firstBase)*penalty);
                             } else {
                                 //some bookkeeping work has to be done
                                 int first = start2;
                                 if( start2 == end2)
                                     first = bRNA.firstAlongCircle(start2);

                                 double[] localBackup = new double[prev2 - first + 1];
                                 double[] globalBackup= new double[prev2 - first + 1];

                                 for(int i=0; i<localBackup.length; i++) {
                                     localBackup[i] = lScore[current1][i+first];
                                     lScore[current1][i+first] = semiCircleScore1[current1][i+first];

                                     globalBackup[i]= gScore[current1][i+first];
                                     gScore[current1][i+first] = semiCircleScore1[current1][i+first];
                                 }

                                 // We only need to extend the computing
                                 // from:  circle(prev1+1, current1) <--> (start2, prev2)
                                 // to:    circle(prev1+1, current1) <--> (start2, current2)
                                 // Furthermore, the algorithm has to be notified that the base
                                 // pair, 'current2', is skipping a loop instead of closing loop,
                                 // to accomplish this, the last parameter is set to be 'false'
                                 localCase4 = localAligner(aRNA, prev1+1, current1,
                                                 bRNA, current2, current2, gScore, semiCircleScore1, semiCircleScore2,
                                                 lScore, circleScore, gTrace, lTrace, circleTrace,
                                                 lSemiTrace1, lSemiTrace2, gSemiTrace1, gSemiTrace2,
                                                 false, matcher, penalty);

                                 //lSemiCircleScore1[current1][current2] = localCase4;

                                 int element = aRNA.firstAlongCircle(current1);
                                 int comp = aRNA.getComponent(element);
                                 int firstBase = 0;
                                 if( comp < aRNA.seqLength())
                                     firstBase = comp;
                                 else
                                     firstBase = comp / aRNA.seqLength() - 1;

                                 globalCase4 = (semiCircleScore1[current1][current2] +
                                                        (leftBondEnd1 - firstBase)*penalty);

                                 //restore the initial status.
                                 for(int i=0; i<localBackup.length; i++) {
                                     lScore[current1][i+first] = localBackup[i];
                                     gScore[current1][i+first] = globalBackup[i];
                                 }
                             }
                             // CASE-4 is finished


                             // THIRD STEP: we compute CASE-5
                             double localCase5 = 0;
                             double globalCase5= 0;

                             if( (start2==end2 && closeLoop==false)) {
                                 localCase5 = semiCircleScore2[current2][current1];
                                 if( localCase5 < 0)
                                     localCase5 = 0;

                                 int element = bRNA.firstAlongCircle(current2);
                                 int comp = bRNA.getComponent(element);
                                 int firstBase = 0;
                                 if( comp < bRNA.seqLength())
                                     firstBase = comp;
                                 else
                                     firstBase = comp / bRNA.seqLength() - 1;

                                 globalCase5 = semiCircleScore2[current2][current1] +
                                                  (leftBondEnd2 - firstBase)*penalty;
                             } else {
                                 //some bookkeeping work has to be done firstly
                                 int first = start1;
                                 if( start1 == end1)
                                     first = aRNA.firstAlongCircle(start1);
                                 double[] localBackup = new double[prev1-first+1];
                                 double[] globalBackup= new double[prev1-first+1];

                                 for(int i=0; i<localBackup.length; i++) {
                                     localBackup[i] = lScore[i+first][current2];
                                     lScore[i+first][current2] = semiCircleScore2[current2][i+first];

                                     globalBackup[i]= gScore[i+first][current2];
                                     gScore[i+first][current2] = semiCircleScore2[current2][i+first];
                                 }

                                 // Based on what has obtained from case1, we only need to extend the computing
                                 // from:   (start1, prev1) <--> circle(prev2+1, current2)
                                 // to:     (start1, current1) <--> circle(prev2+1, current2)
                                 // Furthermore, the algorithm has to be notified that the base
                                 // pair, 'current1', is skipping a loop instead of closing loop
                                 // to accomplish this, last parameter is set to be 'false'

                                 localCase5 = localAligner(aRNA, current1, current1,
                                                 bRNA, prev2+1, current2, gScore, semiCircleScore1,
                                                 semiCircleScore2, lScore, circleScore, gTrace, lTrace, circleTrace,
                                                 lSemiTrace1, lSemiTrace2, gSemiTrace1, gSemiTrace2,
                                                 false, matcher, penalty);

                                 int element = bRNA.firstAlongCircle(current2);
                                 int comp = bRNA.getComponent(element);
                                 int firstBase = 0;
                                 if( comp < bRNA.seqLength())
                                     firstBase = comp;
                                 else
                                     firstBase = comp / bRNA.seqLength() -  1;

                                 globalCase5 = semiCircleScore2[current2][current1] +
                                                            (leftBondEnd2-firstBase)*penalty;

                                 //restore the initial status.
                                 for(int i=0; i<localBackup.length; i++) {
                                     lScore[i+first][current2] = localBackup[i];
                                     gScore[i+first][current2] = globalBackup[i];
                                 }
                             }
                             // CASE-5 is finished !

                             // FOUTH and FIFTH STEPS are to compute CASE-2 and CASE-3
                             int currentBase1 = aRNA.getPairedBase(leftBondEnd1);
                             int currentBase2 = bRNA.getPairedBase(leftBondEnd2);

                             double localCase2 = (lScore[prev1][current2] +
                                             (currentBase1 - leftBondEnd1 + 1)*penalty);
                             double globalCase2= (gScore[prev1][current2] +
                                            (currentBase1 - leftBondEnd1 + 1)*penalty);

                             double localCase3 = (lScore[current1][prev2] +
                                             (currentBase2 - leftBondEnd2 + 1)*penalty);
                             double globalCase3= (gScore[current1][prev2] +
                                            (currentBase2 - leftBondEnd2 + 1)*penalty);

                             double gResult = Math.max(globalCase1, Math.max(globalCase2, Math.max(globalCase3,
                                                 Math.max(globalCase4, globalCase5))));

                             double lResult = Math.max(0, Math.max(localCase1, Math.max(localCase2, Math.max(
                                                        localCase3, Math.max(localCase4, localCase5)))));

                             gScore[current1][current2] = gResult;
                             lScore[current1][current2] = lResult;

                             // global alignment taceback information
                             if( globalCase1>=globalCase2 && globalCase1>=globalCase3
                                      && globalCase1>=globalCase4 && globalCase1>=globalCase5) {
                                 gTrace[current1][current2] = 1;
                             } else if( globalCase4>=globalCase2 && globalCase4>=globalCase3
                                     && globalCase4>=globalCase5) {
                                 gTrace[current1][current2] = 4;
                             } else if( globalCase5>=globalCase2 && globalCase5>=globalCase3) {
                                 gTrace[current1][current2] = 5;
                             } else if( globalCase2>=globalCase3) {
                                 gTrace[current1][current2] = 2;
                             } else {
                                 gTrace[current1][current2] = 3;
                             }

                             // local alignment traceback information
                             if( localCase1>=localCase2 && localCase1>=localCase3 && localCase1>=localCase4
                                              && localCase1>=localCase5) {
                                 lTrace[current1][current2] = 1;
                             } else if( localCase4>=localCase2 && localCase4>=localCase3 && localCase4>=localCase5) {
                                 lTrace[current1][current2] = 4;
                             } else if( localCase5>=localCase2 && localCase5>=localCase3) {
                                 lTrace[current1][current2] = 5;
                             } else if( localCase2 >= localCase3) {
                                 lTrace[current1][current2] = 2;
                             } else {
                                 lTrace[current1][current2] = 3;
                             }

                         } else if( leftBondEnd1 == prevBase1+1 || (end1 == start1 && closeLoop==false) )
                         // component1 is skipping the enclosed LOOP1 while component2 is closing its LOOP2
                         {
                             // three cases are considered:
                             // case1: both LOOPs are kept and matched with each other (0)
                             // case2: LOOP1 is discarded.(1)
                             // case3: LOOP2 is opened by discarding the enclosing pair (2)

                             if( start2==end2 && closeLoop == true)
                                 prev2 = end2 - 1;

                             double localCase1 = 0;
                             double globalCase1= 0;
                             if( end1==start1 && closeLoop == false) {
                                 prev1 = aRNA.beforeAlongCircle(current1);

                                 int element = aRNA.firstAlongCircle(current1);
                                 int comp = aRNA.getComponent(element);
                                 int firstBase = 0;
                                 if( comp < aRNA.seqLength())
                                     firstBase = comp;
                                 else
                                     firstBase = comp / aRNA.seqLength() - 1;

                                 localCase1 = circleScore[current1][current2];
                                 if( localCase1 < 0)
                                     localCase1 = 0;

                                 globalCase1= circleScore[current1][current2] +
                                                           (leftBondEnd1-firstBase)*penalty;
                             } else {
                                 int first = start2;
                                 if( start2 == end2)
                                     first = bRNA.firstAlongCircle(start2-1);

                                 double[] localBackup = new double[current2 - first + 1];
                                 double[] globalBackup= new double[current2 - first + 1];
                                 for(int i=0; i<localBackup.length;i++) {
                                     localBackup[i] = lScore[current1][i+first];
                                     lScore[current1][i+first] = semiCircleScore1[current1][i+first];

                                     globalBackup[i] = gScore[current1][i+first];
                                     gScore[current1][i+first] = semiCircleScore1[current1][i+first];
                                 }

                                 // Normally, we only need to extend computing
                                 // from: circle(prev1+1, current1) <--> (start2, prev2)
                                 // to: circle(prev1+1, current1) <--> circle(start2, current2);
                                 //
                                 // Furthermore, the algorithm has to be notified that pair(current2)
                                 // is closing loop instead of skipping loop
                                 // to acomplish this, the last parameter is set to be 'true'
                                 localCase1 = localAligner(aRNA, prev1+1, current1,
                                                 bRNA, current2, current2, gScore, semiCircleScore1,
                                                 semiCircleScore2, lScore,
                                                 circleScore, gTrace, lTrace, circleTrace,
                                                 lSemiTrace1, lSemiTrace2, gSemiTrace1, gSemiTrace2,
                                                 true, matcher, penalty);

                                 //circleScore[current1][current2] = localCase1;

                                 int element = aRNA.firstAlongCircle(current1);
                                 int comp = aRNA.getComponent(element);
                                 int firstBase = 0;
                                 if( comp < aRNA.seqLength())
                                     firstBase = comp;
                                 else
                                     firstBase = comp / aRNA.seqLength() - 1;

                                 globalCase1 = circleScore[current1][current2]
                                                        + (leftBondEnd1-firstBase)*penalty;

                                 for(int i=0; i<localBackup.length; i++) {
                                     lScore[current1][i+first] = localBackup[i];
                                     gScore[current1][i+first] = globalBackup[i];
                                 }
                             }

                             int currentBase1 = aRNA.getPairedBase(leftBondEnd1);

                             double localCase2 = (lScore[prev1][current2] +
                                                 (currentBase1 - leftBondEnd1 + 1)*penalty);
                             double globalCase2= (gScore[prev1][current2] +
                                                (currentBase1 - leftBondEnd1 + 1)*penalty);

                             double localCase3 = gScore[current1][prev2] + 2*penalty;
                             double globalCase3= gScore[current1][prev2] + 2*penalty;

                             double gResult = Math.max(globalCase1, Math.max(globalCase2, globalCase3));
                             double lResult = Math.max(0, Math.max(localCase1, Math.max(localCase2, localCase3)));

                             gScore[current1][current2] = gResult;
                             semiCircleScore2[current2][current1] = gResult;

                             lScore[current1][current2] = lResult;

                             // global alignment traceback information
                             if( globalCase1 >= globalCase2 && globalCase1 >= globalCase3)
                                 gSemiTrace2[current2][current1] = 1;
                             else if( globalCase3 >= globalCase2)
                                 gSemiTrace2[current2][current1] = 3;
                             else    //case2
                                 gSemiTrace2[current2][current1] = 2;

                             // local alignment traceback information
                             if( localCase1 >= localCase2 && localCase1 >= localCase3)
                                 lSemiTrace2[current2][current1] = 1;
                             else if( localCase3 >= localCase2)
                                 lSemiTrace2[current2][current1] = 3;
                             else
                                 lSemiTrace2[current2][current1] = 2;

                         } else if( leftBondEnd2 == prevBase2+1 || (end2 == start2 && closeLoop == false))
                         // component1 is closing its LOOP1 while component2 is skipping the enclosed LOOP2
                         {
                             // two cases are considered:
                             // case1: both LOOPs are kept and matched with each other(0)
                             // case2: LOOP2 is discarded.(1)
                             // case3: LOOP1 is opened by discarding its enclosing pair(2)

                             if( start1 == end1 && closeLoop == true)
                                 prev1 = start1 - 1;

                             // get the first component of Loop2
                             if( end2 == start2 && closeLoop == false)
                                 prev2 = bRNA.beforeAlongCircle(current2);

                             double localCase1 = 0;
                             double globalCase1= 0;
                             if( end2 == start2 && closeLoop == false) {
                                 localCase1 = circleScore[current1][current2];
                                 if( localCase1 < 0)
                                     localCase1 = 0;

                                 int element = bRNA.firstAlongCircle(current2);
                                 int comp = bRNA.getComponent(element);
                                 int firstBase = 0;
                                 if( comp < bRNA.seqLength())
                                     firstBase = comp;
                                 else
                                     firstBase = comp / bRNA.seqLength() - 1;

                                 globalCase1 = circleScore[current1][current2] +
                                                       (leftBondEnd2 - firstBase)*penalty;
                             } else {
                                 int first = start1;
                                 if( start1 == end1)
                                     first = aRNA.firstAlongCircle(start1-1);

                                 double[] localBackup = new double[current1 - first + 1];
                                 double[] globalBackup= new double[current1 - first + 1];
                                 for(int i=0; i<globalBackup.length; i++) {
                                     localBackup[i] = lScore[i+first][current2];
                                     lScore[i+first][current2] = semiCircleScore2[current2][i+first];

                                     globalBackup[i]= gScore[i+first][current2];
                                     gScore[i+first][current2] = semiCircleScore2[current2][i+first];
                                 }

                                 // Normally, we only need to extend the computing
                                 // from: (start1, prev1) <--> circle(prev2+1, current2)
                                 // to:   circle(start1, current1) <--> circle(prev2+1, current2)
                                 //
                                 // Futhermore, the algorithm has to be notified that pair(current1)
                                 // is enclosing a loop instead of skipping a loop
                                 // to accomplish this, last parameter is set to be 'true'

                                 localCase1 = localAligner(aRNA, current1, current1, bRNA, prev2+1,
                                                 current2, gScore, semiCircleScore1, semiCircleScore2,
                                                 lScore, circleScore, gTrace, lTrace, circleTrace,
                                                 lSemiTrace1, lSemiTrace2, gSemiTrace1, gSemiTrace2,
                                                 true, matcher, penalty);

                                 int element = bRNA.firstAlongCircle(current2);
                                 int comp = bRNA.getComponent(element);
                                 int firstBase = 0;
                                 if( comp < bRNA.seqLength())
                                     firstBase = comp;
                                 else
                                     firstBase = comp / bRNA.seqLength() - 1;

                                 globalCase1 = circleScore[current1][current2] +
                                                       (leftBondEnd2 - firstBase)*penalty;

                                 for(int i=0; i<localBackup.length; i++) {
                                     lScore[i+first][current2] = localBackup[i];
                                     gScore[i+first][current2] = globalBackup[i];
                                 }
                             }

                             int currentBase2 = bRNA.getPairedBase(leftBondEnd2);
                             double localCase2 = (lScore[current1][prev2] +
                                                 (currentBase2 - leftBondEnd2 + 1)*penalty);
                             double globalCase2= (gScore[current1][prev2] +
                                                (currentBase2 - leftBondEnd2 + 1)*penalty);

                             double localCase3 = gScore[prev1][current2] + 2*penalty;
                             double globalCase3= gScore[prev1][current2] + 2*penalty;


                             double gResult = Math.max(globalCase1, Math.max(globalCase2, globalCase3));
                             double lResult = Math.max(0, Math.max(localCase1, Math.max(localCase2, localCase3)));

                             gScore[current1][current2] = gResult;
                             semiCircleScore1[current1][current2] = gResult;

                             lScore[current1][current2] = lResult;


                             // global alignment traceback information
                             if( globalCase1 >= globalCase2 && globalCase1 >= globalCase3) {
                                 gSemiTrace1[current1][current2] = 1;
                             } else if( globalCase3 >= globalCase2) {
                                 gSemiTrace1[current1][current2] = 3;
                             } else {  //case2
                                 gSemiTrace1[current1][current2] = 2;
                             }

                             // local alignment traceback information
                             if( localCase1 >= localCase2 && localCase1 >= localCase3) {
                                 lSemiTrace1[current1][current2] = 1;
                             } else if( localCase3 >= localCase2) {
                                 lSemiTrace1[current1][current2] = 3;
                             } else {
                                 lSemiTrace1[current1][current2] = 2;
                             }
                         } else { // both components are closing the enclosed LOOPs.
                             // three cases are considered:
                             // case1: both components are kept and matched with each other
                             // case2: component1 is discarded
                             // case3: component2 is discarded
                             double match = 0;
                             if( matcher == null){
                                 match = (double)RNA.baseMatch(aRNA.getBaseBond(current1), bRNA.getBaseBond(current2));
                             	System.out.println("Match is Null");
			     }else{  //#########################change#############
                             
        		     	mcomponent = aRNA.getComponent(current1);
        		     	if (mcomponent < aRNA.sequence.length()){
            				m1 = mcomponent;
					m2 = -1;
			
        		        }else {
            				m1 = mcomponent / aRNA.seqLength() - 1;
           		        	m2 = mcomponent % aRNA.seqLength();
        		     	}
				 match = matcher.baseMatch2(aRNA.flag,aRNA.getBaseBond(current1), bRNA.getBaseBond(current2),m1,m2,send_conserved,factor,aRNA.factors_for_conserved);
			     }
                             if( end1 == start1 && closeLoop == true)
                                 prev1 = current1 - 1;
                             if( end2 == start2 && closeLoop == true)
                                 prev2 = current2 - 1;

                             double case1 = (gScore[prev1][prev2] + match);
                             double case2 = semiCircleScore2[current2][prev1] + 2*penalty;
                             double case3 = semiCircleScore1[current1][prev2] + 2*penalty;

                             double result = Math.max(case1, Math.max(case2,case3));

                             gScore[current1][current2] = result;
                             lScore[current1][current2] = Math.max(result, 0);
                             circleScore[current1][current2] = result;

                             if( aRNA.beforeAlongCircle(current1) == -1)  // first component of loop
                                 semiCircleScore2[current2][current1] = result;
                             if( bRNA.beforeAlongCircle(current2) == -1) // first component of loop
                                 semiCircleScore1[current1][current2] = result;

                             if( case1 >= case2 && case1 >= case3)
                                 circleTrace[current1][current2] = 1;
                             else if( case2 >= case3)
                                 circleTrace[current1][current2] = 2;
                             else
                                 circleTrace[current1][current2] = 3;
                         }

                         prevBase2 = component2 % bRNA.seqLength();
                     }

                     prev2 = current2;
                     current2 = bRNA.nextAlongCircle(current2);
                     if(current2 != -1) {
                         component2 = bRNA.getComponent(current2);
                         if( component2 < bRNA.seqLength())
                             leftBondEnd2 = component2;
                         else
                             leftBondEnd2 = component2 / bRNA.seqLength() -1;
                     }
                 } while ( current2 != -1 && current2 <= end2);

                 prevBase1 = component1 % aRNA.seqLength();
             }

             prev1 = current1;
             current1 = aRNA.nextAlongCircle(current1);
             if(current1 != -1) {
                 component1 = aRNA.getComponent(current1);
                 if( component1 < aRNA.seqLength())
                     leftBondEnd1 = component1;
                 else
                     leftBondEnd1 = component1 / aRNA.seqLength() - 1;
             }
         } while ( current1 != -1 && current1 <= end1);

         return lScore[end1][end2];
     }

     //--------------------------------------------------------------------------------------
     // This function find the optimal score from the ouput of structural local alignment
     // algorithm. If required, the detailed alignment maps based on the traceback information
     // will also be constructed through calling "traceLocalMap2D()".
     //--------------------------------------------------------------------------------------
     private static double localMap2D(RNA aRNA, int[] aMap,
                             RNA bRNA, int[] bMap, byte[][] lTrace, byte[][] gTrace,
                             byte[][] circleTrace, byte[][] lSemiTrace1,
                             byte[][] lSemiTrace2, byte[][] gSemiTrace1, byte[][] gSemiTrace2,
                             double[][] lScore, double[][] circleScore)
     {
         double max = Double.NEGATIVE_INFINITY;

         int row = aRNA.postOrderPairs.size();
         int col = bRNA.postOrderPairs.size();

         int RowPos = 0;
         int ColPos = 0;
         for(int j=col-1; j>=0; j--)
         for(int i=row-1; i>=0; i--) {
             if( lScore[i][j] > max) {
                 max = lScore[i][j];
                 RowPos = i;
                 ColPos = j;
             }
         }

         boolean flag1 = (aRNA.beforeAlongCircle(RowPos) != -1);
         boolean flag2 = (bRNA.beforeAlongCircle(ColPos) != -1);

         if( aMap != null && bMap != null) {
             traceLocalMap2D(aRNA, 0, RowPos, aMap, bRNA, 0, ColPos, bMap, lScore,
                             lTrace, gTrace, circleTrace, lSemiTrace1, lSemiTrace2,
                             gSemiTrace1, gSemiTrace2, flag1, flag2, true);
         }
         return max;
     }

     //--------------------------------------------------------------------------------
     // This is the function to reconstruct the alignment map from taceback info along
     // with the LOCAL structure alignment algorithm.
     //--------------------------------------------------------------------------------
     private static void traceLocalMap2D(RNA aRNA, int start1, int end1, int[] aMap,
                          RNA bRNA, int start2, int end2, int[] bMap, double[][] lScore, byte[][] lTrace,
                          byte[][] gTrace, byte[][] circleTrace, byte[][] lSemiTrace1,
                          byte[][] lSemiTrace2, byte[][] gSemiTrace1, byte[][] gSemiTrace2,
                          boolean aflag1, boolean aflag2, boolean LocalContext)
     {
         boolean flag1 = aflag1;
         if( aRNA.beforeAlongCircle(end1) == -1)   // unique choice
             flag1 = false;

         boolean flag2 = aflag2;
         if( bRNA.beforeAlongCircle(end2) == -1)  // unique choice
             flag2 = false;

         int component1 = 0;
         int component2 = 0;
         boolean single1 = false;
         boolean single2 = false;


         do {
             if( LocalContext == true && lScore[end1][end2] <= 0)
                 break;

             component1 = aRNA.getComponent(end1);
             component2 = bRNA.getComponent(end2);

             single1 = (component1 < aRNA.seqLength());
             single2 = (component2 < bRNA.seqLength());

             if( single1 == true && single2 == true) {
                 int traceInfo;
                 if( LocalContext == true)
                     traceInfo = lTrace[end1][end2];
                 else
                     traceInfo = gTrace[end1][end2];

                 if(traceInfo == 2) {
                     end2 = bRNA.beforeAlongCircle(end2);
                     flag2 = (bRNA.beforeAlongCircle(end2) != -1);
                 } else if(traceInfo == 3) {
                     end1 = aRNA.beforeAlongCircle(end1);
                     flag1 = (aRNA.beforeAlongCircle(end1) != -1);
                 } else if( traceInfo == 1) {
                     aMap[component1] = component2 + 1;   // base number is started from "1"
                     bMap[component2] = component1 + 1;
                     end1 = aRNA.beforeAlongCircle(end1);
                     end2 = bRNA.beforeAlongCircle(end2);

                     flag1 = (aRNA.beforeAlongCircle(end1) != -1);
                     flag2 = (bRNA.beforeAlongCircle(end2) != -1);
                 } else {
                     System.out.println("!!!!!!   ERROR !!!!!!!!");
                     System.out.println("single1 : " + single1 + "  single2:" + single2);
                     System.out.println("flag1 : " + flag1 + "  flag2:" + flag2);
                     System.out.println("trace[" + end1 + "][" + end2 + "]: " + traceInfo);
                     System.exit(0);
                 }
             } else if(single1 == true) {
                 if ( flag2 == true){  // component2 is skipping back base pair
                     int traceInfo;
                     if( LocalContext == true)
                         traceInfo = lTrace[end1][end2];
                     else
                         traceInfo = gTrace[end1][end2];

                     if( traceInfo == 1){  // the loop is discarded
                         end2 = bRNA.beforeAlongCircle(end2);
                         flag2 = (bRNA.beforeAlongCircle(end2) != -1);
                     } else if( traceInfo == 2){ // the single base is discarded
                         end1 = aRNA.beforeAlongCircle(end1);
                         flag1 = (aRNA.beforeAlongCircle(end1) != -1);
                     } else if( traceInfo == 3) {
                         int newstart2 = bRNA.beforeAlongCircle(end2) + 1;
                         traceLocalMap2D(aRNA, start1, end1, aMap, bRNA, newstart2, end2, bMap, lScore, null,
                                 gTrace, circleTrace, null, null, gSemiTrace1, gSemiTrace2, flag1, false, false);
                         return;
                     } else {
                         System.out.println("!!!!!!   ERROR !!!!!!!!");
                         System.out.println("single1 : " + single1 + "  single2:" + single2);
                         System.out.println("flag1 : " + flag1 + "  flag2:" + flag2);
                         System.out.println("trace[" + end1 + "][" + end2 + "]: " + traceInfo);
                         System.exit(0);
                     }
                 } else if( flag2 == false){ // component2 is enterring a loop
                     int traceInfo;
                     if( LocalContext == true)
                         traceInfo = lSemiTrace2[end2][end1];
                     else
                         traceInfo = gSemiTrace2[end2][end1];

                     if( traceInfo == 11){   // the pair is discarded
                         end2 --;
                         flag2 = (bRNA.beforeAlongCircle(end2) != -1);

                         // change the alignment context to Global
                         LocalContext = false;
                     } else if( traceInfo == 12){  // the single base is discarded
                         end1 = aRNA.beforeAlongCircle(end1);
                         flag1 = (aRNA.beforeAlongCircle(end1) != -1);
                     } else {
                         System.out.println("!!!!!!   ERROR !!!!!!!!");
                         System.out.println("single1 : " + single1 + "  single2:" + single2);
                         System.out.println("flag1 : " + flag1 + "  flag2:" + flag2);
                         System.out.println("semiTrace2[" + end2 + "][" + end1 + "]: " + traceInfo);
                         System.exit(0);
                     }
                 }
             } else if( single2 == true) {
                 if( flag1 == true){  // component1 is skipping back base pair
                     int traceInfo;
                     if( LocalContext == true)
                         traceInfo = lTrace[end1][end2];
                     else
                         traceInfo = gTrace[end1][end2];

                     if( traceInfo == 1){  // loop is discarded
                         end1 = aRNA.beforeAlongCircle(end1);
                         flag1 = (aRNA.beforeAlongCircle(end1) != -1);
                     } else if( traceInfo == 2){ // single base is discarded
                         end2 = bRNA.beforeAlongCircle(end2);
                         flag2 = (bRNA.beforeAlongCircle(end2) != -1);
                     } else if( traceInfo == 3) {
                         int newstart1 = aRNA.beforeAlongCircle(end1) + 1;
                         traceLocalMap2D(aRNA, newstart1, end1, aMap, bRNA, start2, end2, bMap, lScore,
                                 null, gTrace, circleTrace, null, null, gSemiTrace1, gSemiTrace2, false, flag2,false);
                         return;
                     } else {
                         System.out.println("!!!!!!   ERROR !!!!!!!!");
                          System.out.println("single1 : " + single1 + "  single2:" + single2);
                          System.out.println("flag1 : " + flag1 + "  flag2:" + flag2);
                          System.out.println("trace[" + end1 + "][" + end2 + "]: " + traceInfo);
                          System.exit(0);
                     }
                 } else if( flag1 == false) {
                     int traceInfo;
                     if( LocalContext == true)
                         traceInfo = lSemiTrace1[end1][end2];
                     else
                         traceInfo = gSemiTrace1[end1][end2];

                     if( traceInfo == 11){  // discard the base pair
                         end1 --;
                         flag1 = (aRNA.beforeAlongCircle(end1) != -1);

                         // change the alignment context to Global
                         LocalContext = false;
                     } else if( traceInfo == 12){ //discard the single base
                         end2 = bRNA.beforeAlongCircle(end2);
                         flag2 = (bRNA.beforeAlongCircle(end2) != -1);
                     } else {
                         System.out.println("!!!!!!   ERROR !!!!!!!!");
                          System.out.println("single1 : " + single1 + "  single2:" + single2);
                          System.out.println("flag1 : " + flag1 + "  flag2:" + flag2);
                          System.out.println("semiTrace1[" + end1 + "][" + end2 + "]: " + traceInfo);
                          System.exit(0);
                     }
                 }
             } else {    // both are base pairs
                 if( flag1 == true && flag2 == true){  // both components are skipping back pairs
                     int newstart1 = aRNA.beforeAlongCircle(end1) + 1;
                     int newstart2 = bRNA.beforeAlongCircle(end2) + 1;

                     byte choice;
                     if( LocalContext == true)
                         choice = lTrace[end1][end2];
                     else
                         choice = gTrace[end1][end2];

                     if( choice == 2){  // case2: discard loop1
                         end1 = aRNA.beforeAlongCircle(end1);
                         flag1 = (aRNA.beforeAlongCircle(end1) !=-1);
                     } else if( choice == 3){ // case3: discard loop2
                         end2 = bRNA.beforeAlongCircle(end2);
                         flag2 = (bRNA.beforeAlongCircle(end2) != -1);
                     } else if( choice == 1){  // case1: loop1 <--> loop2
                         flag1 = (aRNA.beforeAlongCircle(newstart1-1) != -1);
                         flag2 = (bRNA.beforeAlongCircle(newstart2-1) != -1);

                         traceLocalMap2D(aRNA, start1, newstart1-1, aMap, bRNA, start2, newstart2-1, bMap, lScore,
                                 lTrace, gTrace, circleTrace, lSemiTrace1, lSemiTrace2,
                                 gSemiTrace1, gSemiTrace2, flag1, flag2, LocalContext);

                         traceLocalMap2D(aRNA, newstart1, end1, aMap, bRNA, newstart2, end2, bMap, null,
                                 null, gTrace, circleTrace, null, null,
                                 gSemiTrace1, gSemiTrace2, false, false, false);
                         return;
                     } else if( choice == 5){ // case5: loop2 <--> whole RNA1
                         traceLocalMap2D(aRNA, start1, end1, aMap, bRNA, newstart2, end2, bMap, lScore, lTrace,
                                 gTrace, circleTrace, lSemiTrace1, lSemiTrace2, gSemiTrace1,
                                 gSemiTrace2, flag1, false, LocalContext);
                         return;
                     } else if( choice == 4){ // case4: loop1 <---> whole RNA2
                         traceLocalMap2D(aRNA, newstart1, end1, aMap, bRNA, start2, end2, bMap, lScore, lTrace,
                                 gTrace, circleTrace, lSemiTrace1, lSemiTrace2, gSemiTrace1,
                                 gSemiTrace2, false, flag1, LocalContext);
                         return;
                     } else {
                         System.out.println("!!!!!!   ERROR !!!!!!!!");
                         System.out.println("single1 : " + single1 + "  single2:" + single2);
                         System.out.println("flag1 : " + flag1 + "  flag2:" + flag2);
                         System.out.println("trace[" + end1 + "][" + end2 + "]: " + choice);
                         System.exit(0);
                     }
                 } else if( flag1 == true){  // component1 is skipping and component2 is entering
                     int newstart1 = aRNA.beforeAlongCircle(end1) + 1;

                     byte choice;
                     if( LocalContext == true)
                         choice = lSemiTrace2[end2][end1];
                     else
                         choice = gSemiTrace2[end2][end1];

                     if( choice == 1){ // case1: loop1 <--> loop2
                         traceLocalMap2D(aRNA, newstart1, end1, aMap, bRNA, start2, end2, bMap, lScore, null,
                                 gTrace, circleTrace, null, null, gSemiTrace1,
                                 gSemiTrace2, false, false, false);
                         return;
                     } else if( choice == 2){  // case2: discard loop1
                         flag1 = (aRNA.beforeAlongCircle(newstart1-1) != -1);

                         traceLocalMap2D(aRNA, start1, newstart1-1, aMap, bRNA, start2, end2, bMap, lScore, lTrace,
                                     gTrace, circleTrace, lSemiTrace1, lSemiTrace2,
                                     gSemiTrace1, gSemiTrace2, flag1, false, LocalContext);
                         return;
                     } else if( choice == 3){  // case3: loop2 is opened
                         end2 --;
                         flag2 = (bRNA.beforeAlongCircle(end2) != -1);

                         // alignment context is changed to Global
                         LocalContext = false;
                     } else {
                         System.out.println("!!!!!!   ERROR !!!!!!!!");
                         System.out.println("single1 : " + single1 + "  single2:" + single2);
                         System.out.println("flag1 : " + flag1 + "  flag2:" + flag2);
                         System.out.println("semiTrace2[" + end2 + "][" + end1 + "]: " + choice);
                         System.exit(0);
                     }
                 } else if( flag2 == true){  // component2 is skipping and component1 is entering
                     int newstart2 = bRNA.beforeAlongCircle(end2) + 1;

                     byte choice;
                     if( LocalContext == true)
                         choice = lSemiTrace1[end1][end2];
                     else
                         choice = gSemiTrace1[end1][end2];

                     if( choice == 1){  // case1: loop1 <--> loop2
                         traceLocalMap2D(aRNA, start1, end1, aMap, bRNA, newstart2, end2, bMap, null, null,
                                     gTrace, circleTrace, null, null, gSemiTrace1, gSemiTrace2,
                                     false, false, false);
                         return;
                     } else if( choice == 2){  // case2: loop2 is discarded
                         flag2 = (bRNA.beforeAlongCircle(newstart2-1) != -1);

                         traceLocalMap2D(aRNA, start1, end1, aMap, bRNA, start2, newstart2-1, bMap, lScore, lTrace,
                                     gTrace, circleTrace, lSemiTrace1, lSemiTrace2, gSemiTrace1,
                                     gSemiTrace2, false, flag2, LocalContext);
                         return;
                     } else if( choice == 3){  // case3: loop1 is opened
                         end1 --;
                         flag1 = (aRNA.beforeAlongCircle(end1) != -1);

                         // alignment context changed to Global
                         LocalContext = false;
                     } else {
                         System.out.println("!!!!!!   ERROR !!!!!!!!");
                         System.out.println("single1 : " + single1 + "  single2:" + single2);
                         System.out.println("flag1 : " + flag1 + "  flag2:" + flag2);
                         System.out.println("semiTrace1[" + end1 + "][" + end2 + "]: " + choice);
                         System.exit(0);
                     }
                 } else if(flag1 == false && flag2 == false){  // both are entering loop
                     byte choice = circleTrace[end1][end2];

                     // make sure the alignment context is Global
                     LocalContext = false;

                     if( choice == 3){  // case3: pair2 is discarded
                         end2 --;
                         flag2 = (bRNA.beforeAlongCircle(end2) != -1);
                     } else if( choice == 2){ // case2: pair1 is discarded
                         end1 --;
                         flag1 = (aRNA.beforeAlongCircle(end1) != -1);
                     } else if( choice == 1){  // case1: match
                         int left1 = component1 / aRNA.seqLength() - 1;
                         int right1 = component1 % aRNA.seqLength();

                         int left2 = component2 / bRNA.seqLength() - 1;
                         int right2 = component2 % bRNA.seqLength();

                         aMap[left1] = left2 + 1;    // base number is started from "1"
                         aMap[right1] = right2 + 1;
                         bMap[left2] = left1 + 1;
                         bMap[right2] = right1 + 1;

                         end1 --;
                         end2 --;
                         flag1 = (aRNA.beforeAlongCircle(end1) != -1);
                         flag2 = (bRNA.beforeAlongCircle(end2) != -1);
                     } else {
                         System.out.println("!!!!!!   ERROR !!!!!!!!");
                         System.out.println("single1 : " + single1 + "  single2:" + single2);
                         System.out.println("flag1 : " + flag1 + "  flag2:" + flag2);
                         System.out.println("circleTrace[" + end1 + "][" + end2 + "]: " + choice);
                         System.out.println("choice: " + choice);
                         System.exit(0);
                     }
                 }
             }  // end of both pairs
         } while( start1 <= end1 && start2 <= end2);
     }



     //-------------------------------------------------------------------------------
     // This function format a fine structure alignment based on the information from
     // the alignment maps plus the two RNA secondary structures. A matcher
     // object is optional, which could be used to shown the detailed information on
     // the aligned columns: idenical, mismatch( positive/negative ).
     // ------------------------------------------------------------------------------
     public static String alignMatch(RNA aRNA, int[] aMap, RNA bRNA, int[] bMap)
     {
         return alignMatch(aRNA, aMap, bRNA, bMap, null);
     }

     public static String alignMatch(RNA aRNA, int[] aMap, RNA bRNA, int[] bMap, Matcher m)
     {
         StringBuffer result = null;
         String name1 = aRNA.getName();
         String name2 = bRNA.getName();

         int pos1, posEnd1;      // starting/ending column at structure A;
         int pos2, posEnd2;      // starting/ending column at structure B;

         int[] region = HomoUtil.getAlignedRegion(aRNA, aMap);
	 pos1 = region[0];
         posEnd1 = region[1];
         
         ////////////////////////checkers////////////////////////////////

                   // System.out.print("point 1 in RNA : pos1 = "+ pos1+"  posEnd1 = "+posEnd1  );
/////////////////////////////////////////////////////////////////////////
	
	region = HomoUtil.getAlignedRegion(bRNA, bMap);


         pos2 = region[0];
         posEnd2 = region[1];

         ////////////////////////checkers////////////////////////////////

                    //System.out.print("\npoint 2 in RNA : pos2= "+pos2+" posEnd2 = "+posEnd2 );
/////////////////////////////////////////////////////////////////////////
         
	if( pos1>=posEnd1 || pos2>=posEnd2)
             return null;  //new StringBuffer("\nVoid\n");
         else
             result = new StringBuffer();

         StringBuffer line1 = new StringBuffer();
         StringBuffer line2 = new StringBuffer();
         StringBuffer line3 = new StringBuffer();
         StringBuffer line4 = new StringBuffer();
         StringBuffer alignLine = new StringBuffer();

         int primAlign=0, secAlign=0;
         int primIden =0, secIden =0;
         int primGap=0,  secGap=0;
         int primMis=0,  secMis=0;
         int totalLen =0, alignLen = 0;

         boolean stop = false;

         ////////////////////////checkers////////////////////////////////

        // System.out.print("point 3 in RNA" );
/////////////////////////////////////////////////////////////////////////
         int test2=0;
	 while( stop != true) {
             
         int test3 =0;
         ////////////////////////checkers////////////////////////////////

            // System.oiut.print("point 4 in RNA times=  "+test2 );
            // test2++;
/////////////////////////////////////////////////////////////////////////
	     int n1 = name1.length();
             int n2 = name2.length();
             int prefix = 0;
             if( n1 >= n2)
                 prefix = n1;
             else
                 prefix = n2;

             line3.append(name1 + ": ");
             line4.append(name2 + ": ");

             if( prefix > n1) {
                 for(int k = 0; k < prefix - n1; k++)
                     line3.append(" ");
             }
             if( prefix > n2) {
                 for(int k=0; k< prefix - n2; k++)
                     line4.append(" ");
             }

             line3.append(" " + (pos1+1) + " ");
             line4.append(" " + (pos2+1) + " ");

             n1 = line3.length();
             n2 = line4.length();
             if( n1 >= n2)
                 prefix = n1;
             else
                 prefix = n2;
             if( prefix > n1) {
                 for(int k = 0; k < prefix - n1; k++)
                     line3.append(" ");
             }
             if( prefix > n2) {
                 for(int k=0; k< prefix - n2; k++)
                     line4.append(" ");
             }
             for(int i=0; i<prefix; i++) {
                 line1.append(" ");
                 line2.append(" ");
             }

             for(int i=0; i<line4.length(); i++)
                 alignLine.append(" ");

             // the real alignment is processed one column by column 
             // up to 45 columns as a block
             for(int j=0; j<200; j++) {
                
         ////////////////////////checkers////////////////////////////////

          //   System.out.print("\npoint 4 in RNA times=  "+test2 + " : for" + test3 );
            // test2++;
	    // test3++;	
	    // System.out.print("\n\n pos1 = "+pos1+" pos2= "+pos2);
/////////////////////////////////////////////////////////////////////////
		 if( aMap[pos1] == pos2 + 1){  // matched
                     totalLen ++;
                     alignLen ++;

                     if( aRNA.getBase(pos1).equals("n") == true)
                         line3.append("n");
                     else
                         line3.append(aRNA.getBase(pos1));

                     if( bRNA.getBase(pos2).equals("n") == true)
                         line4.append("n");
                     else
                         line4.append(bRNA.getBase(pos2));

                     if( aRNA.getPairedBase(pos1) == -1){ // this column is a single base column
                         primAlign ++;
                         line1.append(".");
                         line2.append(".");

                         if( m == null) {
                             alignLine.append("|");
                             if( aRNA.getBase(pos1).equals(bRNA.getBase(pos2)) )
                                 primIden ++;
                             else
                                 primMis ++;
                         } else {
                             double score = m.baseMatch(aRNA.getBase(pos1), bRNA.getBase(pos2));
                             if( score < 0){
                                 alignLine.append(" ");
                                 primMis ++;
                             } else if( score == 0){
                                 alignLine.append(".");
                                 primMis ++;
                             } else if( score < m.getMaxSingle()){
                                 alignLine.append(":");
                                 primMis ++;
                             } else{
                                 alignLine.append("|");
                                 primIden ++;
                             }
                         }
                     } else {  // this column is involving in a base pair.
                         if(aRNA.getPairedBase(pos1) > pos1){  // 5'-end of the pair
                             line1.append("(");
                             line2.append("(");
                             secAlign ++;

                             String bond1 = aRNA.getBase(pos1) + aRNA.getBase(aRNA.getPairedBase(pos1));
                             String bond2 = bRNA.getBase(pos2) + bRNA.getBase(bRNA.getPairedBase(pos2));

                             if( m == null) {
                                 alignLine.append("|");
                                 if( aRNA.getBase(pos1).equals(bRNA.getBase(pos2)) )
                                     secIden ++;
                                 else
                                     secMis ++;
                             } else{
                                 double score = m.baseMatch(bond1, bond2);
                                 if( score < 0){
                                     alignLine.append(" ");
                                     secMis ++;
                                 } else if( score <=0){
                                     alignLine.append(".");
                                     secMis ++;
                                 } else if( score < m.getMaxPair()){
                                     alignLine.append(":");
                                     secMis ++;
                                 } else{
                                     alignLine.append("|");
                                     secIden ++;
                                 }
                             }
                         } else {  // 3'-end of the pair
                             line1.append(")");
                             line2.append(")");
                             secAlign ++;

                             String bond1 = aRNA.getBase(aRNA.getPairedBase(pos1)) + aRNA.getBase(pos1);
                             String bond2 = bRNA.getBase(bRNA.getPairedBase(pos2)) + bRNA.getBase(pos2);

                             if( m == null) {
                                 alignLine.append("|");
                                 if( aRNA.getBase(pos1).equals(bRNA.getBase(pos2)) )
                                     secIden ++;
                                 else
                                     secMis ++;
                             } else {
                                 double score = m.baseMatch(bond1, bond2);
                                 if( score < 0){
                                     alignLine.append(" ");
                                     secMis ++;
                                 } else if( score <=0){
                                     alignLine.append(".");
                                     secMis ++;
                                 } else if( score < m.getMaxPair()){
                                     alignLine.append(":");
                                     secMis ++;
                                 } else{
                                     alignLine.append("|");
                                     secIden ++;
                                 }
                             }
                         } // end of 3'
                     } // end of base pair column

                     pos1++;
                     pos2++;
                 } else if( aMap[pos1]==-1 && bMap[pos2]==-1){ // two gaps at the two consecutive columns
                     totalLen += 1;

                     if( aRNA.getBase(pos1).equals("n") == true)
                         pos1 ++;
                     else if( bRNA.getBase(pos2).equals("n") == true)
                         pos2 ++;
                     else {
                         // decide the structure at current position for 'aRNA'
                         if( aRNA.getPairedBase(pos1) == -1) {
                             line1.append(".");
                             line2.append(" ");
                             primAlign ++;
                         } else {
                             if( aRNA.getPairedBase(pos1) > pos1) {
                                 line1.append("(");
                                 line2.append(" ");
                             } else {
                                 line1.append(")");
                                 line2.append(" ");
                             }
                             secAlign ++;
                         }

                         alignLine.append(" ");
                         line3.append(aRNA.getBase(pos1));
                         line4.append("-");
                         pos1 ++;
                     }
                 } else if(pos1<=pos2){/////////////////////////////////NEW//////////////////////////////////////
			if( aMap[pos1] > pos2 + 1){  // gap at aRNA
                     totalLen ++;

                     if( bRNA.getBase(pos2).equals("n") == true)
                         pos2 ++;
                     else {
                         line1.append(" ");
                         if( bRNA.getPairedBase(pos2) == -1)
                         {
                             line2.append(".");
                             primGap ++;
                         } else {
                             if( bRNA.getPairedBase(pos2) > pos2)
                                 line2.append("(");
                             else
                                 line2.append(")");
 
                             secGap ++;
                         }

                         line3.append("-");
                         line4.append(bRNA.getBase(pos2));
                         alignLine.append(" ");
                         pos2++;
                     }
                    } else if( bMap[pos2] > pos1 + 1){ // gap at bRNA
                     totalLen ++;

                     if( aRNA.getBase(pos1).equals("n") == true)
                         pos1 ++;
                     else {
                         if( aRNA.getPairedBase(pos1) == -1)
                             line1.append(".");
                         else {
                             if( aRNA.getPairedBase(pos1) > pos1)
                                 line1.append("(");
                             else
                                 line1.append(")");
                         }
                         line2.append(" ");
                         line3.append(aRNA.getBase(pos1));
                         line4.append("-");
                         alignLine.append(" ");
                         pos1++;
                     }
                   }
		}else if (pos1>pos2){/////////////////////////////NEW//////////////////////////////////
                
                    if( bMap[pos2] > pos1 + 1){ // gap at bRNA
                     totalLen ++;

                     if( aRNA.getBase(pos1).equals("n") == true)
                         pos1 ++;
                     else {
                         if( aRNA.getPairedBase(pos1) == -1)
                             line1.append(".");
                         else {
                             if( aRNA.getPairedBase(pos1) > pos1)
                                 line1.append("(");
                             else
                                 line1.append(")");
                         }
                         line2.append(" ");
                         line3.append(aRNA.getBase(pos1));
                         line4.append("-");
                         alignLine.append(" ");
                         pos1++;
                     }
                   }
		   else	if( aMap[pos1] > pos2 + 1){  // gap at aRNA
                     totalLen ++;

                     if( bRNA.getBase(pos2).equals("n") == true)
                         pos2 ++;
                     else {
                         line1.append(" ");
                         if( bRNA.getPairedBase(pos2) == -1)
                         {
                             line2.append(".");
                             primGap ++;
                         } else {
                             if( bRNA.getPairedBase(pos2) > pos2)
                                 line2.append("(");
                             else
                                 line2.append(")");
 
                             secGap ++;
                         }

                         line3.append("-");
                         line4.append(bRNA.getBase(pos2));
                         alignLine.append(" ");
                         pos2++;
                     }
		} 
	     }	
	     else{//none of the cases are getting satisfied
			System.out.print("\n\nNone of the cases are getting satisfied!");
			System.out.print("\npos1= "+pos1+" pos2 = "+pos2);
			System.out.print("\naMap[pos1] = "+aMap[pos1]);
                        System.out.print("\nbMap[pos2] = "+bMap[pos2]);
			pos1=posEnd1;
			
		 }
            if(pos1 > posEnd1 || pos2 > posEnd2)
              break;
             }

             line1.append("\n");
             line2.append("\n");
             alignLine.append("\n");
             line3.append(" " + pos1 + "\n");
             line4.append(" " + pos2 + "\n");

             result.append(line1);
             result.append(line2);
             result.append(line3);
             result.append(alignLine);
             result.append(line4);
             result.append("\n");

             line1.delete(0, line1.length());
             line2.delete(0, line2.length());
             alignLine.delete(0, alignLine.length());
             line3.delete(0, line3.length());
             line4.delete(0, line4.length());

             if( pos1 > posEnd1 || pos2 > posEnd2)
                stop = true;
	//	break;
        }

        String header = "   Query: " + (primAlign+secAlign) + " (ss:" + primAlign + ",ds:" + secAlign + ")\n" +
                 "Identity: str: " + (100*alignLen/totalLen) + "%; seq:" +
                 (((primAlign+secAlign)!=0)?(100*(primIden+secIden)/(primAlign+secAlign)):0) + "% (ss:" + 
                 (primAlign!=0?100*primIden/primAlign:0) + "%, ds:" + (secAlign!=0?100*secIden/secAlign:0) + "%)\n"
                 + "Gap: " + (primGap+secGap) + " (ss:" + primGap + ", ds:" + secGap + ")  " +
                 "Mismatch: " + (primMis+secMis) + " (ss:" + primMis + ", ds:" + secMis + ")\n";

        return new String(header + result);
    }
}
