/**********************************************************************
 *
 *       (c) Copyright 2003
 *       All rights reserved
 *       Programs written by Jianghui Liu (NJIT)
 *
 *       Permission to use, copy, modify, and distribute this software
 *       and its documentation for any purpose and without fee is her-
 *       eby granted, provided that this copyright notice appears in
 *       all copies.   Programmer(s) makes no representations about
 *       the suitability of this software for any purpose.  It is pro-
 *       vided "as is" without express or implied warranty.
 *
 *       08/06/2003
 *
 **********************************************************************/
package RNA;
import java.io.*;
import java.util.*;
import java.text.NumberFormat;

public class HomoAligner 
{
    private String HomoID1, HomoID2;   // a pair of homologous RNAs;
    private String HomoDB1, HomoDB2;   // two structures for each specis;
    private String MatrixFile;         // file containing scoreing matrices;
    private int Penalty;               // atom gap penalty;

    private RNA[] HomoStruct1 = null;    // aligned structural region A;
    private RNA[] HomoStruct2 = null;    // aligned structural region B;
    private int AlignedStart1, AlignedEnd1;   // region A information;
    private int AlignedStart2, AlignedEnd2;   // region B information;

    public HomoAligner(String id1, String db1, String id2, String db2, String scoreFile, int p)
    {
        HomoID1 = id1;
        HomoDB1 = db1;
        HomoID2 = id2;
        HomoDB2 = db2;
        MatrixFile = scoreFile;
        Penalty = p;
    }

    public static void main(String[] args)
    {
        if( args.length == 0 || args[0].equalsIgnoreCase("-h") || args[0].equalsIgnoreCase("-help"))
        {
            System.out.println("Usage: ");
            System.out.println("   java HomoAligner -db1 database1 -ho1 homoID1");
            System.out.println("   [-db2 database2] -ho2 homoID2 ");
            System.out.println("   [-S|s scorematrix_file] [-O|o output_file] [-P|p gap_penalty] ");
            System.out.println("   Default Settings: ");
            System.out.println("   Output File (Default): junk.out");
            System.out.println("   ScoreMatrices File (Default): scoreMat.structure");
            System.out.println("   Gap penalty (Default): -6");
            System.exit(0);
        }

        String strDB1=null, strID1=null, strDB2=null, strID2=null;
        String Score_matrices = "scoreMat.structure";
        int Penalty = -6;
        String Output_file = "junk.out";

        for(int i=0; i<args.length; i++)
        {
            if( args[i].equalsIgnoreCase("-db1"))
            {
                strDB1 = new String(args[i+1]);
                i++;
            }
            else if( args[i].equalsIgnoreCase("-db2"))
            {
                strDB2 = args[i+1];
                i++;
            }
            else if( args[i].equalsIgnoreCase("-ho1"))
            {
                strID1 = args[i+1];
                i++;
            }
            else if( args[i].equalsIgnoreCase("-ho2"))
            {
                strID2 = args[i+1];
                i++;
            }
            else if( args[i].equalsIgnoreCase("-p"))
            {
                Penalty = Integer.parseInt(args[i+1]);
                i++;
            }
            else if( args[i].equalsIgnoreCase("-s"))
            {
                Score_matrices = args[i+1];
                i++;
            }
            else if( args[i].equalsIgnoreCase("-o"))
            {
                Output_file = args[i+1];
                i++;
            }
            else
            {
                System.out.println("Illegal parameter " + args[i]);
                System.exit(0);
            }
        }

        if( strID1 == null || strID2 == null || strDB1 == null)
        {
            System.out.println(" parameters not set up properly!");
            System.out.println("Try '-h' for help");
            System.exit(0);
        }
        if( strDB2 == null)
            strDB2 = strDB1;

        System.out.println("\n\nYour Settings are:");
        System.out.println("First homology: " + strID1 + " from: " + strDB1);
        System.out.println("Second homology: " + strID2 + " from: " + strDB2);
        System.out.println("Score matrices File: " + Score_matrices);
        System.out.println("Gap Penalty: " + Penalty);
        System.out.println("Output File: " + Output_file + "\n");

        HomoAligner aligner = new HomoAligner(strID1, strDB1, strID2, strDB2, Score_matrices, Penalty);

        //System.out.println(aligner.getMultiAlignment().getStoAlignment());
        //RNA[] homo = HomoUtil.findAllLocality(strDB1, strID1, strDB2, strID2);

        try {
            PrintWriter output = null;
            String result =  aligner.dispHomoAlignment(5) ;
            output = new PrintWriter(new BufferedWriter(new FileWriter(Output_file)));
            //output.println(aligner.getMultiAlignment().getStoAlignment());
            output.println(result);
            output.close();
        } catch( IOException e) {
            e.printStackTrace();
            System.exit(1);
        }
    }

    //------------------------------------------------------------------------
    // This is the kernal of homologous searching: given two homologous
    // RNAs, the optimal regions for structural alignment are determined
    // where each region is contained by one structure. Cross-database
    // searching is then conducted using the above two structures to retrieve 
    // top ranked candidates from each specis database. The ultimate
    // multi-alignment is then constructed.
    //------------------------------------------------------------------------
    public MultiAlignment getMultiAlignment()
    {
        if( HomoStruct1 == null || HomoStruct2 == null ){
            RNA[] homo = HomoUtil.getMotifLocality(HomoDB1, HomoID1, HomoDB2, HomoID2);
            if( homo == null ) {
                System.out.println("Fatal Error: Homologous region cannot be found!");
                System.exit(1);
            }

            int n = homo.length / 2;
            HomoStruct1 = new RNA[n];
            HomoStruct2 = new RNA[n];
            for(int i=0; i<n; i++){
                HomoStruct1[i] = homo[2*i];
                HomoStruct2[i] = homo[2*i+1];
            }
         }

        // perform cross-searching against the databases respectively:
        // HomoStruct1 ---> HomoDB2;  HomoStruct2 --> HomoDB1;
        System.out.println("\n Searching against database: " + HomoDB2);
        DBSearcher s1 = new DBSearcher(HomoStruct1[0]);
        RNA[] TopSet1 = null, TopSet2 = null;
        TopSet1 = s1.getTopN(HomoDB2, 50, MatrixFile, Penalty);   // select top 50!

        System.out.println("\n Searchng against database: " + HomoDB1);
        DBSearcher s2 = new DBSearcher(HomoStruct2[0]);
        TopSet2 = s2.getTopN(HomoDB1, 50, MatrixFile, Penalty);

        // initialize the multialignment by setting it up from the two homologies
        String codeTableFile = "codeTable.properties";
        Matcher m = new Matcher(codeTableFile, MatrixFile);
        int[] MapStruct1 = new int[HomoStruct1[0].seqLength()];
        int[] MapStruct2 = new int[HomoStruct2[0].seqLength()];
        for(int k=0; k<MapStruct1.length; k++)
            MapStruct1[k] = -1;
        for(int k=0; k<MapStruct2.length; k++)
            MapStruct2[k] = -1;
        RNA.localMatch2D(HomoStruct1[0], HomoStruct2[0], MapStruct1, MapStruct2, m, Penalty);
        int[] seg = HomoUtil.getAlignedRegion(HomoStruct1[0], MapStruct1);
        AlignedStart1 = seg[0];
        AlignedEnd1 = seg[1];
        seg = HomoUtil.getAlignedRegion(HomoStruct2[0], MapStruct2);
        AlignedStart2 = seg[0];
        AlignedEnd2 = seg[1];

        //System.out.println("first: " + AlignedStart1 + "<-->" + AlignedEnd1 + 
        //                   " second: " + AlignedStart2 + "<-->" + AlignedEnd2);

        MultiAlignment alignment = new MultiAlignment(HomoStruct1[0], MapStruct1, HomoStruct2[0], MapStruct2);

        //  processing the first set of RNAs
        System.out.println("\nProcessing the first set of top ranks with " + TopSet1.length + " items ...");
        for(int i=0; i<TopSet1.length; i++)
        {
            if( TopSet1[i] == null)
                continue;

            int[] dataMap = new int[TopSet1[i].seqLength()];
            for(int k=0; k<dataMap.length; k++)
                dataMap[k] = -1;
            for(int k=0; k<MapStruct1.length; k++)
                MapStruct1[k] = -1;
            RNA.localMatch2D(HomoStruct1[0], TopSet1[i], MapStruct1, dataMap, m, Penalty);

            int[] region = HomoUtil.getAlignedRegion(HomoStruct1[0], MapStruct1);
            int overlapStart = 0, overlapEnd = 0;
            if( region[0] > AlignedStart1)
                overlapStart = region[0];
            else
                overlapStart = AlignedStart1;

            if( region[1] < AlignedEnd1)
                overlapEnd = region[1];
            else
                overlapEnd = AlignedEnd1;

            int threshold = 75;   // 75% overlapping
            if( ((overlapEnd-overlapStart+1)*100.0/(AlignedEnd1-AlignedStart1+1)) >= threshold )
            {
                System.out.println("merging " + TopSet1[i].getName());
                MultiAlignment anAlignment = new MultiAlignment(HomoStruct1[0], MapStruct1, TopSet1[i], dataMap);
                alignment.merge(anAlignment, MatrixFile, Penalty);
            }
            else {
               System.out.println("overlap length: " + (overlapEnd-overlapStart+1));
               System.out.println("the real length: " + (AlignedEnd1-AlignedStart1+1));
               System.out.println("throwing away " + TopSet1[i]);
            }
        } 

        // expand the multi-alignment by merging the second set of RNAs
        System.out.println("\nProcessing the second set of top ranks ... ");
        for(int i=0; i<TopSet2.length; i++)
        {
            if( TopSet2[i] == null)
                continue;

            int[] dataMap = new int[TopSet2[i].seqLength()];
            for(int k=0; k<dataMap.length; k++)
                dataMap[k] = -1;
            for(int k=0; k<MapStruct2.length; k++)
                MapStruct2[k] = -1;
            RNA.localMatch2D(HomoStruct2[0], TopSet2[i], MapStruct2, dataMap, m, Penalty);

            int[] region = HomoUtil.getAlignedRegion(HomoStruct2[0], MapStruct2);
            int overlapStart = 0, overlapEnd = 0;
            if( region[0] > AlignedStart2)
                overlapStart = region[0];
            else
                overlapStart = AlignedStart2;

            if( region[1] < AlignedEnd2)
                overlapEnd = region[1];
            else
                overlapEnd = AlignedEnd2;

            int threshold = 75;   // 75% overlaped region
            if( (overlapEnd - overlapStart+1) * 100.0 / (AlignedEnd2 - AlignedStart2+1) >= threshold )
            {
                System.out.println("merging " + TopSet2[i].getName());
                MultiAlignment anAlignment = new MultiAlignment(HomoStruct2[0], MapStruct2, TopSet2[i], dataMap);
                System.out.println("prepared the alignment");
                alignment.merge(anAlignment, MatrixFile, Penalty);
                System.out.println("merged with result:");
            }
            else {
               System.out.println("overlap length: " + (overlapEnd-overlapStart+1));
               System.out.println("the real length: " + (AlignedEnd2-AlignedStart2+1));
               System.out.println("throwing away " + TopSet2[i].getName());
            }
        } 
        System.out.println(" Done ! ");
        return alignment;
    }

    //-------------------------------------------------------
    // print out the alignment detail of homologous regions
    //-------------------------------------------------------
    public String dispHomoAlignment(int topN)
    {
        int total = 0;
        if( HomoStruct1 == null || HomoStruct2 == null){
            RNA[] homo = HomoUtil.getMotifLocality(HomoDB1, HomoID1, HomoDB2, HomoID2);
            total = homo.length / 2;
            HomoStruct1 = new RNA[total];
            HomoStruct2 = new RNA[total];
            for(int i=0; i<total; i++){
                HomoStruct1[i] = homo[2*i];
                HomoStruct2[i] = homo[2*i+1];
            }
        } else {
            total = HomoStruct1.length;
        }

        StringBuffer result = new StringBuffer();
        String codeTableFile = "codeTable.properties";
        Matcher m = new Matcher(codeTableFile, MatrixFile);

        int count = 0;
        for(int j=0; j<total; j++){
            if( HomoStruct1[j] == null || HomoStruct2[j] == null ) {
                continue;
            }

            int[] aMap = new int[HomoStruct1[j].seqLength()];
            for(int i=0; i<aMap.length; i++)
                aMap[i] = -1;

            int[] bMap = new int[HomoStruct2[j].seqLength()];
            for(int i=0; i<bMap.length; i++)
                bMap[i] = -1;

            RNA.localMatch2D(HomoStruct1[j], HomoStruct2[j], aMap, bMap, m, Penalty);
            result.append( RNA.alignMatch(HomoStruct1[j], aMap, HomoStruct2[j], bMap, m) );
            result.append("\n\n");

            count ++;
            System.out.println("Got " + count + " pair(s) of motifs.");
            if( count >= topN )
                break;
        }
        return result.toString();
    }
}
