/**************************************************************************
 *                                                                        *
 *       (c) Copyright 2003                                               *
 *       All rights reserved                                              *
 *       Programs written by Jianghui Liu (NJIT)                          *
 *                                                                        *
 *       Permission to use, copy, modify, and distribute this software    *
 *       and its documentation for any purpose and without fee is her-    *
 *       eby granted, provided that this copyright notice appears in      *
 *       all copies.   Programmer(s) makes no representations about       *
 *       the suitability of this software for any purpose.  It is pro-    *
 *       vided "as is" without express or implied warranty.               *
 *                                                                        *
 *       08/06/2003                                                       *
 **************************************************************************/
package RNA;
import java.util.*;
import java.text.NumberFormat;

public class ProfileMatcher
{
    private double[][] profileMatrix;    // position-dependant scoring matrix
    private RNA        profileStruct;    // the corresponding profile secondary structure

    private Hashtable  SScode;           // possible nucleotide at SS region
    private Hashtable  DScode;           // possible nucleotide-pair at DS region
    private final int  SScodeSize = 5;   // the size of 'SScode': four(4) nucleotides plus gap
    private final int  DScodeSize = 17;  // the size of 'DScode': sixteen(16) pairs plus gap

    //-----------------------------------------------------------------------------
    // The scoring marix shall be created from a structure-oriented multi-alignment
    //-----------------------------------------------------------------------------
    public ProfileMatcher(MultiAlignment multi)
    {
        setupCode();

        profileStruct = multi.getProfileRNA();
        int nComponent = profileStruct.getNumberOfComponent();
        int nMember = multi.getRow();

        profileMatrix = new double[nComponent][];
        for(int i=0; i<nComponent; i++) {
            int[] pos = profileStruct.getComponentPos(i);
            if( pos.length == 1 ){   // this component is in SS region
                profileMatrix[i] = new double[SScodeSize];
                for(int j=0; j<profileMatrix[i].length; j++)
                    profileMatrix[i][j] = 0;

                for(int j=0; j<nMember; j++) {
                    String base = multi.getBase(j, pos[0]);
                    String indexStr = (String)SScode.get(base);

                    if( indexStr == null) {
                        System.out.println("\nFatal Error when constructing scoring matrix!");
                        System.out.println("Illegal code '" + base + "' appears in " + (j+1) + 
                                           "th member of the multi-alignment!\n");
                        System.exit(0);
                    }

                    int index = Integer.parseInt(indexStr);
                    profileMatrix[i][index] ++;
                }
            } else if( pos.length == 2){  // this column is of double strand
                profileMatrix[i] = new double[DScodeSize];
                for(int j=0; j<profileMatrix[i].length-1; j++)
                    profileMatrix[i][j] = 0.1;

                profileMatrix[i][profileMatrix[i].length-1] = 0;   // this corresponds to gap

                for(int j=0; j<nMember; j++) {
                    String base1 = multi.getBase(j, pos[0]);
                    String base2 = multi.getBase(j, pos[1]);

                    String indexStr = (String)DScode.get(base1+base2);
                    if( indexStr == null || indexStr.equals("null") == true) {
                        System.out.println("\nFatal Error when constructing scoring matrix!");
                        System.out.println("Illegal code '" + base1 + "-" + base2 + "' appears in " + (j+1) + 
                                           "th member of the multi-alignment!\n");
                        System.exit(0);
                    }

                    int index = Integer.parseInt(indexStr);
                    profileMatrix[i][index] ++;
                }
            }
        }

        // get the frequency for each component position.
        for(int i=0; i<nComponent; i++) {
            for(int j=0; j<profileMatrix[i].length; j++) {
                profileMatrix[i][j] = (profileMatrix[i][j] / nMember);
                if( profileMatrix[i][j] <= 0 ) {
                    if( profileMatrix[i].length == SScodeSize )
                        profileMatrix[i][j] = -1;
                    else
                        profileMatrix[i][j] = -2;
                }
            }
        }
    }

    //------------------------------------------------
    // Foramt the score matrix for the output
    //------------------------------------------------
    public String dispMatrix()
    {
        String[] keys = new String[22];
        keys[0] = "A";
        keys[1] = "C";
        keys[2] = "G";
        keys[3] = "U";
        keys[4] = "-";
        keys[5] = "AA";
        keys[6] = "AC";
        keys[7] = "AG";
        keys[8] = "AU";
        keys[9] = "CA";
        keys[10]= "CC";
        keys[11]= "CG";
        keys[12]= "CU";
        keys[13]= "GA";
        keys[14]= "GC";
        keys[15]= "GG";
        keys[16]= "GU";
        keys[17]= "UA";
        keys[18]= "UC";
        keys[19]= "UG";
        keys[20]= "UU";
        keys[21]= "--";

        StringBuffer header = new StringBuffer();
        header.append("  ");
        StringBuffer[] lines = new StringBuffer[keys.length];
        for(int i=0; i<lines.length; i++) {
            lines[i] = new StringBuffer();
            lines[i].append(fixedWidth(keys[i],2));
        }

        RNA temple = getProfileRNA();
        NumberFormat nf = NumberFormat.getInstance();
        nf.setMaximumFractionDigits(2);
        for(int i=0; i<profileMatrix.length; i++) {
            header.append(fixedWidth(temple.getPosBond(i), 6));
            if( profileMatrix[i].length == SScodeSize ) {
                for(int k=0; k<5; k++)
                    lines[k].append(fixedWidth(nf.format(profileMatrix[i][k]), 6));
                for(int k=5; k<keys.length; k++)
                    lines[k].append(fixedWidth("-", 6));
            } else {
                for(int k=0; k<5; k++)
                   lines[k].append(fixedWidth("-", 6));
                for(int k=5; k<keys.length; k++)
                    lines[k].append(fixedWidth(nf.format(profileMatrix[i][k-5]), 6));
            }
        }

        StringBuffer ret = new StringBuffer();
        ret.append(header + "\n");
        for(int i=0; i<lines.length; i++)
            ret.append(lines[i] + "\n");

        return new String(ret);
    }
                

    public RNA getProfileRNA()
    {
        return profileStruct;
    }

    //------------------------------------------------------------
    // Given a component and the position, the score is returned
    // when the component is placed at that position.
    //------------------------------------------------------------
    public double getScore(String element, int position)
    {
        String str = null;
        int[] pos = profileStruct.getComponentPos(position);
        if( pos.length == 1) 
            str = (String)SScode.get(element);
        else if( pos.length == 2)
            str = (String)DScode.get(element);

        if( str == null) {
            //System.out.println("Error in ProfileMatcher: '" + element + "'" + " is not a valid component!\n");
            //System.out.println("score of 0 (zero) is used instead!");
            return 0;
        }

        int index = Integer.parseInt(str);
        return profileMatrix[position][index];
    }

    private void setupCode()
    {
        SScode = new Hashtable();
        SScode.put("A", "0");
        SScode.put("C", "1");
        SScode.put("G", "2");
        SScode.put("T", "3");
        SScode.put("U", "3");
        SScode.put("-", "4");

        DScode = new Hashtable();
        DScode.put("AA", "0");
        DScode.put("AC", "1");
        DScode.put("AG", "2");
        DScode.put("AU", "3");
        DScode.put("AT", "3");
        DScode.put("CA", "4");
        DScode.put("CC", "5");
        DScode.put("CG", "6");
        DScode.put("CU", "7");
        DScode.put("CT", "7");
        DScode.put("GA", "8");
        DScode.put("GC", "9");
        DScode.put("GG", "10");
        DScode.put("GU", "11");
        DScode.put("GT", "11");
        DScode.put("UA", "12");
        DScode.put("TA", "12");
        DScode.put("UC", "13");
        DScode.put("TC", "13");
        DScode.put("UG", "14");
        DScode.put("TG", "14");
        DScode.put("UU", "15");
        DScode.put("TT", "15");
        DScode.put("--", "16");
        DScode.put("-", "16");
    }

    //--------------------------------------------------------
    // String 'str' will be formated in a fixed width 'width'
    //--------------------------------------------------------
    private String fixedWidth(String str, int width)
    {
        StringBuffer ret = new StringBuffer();
        int length = str.length();
        if( length > width)
            length = width;
        ret.append(str.substring(0, length));

        for(int i=0; i<width-length; i++)
            ret.insert(0, " ");
        return ret.toString();
    }
}
