#ifndef __ABSTRACT_MATRIX_WRITER__
#define __ABSTRACT_MATRIX_WRITER__

#include <Vec.h>
#include <Vector3D.h>
#include <MatrixWriterBase.h>
#include <ResourceBundle.h>

using namespace std;

class AbstractMatrixWriter : public MatrixWriterBase {

 public:

  AbstractMatrixWriter();

  virtual ~AbstractMatrixWriter() { }

  virtual double getFrequency(const string& s1, char c) const;

  virtual double getFrequency(const string& s1, const string& s2, const string& pair) const;

  /** returns limit for columns to be drawn */
  virtual double getInfLimit() const { return infLimit; }

  /** returns limit for columns to be drawn */
  virtual double getInfLimitRelative() const { return infLimitRelative; }

  virtual Vec<double> getFrequencies(const string& s1, const string& alphabet) const;

  virtual int getLogoMode() const { return logoMode; }

  virtual Vec<Vector3D> getPairColors(const Vec<string>& pairs) const;

  virtual Vec<Vector3D> getPairColors2(const Vec<string>& pairsOrig) const;

  /** use parameter file stored in ResourceBundle as lookup table for colors */
  virtual Vec<Vector3D> getPairColors3(const Vec<string>& pairs) const;

  /** use rainbow encoding */
  virtual Vec<Vector3D> getPairColors4(const Vec<string>& pairs, double mutInf) const;

  /** another attempt at a good palette for 16 base pairs
      red: complementary, green: AA, CC etc, blue: rest 
  */
  virtual Vec<Vector3D> getPairColors5(const Vec<string>& pairsOrig) const;

  /** another attempt at a good palette for 16 base pairs
   *  red: complementary, yellow: GU,  green: AA, CC etc, blue: rest 
   */
  virtual Vec<Vector3D> getPairColorsProtein1(const Vec<string>& pairsOrig) const;

  virtual Vector3D getSingleColor(char letter) const;

  virtual Vec<Vector3D> getSingleColors(const string& letters) const;

  /** returns true if character is a gap character. 
   * TODO : use CharacterAlphabet class instead.
   */
  virtual bool isGap(char c) const {
    return (c == '.') || (c == '-');
  }

  /** returns color mode */
  virtual int getColorMode() const { return colorMode; }

  /** returns potential upper limit ("cap") for mutual information stacks */
  virtual double getInfUpperLimit() const { return infUpperLimit; }

  virtual int getLetterMode() const { return letterMode; }

  virtual int getLetterMode2() const { return letterMode2; }

  virtual void setErrorMode(int mode) { 
    errorMode = mode;
    cout << "Setting errorMode to: " << mode << endl;
  }

  virtual void setLetterMode(int n) { letterMode = n; }

  virtual void setLetterMode2(int n) { letterMode2 = n; }

  /** sets parameters in string form */
  virtual void setResourceBundle(const ResourceBundle& rb) {
    resources = rb;
    string s = rb.getString("wwwImageBase");
    if (s.size() > 0) {
      wwwImageBase = s;
    }
  }

  /** sets algorithm for scorer of upper matrix triangle */
  virtual void setAlphabet(string s) { 
    alphabet = s;
  }

  virtual void setArrowMode(int m) { arrowMode = m; }

  /** sets mode for coloring matrix elements */
  virtual void setColorMode(int mode) { colorMode = mode; }

  virtual void setGapMode(int mode) { gapMode = mode; }

  /** only write columns corresponding to values greater than infLimit */
  virtual void setInfLimit(double x) { infLimit = x; }

  /** only write columns corresponding to values greater than infLimitRelative standard deviations */
  virtual void setInfLimitRelative(double x) { infLimitRelative = x; }

  virtual void setInfUpperLimit(double x) { infUpperLimit = x; }

  virtual void setLogoMode(int n) { logoMode = n; }

  virtual void setProteinMode(bool b) { proteinMode = b; }

  /** returns URL base for texture mapping images */
  virtual const string& getWwwImageBase() const { return wwwImageBase; }  

  /** sets URL base for texture mapping images */
  virtual void setWwwImageBase(const string& s) {
    wwwImageBase = s;
  }

  /** returns URL of image for letters like "C" or "A". Remeber to have slash at end of directory names! */
  virtual string getImageUrl(char c) const {
    string empty;
    if ((wwwImageBase.size() == 0) || (c == ' ')) {
      return empty; // return empty string
    }
    return wwwImageBase + c + ".jpg";
  }

  virtual double getStretchZ() { return stretchZ; }

  /** if true, writes ground plate */
  virtual void setGroundPlateColor(Vector3D c) { groundPlateColor = c; }

  /** if true, writes ground plate */
  virtual void setGroundPlateMode(bool b) { groundPlateMode = b; }

  virtual void setStretchZ(double x) { stretchZ = x; }

 protected:

  void
  writeLogo(ostream& os,
	    const Alignment& ali,
	    double inf,
	    double infError,
	    int pos) const;

  void
  writeLogoFlat(ostream& os,
		const Alignment& ali,
		double inf,
		double infError,
		int pos) const;

  void writeLogos(ostream& os, const Alignment& ali,
		  const Vec<double>& infVec,
		  const Vec<double>& infErrorVec) const;

  void writeLogosFlat(ostream& os, const Alignment& ali,
		      const Vec<double>& infVec,
		      const Vec<double>& infErrorVec) const;

  
  virtual bool writePosStack(ostream& os,
			     const Alignment& ali,
			     const Vec<Stem>& refStems,
			     unsigned int i,
			     unsigned int j,
			     double mutInf,
			     double mutInfError,
			     int letterStyle) const;

  virtual void copy(const AbstractMatrixWriter& other);

  virtual void finalize(ostream& os) const = 0;

  /** calls getPairColorsX according to colorMode */
  virtual Vec<Vector3D> getPairColorsMode(const Vec<string>& pairs, double mutInf) const;

  virtual Vec<double> getPairFrequencies(const string& s1, const string& s2, const Vec<string>& pairs) const;

  /** returns all possible 16 pairs */
  virtual Vec<string> getPairs() const;

  /** how many draw commands were issued so far? */
  virtual unsigned int getPrimitiveCounter() {
    return primitiveCounter;
  }

  bool getProteinMode() { return proteinMode; }

  Vector3D getResourceColor(const string& name) const;

  virtual int getVerboseLevel() const { return verboseLevel; }

  /** returns true if mutual information is both above the simple threshold
   * and if it is enough standard deviation above zero.
   */
  virtual bool isInformationAboveThreshold(double mutInf,
					   double mutInfError) const;


  virtual void setVerboseLevel(int d) { verboseLevel = d; }

  virtual void writeBox(ostream& os,
			const Vector3D& pos,
			double height, 
			const Vector3D& col ) const = 0;
  
  /** central method for writing single box */
  virtual void writeBox(ostream& os,
			const Vector3D& pos,
			const Vector3D& size, 
			const Vector3D& col,
			bool faceMode = true) const = 0;

  /** writes box with letter as set of lines.
      The string extra qualifies a potential different image. Defined so far: empty and "r" for 90 right rotated images */
  virtual void writeBoxLetterBoth(ostream& os,
				  const Vector3D& pos,
				  double height, 
				  const Vector3D& col,
				  char letterFront,
				  char letterLeft,
				  int letterStyle) const = 0;  

  virtual void writeBoxStackBoth(ostream& os,
				 const Vector3D& pos,
				 double height, 
				 double errorHeight,
				 const Vec<double>& frequencies,
				 const Vec<Vector3D>& colors,
				 const string& lettersFront,
				 const string& lettersLeft,
				 double localInfUpperLimit) const;

  virtual void writeBoxStackFlat(ostream& os,
				 const Vector3D& pos,
				 double height, 
				 double errorHeight,
				 const Vec<double>& frequencies,
				 const Vec<Vector3D>& colors,
				 const string& letters,
				 int faceMode,
				 int letterStyle) const = 0;
  
  virtual void writeBoxStackFlat2(ostream& os,
				  const Vector3D& pos,
				  double height, 
				  double errorHeight,
				  const Vec<double>& frequencies,
				  const Vec<Vector3D>& colors,
				  const string& letters,
   				  int faceMode, 
 				  int letterStyle) const = 0; 

  /** writes comment to XML file. Avoid "-" characters in comment!
      no comments within tag! 
      @see http://www.extropia.com/tutorials/xml/comments.html
  */
  virtual void writeComment(ostream& os, const string& comment) const = 0;

  /** writes box with letter as texture map.
      The string extra qualifies a potential different image. Defined so far: empty and "r" for 90 right rotated images */
  virtual void writeOpenBoxLetterBoth(ostream& os,
				      const Vector3D& pos,
				      double height, 
				      const Vector3D& col,
				      char letterFront,
				      char letterLeft,
				      int letterStyle) const = 0;

  /* ATTRIBUTES */

  string alphabet;

  int arrowMode;

  int colorMode;

  Vector3D errorColor;

  int errorMode;

  double frequencyLimit;

  int gapMode;

  double gridWidth;

  Vector3D groundPlateColor;

  bool groundPlateMode;

  double infLimit; // cutoff value for mutual information

  double infLimitRelative; // cutoff value: how many standard deviations does the mutual information have to be?

  double infUpperLimit;

  int legendMode;

  int letterMode;

  int letterMode2;

  int logoMode;

  mutable unsigned int primitiveCounter; // how many draw commands? 

  bool proteinMode;

  Vector3D refColor;

  double referenceGap; // how much "air" between regular stack and reference cube ?

  ResourceBundle resources;

  Vector3D stencilColor;

  double stretchZ;

  int verboseLevel;

  string wwwImageBase;  
};

#endif
