// --*- C++ -*------x---------------------------------------------------------
// $Id: Vrml2MatrixWriter.h,v 1.12 2005/09/22 18:01:57 bindewae Exp $
//
// Class:           VrmlMatrixWriter
// 
// Base class:      -
//
// Derived classes: - 
//
// Author:          Eckart Bindewald
//
// Project name:    -
//
// Date:            07/2005
//
// Description:     writes VRML 2.0 files
// 
// Reviewed by:     -
// -----------------x-------------------x-------------------x-----------------

#ifndef __VRML2_MATRIX_WRITER_H__
#define __VRML2_MATRIX_WRITER_H__

// Includes

#include <iostream>

#include <debug.h>
#include <AbstractMatrixWriter.h>
// #include <CompensationScorer.h>
#include <Stem.h>
#include <Vector3D.h>
#include <Alignment.h>
#include <Letter3D.h>

using namespace std;

/** @author Eckart Bindewald
    @see    Tom Schneider's 1986 and 1992 papers on mutual information
    @review - */
class Vrml2MatrixWriter : public AbstractMatrixWriter {

public:

  enum { LETTER_MODE_NONE = 0, LETTER_MODE_TEXT = 1, LETTER_MODE_LINES = 2, LETTER_MODE_TEXTURE = 3 };

  Vrml2MatrixWriter();

  Vrml2MatrixWriter(const Vrml2MatrixWriter& other);

  virtual ~Vrml2MatrixWriter();

  /* OPERATORS */

  /** Assigment operator. */
  Vrml2MatrixWriter& operator = (const Vrml2MatrixWriter& orig);

  friend ostream& operator << (ostream& os, const Vrml2MatrixWriter& rval);

  friend istream& operator >> (istream& is, Vrml2MatrixWriter& rval);

  /* PREDICATES */

  /** returns color mode */
  virtual int getColorMode() const { return colorMode; }

  /** returns limit for columns to be drawn */
  virtual double getInfLimit() const { return infLimit; }


  virtual const Vec<Letter3D>& getLetters3D() const { return letters3D; }

  virtual bool getReadingFrameMode() const { return readingFrameMode;  }

  /** returns URL base for texture mapping images */
  virtual const string& getWwwImageBase() const { return wwwImageBase; }

  /** writes comment to XML file. Avoid "-" characters in comment!
      no comments within tag! 
      @see http://www.extropia.com/tutorials/xml/comments.html
  */
  void writeComment(ostream& os, const string& comment) const {
    os << "# " << comment << endl; 
  }

  /** central method */
  
  /** central method */
  virtual void writeVrml(ostream& os,
			 const Alignment& ali,
			 const Vec<Vec<double> >& matrix,
			 const Vec<Vec<double> >& errorMatrix,
			 const Vec<double>& infVec,
			 const Vec<double>& infErrorVec,
			 const Vec<Stem>& referenceStems) const;


  /* MODIFIERS */

  virtual void copy(const Vrml2MatrixWriter& other);

  virtual void setLetters3D(const Vec<Letter3D>& other) { letters3D = other; }

  virtual void setReadingFrameMode(bool b) {
    readingFrameMode = b;
  }
  
  /** expect eight "RNA", "DNA" or "PROTEIN" */
  virtual void setSequenceType(const string& s) {
    sequenceType = s;
  }

  /** sets URL base for texture mapping images */
  virtual void setWwwImageBase(const string& s) {
    wwwImageBase = s;
  }

 private:

  /* PREDICATES */

  void computeLineBorderPointsXZ(const Vector3D& p1, 
				 const Vector3D& p2,
				 Vector3D& pp0, Vector3D& pp1, Vector3D& pp2, Vector3D& pp3,
				 double thickness) const;

  void computeLineBorderPointsYZ(const Vector3D& p1, 
				 const Vector3D& p2,
				 Vector3D& pp0, Vector3D& pp1, Vector3D& pp2, Vector3D& pp3,
				 double thickness) const;

  /** returns url of image */
  string getImageUrl(char letter, const string& extra) const;

  /** central method for writing single box
      faceMode 0: write in x-z plane, 1: write in y-z plane */
  void writeLetter(ostream& os,
		   char c,
		   const Vector3D& posOrig,
		   const Vector3D sizeOrig,
		   const Vector3D& col,
		   int faceMode,
		   int letterStyle) const;

  /** central method for writing single box
      faceMode 0: write in x-z plane, 1: write in y-z plane */
  void writeLetterText(ostream& os,
		       char c,
		       const Vector3D& posOrig,
		       const Vector3D sizeOrig,
		       const Vector3D& col,
		       int faceMode) const;

  /** central method for writing single box
      faceMode 0: write in x-z plane, 1: write in y-z plane */
  void writeLetterLines(ostream& os,
			char c,
			const Vector3D& posOrig,
			const Vector3D sizeOrig,
			const Vector3D& col,
			double thickness,
			int faceMode) const;

  /** central method for writing single box
      faceMode 0: write in x-z plane, 1: write in y-z plane */
  void writeLetterTexture(ostream& os,
			  char c,
			  const Vector3D& posOrig,
			  const Vector3D sizeOrig,
			  const Vector3D& col,
			  int faceMode) const;

  /** writes box with letter as texture map.
      The string extra qualifies a potential different image. Defined so far: empty and "r" for 90 right rotated images */
  void writeBoxLetterBoth(ostream& os,
			  const Vector3D& pos,
			  double height, 
			  const Vector3D& col,
			  char letterFront,
			  char letterLeft,
			  int letterStyle) const;

  /*
  void writeBoxStackBoth(ostream& os,
			 const Vector3D& pos,
			 double height, 
			 const Vec<double>& frequencies,
			 const Vec<Vector3D>& colors,
			 const string& lettersFront,
			 const string& lettersLeft,
			 int letterStyle) const;
  */

  /** write coordinates of 3D point */
  void writeCoord(ostream& os, const Vector3D& p) const;

  /** approximates lines with single face
      @see http://www.cs.vu.nl/~eliens/documents/vrml/reference/BOOK.HTM 
  */
  void writeLine(ostream& os, const Vector3D& p1, const Vector3D& p2, const Vector3D& col,
		 double thickness, int faceMode ) const;

  /** approximates lines with single face
      @see http://www.cs.vu.nl/~eliens/documents/vrml/reference/BOOK.HTM 
  */
  void writeLine(ostream& os, const Vec<Vector3D>& p, const Vector3D& col,
		 double thickness, int faceMode ) const;

  /** approximates lines with single face
      @see http://www.cs.vu.nl/~eliens/documents/vrml/reference/BOOK.HTM 
  */
  void writeFace(ostream& os, const Vec<Vector3D>& p, const Vector3D& col,
		 string imageUrl, int faceMode ) const;

  void writeVrmlCone(ostream& os,
		     const Vector3D& pos,
		     const Vector3D& rot,
		     double angle,
		     const Vector3D& col ) const;

  void writeVrmlBoxStack(ostream& os,
			 const Vector3D& pos,
			 double height, 
			 const Vec<double>& frequencies,
			 const Vec<Vector3D>& colors ) const;

  void
  writeVrmlBoxStack(ostream& os,
		    const Vector3D& pos,
		    double height, 
		    const Vec<double>& frequencies,
		    const Vec<Vector3D>& colors,
		    const string& letters,
		    int faceMode,
		    int letterStyle) const;

  void
  writeBoxStackFlat(ostream& os,
		    const Vector3D& pos,
		    double height, 
		    double errorHeight,
		    const Vec<double>& frequencies,
		    const Vec<Vector3D>& colors,
		    const string& letters,
		    int faceMode,
		    int letterStyle) const;
  
  void
  writeBoxStackFlat2(ostream& os,
		     const Vector3D& pos,
		     double height, 
		     double errorHeight,
		     const Vec<double>& frequencies,
		     const Vec<Vector3D>& colors,
		     const string& letters,
		     int faceMode,
		     int letterStyle) const;

  void writeVrmlReadingFrame(ostream& os,
			     const Alignment& ali,
			     int pos) const;

  void writeVrmlReadingFrame(ostream& os, const Alignment& ali) const;

  void writeVrmlBoxLetter(ostream& os,
			  const Vector3D& pos,
			  double height, 
			  const Vector3D& col,
			  char letter,
			  int faceMode,
			  int letterStyle) const;

  void writeVrmlBoxLetterFlat(ostream& os,
			      const Vector3D& pos,
			      double height, 
			      const Vector3D& col,
			      char letter,
			      int faceMode,
			      int letterStyle) const;

  void writeVrmlBoxLetterFlat2(ostream& os,
			       const Vector3D& pos,
			       double height, 
			       const Vector3D& col,
			       char letter,
			       int faceMode,
			       int letterStyle) const;
    
  void writeVrmlBox(ostream& os,
		    const Vector3D& pos,
		    double height, 
		    const Vector3D& col ) const;

  void
  writeVrmlBox(ostream& os,
	       const Vector3D& pos,
	       const Vector3D& size, 
	       const Vector3D& col ) const;

  void
  writeVrmlBox(ostream& os,
	       const Vector3D& pos,
	       const Vector3D& size, 
	       const Vector3D& col,
	       const string& imageUrl) const;

  void writeVrmlBox(ostream& os,
		    const Vector3D& pos,
		    const Vector3D& size,
		    const Vector3D& rot,
		    double angle,
		    const Vector3D& col ) const;

  void writeVrmlMatching(ostream& os,
			 const Alignment& ali,
			 int pos) const;
  void
  writeVrmlMatching(ostream& os,
		    const Alignment& ali) const;


  /** returns all possible 16 pairs */
  Vec<string> getPairs() const;

  /** calls getPairColorsX according to colorMode */
  Vec<Vector3D> getPairColorsMode(const Vec<string>& pairs, double mutInf) const;
  
  Vec<double> getPairFrequencies(const string& s1, const string& s2, const Vec<string>& pairs) const;

  /**
     initialize vrml world
     backround code take from: 
     http://www.dform.com/inquiry/tutorials/3dsmax/background/bgascii.html
  */
  void initVrml(ostream& os, unsigned int len) const;
    
  void finalize(ostream& os) const;

  virtual void writeBox(ostream& os,
		const Vector3D& pos,
		double height, 
		const Vector3D& col ) const {
    writeVrmlBox(os, pos, height, col);
  }

  virtual void writeBox(ostream& os,
		const Vector3D& pos,
		const Vector3D& size, 
		const Vector3D& col,
		bool faceMode = true) const {
    writeVrmlBox(os, pos, size, col); // TODO : ignoring face Mode
 }

  /** writes box with letter as texture map.
      The string extra qualifies a potential different image. Defined so far: empty and "r" for 90 right rotated images */
  void writeOpenBoxLetterBoth(ostream& os,
			      const Vector3D& pos,
			      double height, 
			      const Vector3D& col,
			      char letterFront,
			      char letterLeft,
			      int letterStyle) const {
    writeBoxLetterBoth(os, pos, height, col, letterFront, letterLeft, letterStyle);
  }


  /* ATTRIBUTES */

  Vec<Letter3D> letters3D;

  bool readingFrameMode;

  // int colorMode;

//   int letterMode;

//   int letterMode2;

  double dxLetterFlat; // = 0.1;

  double dxLetterFlat2; // = 0.1;

  double dxLetterFront; // = 0.1;

  double dxLetterLeft; // = -0.2;

  double dyLetterFlat; // = -0.6;

  double dyLetterFlat2; // = -0.6;

  double dyLetterFront; // = -0.6;

  double dyLetterLeft; //  = 0.3;

  double dzLetter;
  
  // double frequencyLimit;

  double gapLimit;

  // double infLimit; // = 0.5;

  double letterHeight; //  = 0.5;
  
  double letterLineThickness;

  // double stretchZ; // = 10.0;

  // string alphabet;

  string sequenceType;

  string wwwImageBase;

//   CompensationScorer scorer;

//   CompensationScorer scorer2;

  // ResourceBundle resources;

  // VrmlMatrixColorScheme colorScheme;

};

#endif /* __ACLASS_H__ */

