// --*- C++ -*------x---------------------------------------------------------
// $Id: JVXMatrixWriter.cc,v 1.17 2005/11/14 16:01:46 bindewae Exp $
//
// Class:           JVXMatrixWriter
// 
// Base class:      VrmlMatrixWriterBase
//
// Derived classes: - 
//
// Author:          Eckart Bindewald
//
// Project name:    -
//
// Date:            $Date: 2005/11/14 16:01:46 $
//
// Description:     writes JVX file (input file for JavaView)
// -----------------x-------------------x-------------------x-----------------

#include <JVXMatrixWriter.h>
#include <Matrix3D.h>
// #include <vectornumerics.h>
#include <aligncolor_help.h>
#include <stemhelp.h>
#include <StringTools.h>
#include <colorFunctions.h>
#include <AlignColorConstants.h>

// #define VRML2MATRIXWRITER_VERBOSE

// ---------------------------------------------------------------------------
//                                   JVXMatrixWriter
// -----------------x-------------------x-------------------x-----------------

/* CONSTRUCTORS */

/* default constructor */
JVXMatrixWriter::JVXMatrixWriter() : readingFrameMode(false), boxDivideNum(1), boxZMin(0.5), 
				     dxLetterFlat(0.8), dxLetterFlat2(0.0), dxLetterFront(0.1),  dyLetterFlat(0.0), dyLetterFlat2(0.0),
				     dyLetterFront(-0.6), dxLetterLeft(-0.05), dyLetterLeft(0.3),
				     gapLimit(1),
				     letterHeight(0.9), letterWidth(0.9), sequenceType("RNA")
{
}

/* copy constructor */
JVXMatrixWriter::JVXMatrixWriter(const JVXMatrixWriter& other)
{
  copy(other);
}

/* destructor */
JVXMatrixWriter::~JVXMatrixWriter() { }


/* OPERATORS */

/** Assigment operator. */
JVXMatrixWriter& 
JVXMatrixWriter::operator = (const JVXMatrixWriter& orig)
{
  if ((&orig) != this) {
    copy(orig);
  }
  return *this;
}

ostream& 
operator << (ostream& os, const JVXMatrixWriter& rval)
{
  ERROR("Ouput operator not yet implemented!", exception);
  return os;
}

istream& 
operator >> (istream& is, JVXMatrixWriter& rval)
{
  ERROR("Input operator not yet implemented!", exception);
  return is;
}

/* PREDICATES */



/** writes comment to XML file. Avoid "-" characters in comment!
    no comments within tag! 
    @see http://www.extropia.com/tutorials/xml/comments.html
*/
void
JVXMatrixWriter::writeComment(ostream& os, const string& comment) const
{
  os << "<!-- " << comment << "-->" << endl;
}

/** writes single point */
void
JVXMatrixWriter::writePoint(ostream& os,
			    double x, double y, double z) const
{
  os << "<p> " << x << " " << y << " " << z << " </p>" << endl; 
  primitiveCounter += 1;
}

/** writes single point */
void
JVXMatrixWriter::writeColor(ostream& os,
			    double r, double g, double b) const
{
  os << "<c> " << r << " " << g << " " << b << " </c>" << endl; 
}

/** central method for writing single box
 faceMode 0: write in x-z plane, 1: write in y-z plane */
void
JVXMatrixWriter::writeLetter(ostream& os,
			     char c,
			     const Vector3D& posOrig,
			     const Vector3D sizeOrig,
			     const Vector3D& col,
			     int faceMode) const
{
  // cout << "Starting writeLetter: " << c << endl;
  // first try to find letter in read alphabet:
  if (c == ' ') {
    return;
  }
  Vec<Letter3D>::size_type idx = letters3D.size();
  for (Vec<Letter3D>::size_type i = 0; i < letters3D.size(); ++i) {
    if (c == letters3D[i].getLetter()) {
      idx = i;
      break;
    }
  }
  if (idx >= letters3D.size()) {
    cout << "Warning: could not find bitmap for letter " << c << endl;
    return; // letter not found
  }
  const Letter3D& l3d = letters3D[idx];
  // cout << "Using letter: " << l3d << endl;
  // const Vec<Vec<int> >& bitMap = l3d.getMatrix();
  os << "<geometry>" << endl;
  os << " <pointSet dim=\"3\" point=\"show\" color=\"hide\" >" << endl;
  os << "  <points>" << endl;
  Vector3D size = sizeOrig; // Vector3D(1.0, 1.0, 1.0);
  Vector3D pos = posOrig - (size * 0.5); // orig: middle, pos: lower left corner
  double dx = size.x() / l3d.getNumberColumns();
  // double dy = size.y() * 0.5;
  double dy = size.y() / l3d.getNumberRows();
  double dz = size.z() / l3d.getNumberRows();
  Vec<Letter3D::point_type> points = l3d.getPoints();
  Vec<Letter3D::line_type> lines = l3d.getLines();
  // unsigned int pointCounter = 0;
  // rows: height
//   for (Vec<Vec<Vec<unsigned int> > >::size_type i = 0; i < bitMap.size(); ++i) {
//     // columns: width
//     for (Vec<Vec<unsigned int> >::size_type j = 0; j < bitMap[i].size(); ++j) {
//       if (bitMap[i][j] > 0) {
// 	writePoint(os, pos.x() + (j * dx), pos.y() - 0.01, pos.z() + size.z() - (i * dz));
// 	++pointCounter;
//       }
//     }
//   }
  for (Vec<Letter3D::point_type>::size_type i = 0; i < points.size(); ++i) {
    switch (faceMode) {
    case Letter3D::FACE_XZ: // write in x-z plane
      writePoint(os, pos.x() + (points[i].first * dx), pos.y(), 
		 pos.z() + points[i].second * dz);
      break;
    case Letter3D::FACE_YZ:  // write in y-z plane
      writePoint(os, pos.x(), pos.y() - (points[i].first * dx), 
		 pos.z() + points[i].second * dz);
      break;
    case Letter3D::FACE_XY180:  // write in x-y plane, heads up
      writePoint(os, pos.x() - (points[i].first * dx), pos.y() - points[i].second * dy,
		 pos.z() );
      break;
    case Letter3D::FACE_XY90:  // write in x-y plane, 90 degrees turned
      writePoint(os, pos.x() - (points[i].second * dx), pos.y() + points[i].first * dy,
		 pos.z() );
      break;
    default: ERROR("Internal error: Unknown face mode!", exception);
    }
  }
  os << "  <color>" << col.x() << " " << col.y() << " " << col.z() 
     << "</color>" << endl;
    os << "  <thickness> 1 </thickness>" << endl;
  os << "  </points>" << endl;
//   os << "  <colors>" << endl;
//   for (unsigned int i = 0; i < pointCounter; ++i) {
//     writeColor(os, col.x(), col.y(), col.z());
//   }
//   os << "  </colors>" << endl;
  os << " </pointSet>" << endl;
  os << " <lineSet line=\"show\" thicknesses=\"hide\" color=\"hide\">" << endl;
  os << "  <lines>" << endl;
  for (Vec<Letter3D::line_type>::size_type i = 0; i < lines.size(); ++i) {
    for (Letter3D::line_type::size_type j = 0; j + 1< lines[i].size(); ++j) {
      os << "  <l>" << lines[i][j] << " " << lines[i][j+1] << "</l>" << endl;
      primitiveCounter++;
    }
  }
  os << "   <color>" << col.x() << " " << col.y() << " " << col.z() 
     << "</color>" << endl;
  os << "  <thickness> 4 </thickness>" << endl;
  os << "  </lines>" << endl;
  os << " </lineSet>" << endl;
  os << "</geometry>" << endl;
}

/** central method for writing single box, open at bottom and top */
void
JVXMatrixWriter::writeOpenBoxRaw(ostream& os,
				 const Vector3D& pos,
				 const Vector3D& size, 
				 const Vector3D& col ) const
{
  double xh = 0.5 * size.x();
  double yh = 0.5 * size.y();
  double zh = 0.5 * size.z();
  os << "<geometry>" << endl;
  os << " <pointSet dim=\"3\" point=\"hide\">" << endl;
  os << "  <points>" << endl;
  writePoint(os, pos.x() + xh, pos.y() + yh, pos.z() + zh);
  writePoint(os, pos.x() + xh, pos.y() + yh, pos.z() - zh);
  writePoint(os, pos.x() + xh, pos.y() - yh, pos.z() + zh);
  writePoint(os, pos.x() + xh, pos.y() - yh, pos.z() - zh);
  writePoint(os, pos.x() - xh, pos.y() + yh, pos.z() + zh);
  writePoint(os, pos.x() - xh, pos.y() + yh, pos.z() - zh);
  writePoint(os, pos.x() - xh, pos.y() - yh, pos.z() + zh);
  writePoint(os, pos.x() - xh, pos.y() - yh, pos.z() - zh);
  os << "  </points>" << endl;
  os << " </pointSet>" << endl;

  if (true || writerParameters.lineMode) {
    os << " <lineSet line=\"show\">" << endl
       << "  <lines>" << endl
       << "   <l>0 1</l>" << endl // connecting "right" face of cube
       << "   <l>1 3</l>" << endl
       << "   <l>3 2</l>" << endl
       << "   <l>2 0</l>" << endl
       << "   <l>4 5</l>" << endl // connecting "left face" of cube
       << "   <l>5 7</l>" << endl
       << "   <l>6 7</l>" << endl
       << "   <l>4 6</l>" << endl
       << "   <l>0 4</l>" << endl // connections between left and right side
       << "   <l>1 5</l>" << endl
       << "   <l>3 7</l>" << endl
       << "   <l>2 6</l>" << endl
       << "   <color>0 0 0</color>" << endl
       << "  </lines>" << endl
       << " </lineSet>" << endl;
    primitiveCounter += 12;
  }

  os << " <faceSet color=\"hide\" face=\"show\" edge=\"hide\">" << endl;
  os << "  <faces>" << endl;
  os << "   <f> 0 1 3 2 </f>" << endl;
  os << "   <f> 2 3 7 6 </f>" << endl;
  os << "   <f> 4 5 7 6 </f>" << endl;
  os << "   <f> 0 1 5 4 </f>" << endl;
  primitiveCounter += 4;
  os << "   <color> " << col.x() << " " << col.y() << " " << col.z() << " </color>" << endl;
  os << "  </faces>" << endl;
  os << " </faceSet>" << endl;
  os << "</geometry>" << endl;
}
 
/** central method for writing single box, open at bottom and top */
void
JVXMatrixWriter::writeOpenBox(ostream& os,
			      const Vector3D& pos,
			      const Vector3D& size, 
			      const Vector3D& col ) const
{
  // cout << "Called write open box: " << pos << " " << size << endl;
  double dz = size.z();
  if ((dz < boxZMin) || (boxDivideNum < 2)) {
    writeOpenBoxRaw(os, pos, size, col);
  }
  // cout << "Dividing box in pieces: " << dz << " " << boxDivideNum << endl;
  double deltaz = dz / boxDivideNum;
  double posz = pos.z() - (0.5 * dz) + 0.5 * deltaz;
  Vector3D newSize = Vector3D(size.x(), size.y(), deltaz);
  Vector3D newPos(pos);
  for (unsigned int i = 0; i < boxDivideNum; ++i) {
    newPos.z(posz + (i * deltaz));
    writeOpenBoxRaw(os, newPos , newSize, col);
  }
}

/** central method for writing single box */
void
JVXMatrixWriter::writeBox(ostream& os,
			  const Vector3D& pos,
			  const Vector3D& size, 
			  const Vector3D& col,
			  bool faceMode) const
{
  double xh = 0.5 * size.x();
  double yh = 0.5 * size.y();
  double zh = 0.5 * size.z();

  os << "<geometry>" << endl;
  os << " <pointSet dim=\"3\" point=\"hide\">" << endl;
  os << "  <points>" << endl;
  writePoint(os, pos.x() + xh, pos.y() + yh, pos.z() + zh);
  writePoint(os, pos.x() + xh, pos.y() + yh, pos.z() - zh);
  writePoint(os, pos.x() + xh, pos.y() - yh, pos.z() + zh);
  writePoint(os, pos.x() + xh, pos.y() - yh, pos.z() - zh);
  writePoint(os, pos.x() - xh, pos.y() + yh, pos.z() + zh);
  writePoint(os, pos.x() - xh, pos.y() + yh, pos.z() - zh);
  writePoint(os, pos.x() - xh, pos.y() - yh, pos.z() + zh);
  writePoint(os, pos.x() - xh, pos.y() - yh, pos.z() - zh);
  os << "  </points>" << endl;
  os << " </pointSet>" << endl;
  if (writerParameters.lineMode) {
    os << " <lineSet line=\"show\">" << endl
       << "  <lines>" << endl
       << "   <l>0 1</l>" << endl // connecting "right" face of cube
       << "   <l>1 3</l>" << endl
       << "   <l>3 2</l>" << endl
       << "   <l>2 0</l>" << endl
       << "   <l>4 5</l>" << endl // connecting "left face" of cube
       << "   <l>5 7</l>" << endl
       << "   <l>6 7</l>" << endl
       << "   <l>4 6</l>" << endl
       << "   <l>0 4</l>" << endl // connections between left and right side
       << "   <l>1 5</l>" << endl
       << "   <l>3 7</l>" << endl
       << "   <l>2 6</l>" << endl
       << "   <color>0 0 0</color>" << endl
       << "  </lines>" << endl
       << " </lineSet>" << endl;
    primitiveCounter += 12;
  }
  if (faceMode) {
    os << " <faceSet color=\"hide\" face=\"show\" edge=\"hide\">" << endl;
    os << "  <faces>" << endl;
    os << "   <f> 0 1 3 2 </f>" << endl;
    os << "   <f> 2 3 7 6 </f>" << endl;
    os << "   <f> 4 5 7 6 </f>" << endl;
    os << "   <f> 0 1 5 4 </f>" << endl;
    os << "   <f> 0 2 6 4 </f>" << endl;
    os << "   <f> 1 3 7 5 </f>" << endl;
    os << "   <color> " << col.x() << " " << col.y() << " " << col.z() << " </color>" << endl;
    os << "  </faces>" << endl;
    os << " </faceSet>" << endl;
    primitiveCounter += 6;
  }
  os << "</geometry>" << endl;
}

void
JVXMatrixWriter::writePointLayer(ostream& os,
				 const Vector3D& pos,
				 double dz, 
				 double width) const
{
  double xh = 0.5 * width;
  double yh = 0.5 * width;
  writePoint(os, pos.x() + xh, pos.y() - yh, pos.z() + dz);
  writePoint(os, pos.x() + xh, pos.y() + yh, pos.z() + dz);
  writePoint(os, pos.x() - xh, pos.y() + yh, pos.z() + dz);
  writePoint(os, pos.x() - xh, pos.y() - yh, pos.z() + dz);
}

/** writes row of boxes, expects correct order */
void
JVXMatrixWriter::writeBoxStackRaw(ostream& os,
				const Vector3D& pos,
				double height, 
				const Vec<double>& frequencies,
				const Vec<Vector3D>& colors ) const
{
  os << "<geometry>" << endl;
  os << " <pointSet dim=\"3\" point=\"hide\">" << endl;
  os << "  <points>" << endl;
  writePointLayer(os, pos, 0.0, 1.0);
  double currentHeight = 0.0;
  for (unsigned int i = 0; i < frequencies.size(); ++i) {
//   	writeBox(os, Vector3D(pos.x(), pos.y(), 
// 			      currentHeight + (0.5 * height * frequencies[i])),
// 		 height*frequencies[i], colors[i]);
    writePointLayer(os, pos, currentHeight + (height * frequencies[i]), 1.0);
    currentHeight += frequencies[i];
  }
  os << "  </points>" << endl;
  os << " </pointSet>" << endl;
  os << " <faceSet color=\"show\" face=\"show\" edge=\"hide\">" << endl;
  os << "  <faces>" << endl;
  unsigned int pc = 0;
  unsigned int d = 4;
  for (unsigned int i = 0; i < frequencies.size(); ++i) {
    for (unsigned int j = 0; j < d; ++j) {
      os << "   <f> " << pc + j << " " << pc + j + 1 << " " << pc + j + 1 + d << " " << pc + j + d << "</f>" << endl;
      primitiveCounter++;
    }
    pc += d;
  }
  // "lid" :
  os << "   <f> " << pc << " " << pc + 1 << " " << pc + 2 << " " << pc + 3 << "</f>" << endl;  
  os << "  </faces>" << endl;
  os << " </faceSet>" << endl;
  os << "</geometry>" << endl;
}

void
JVXMatrixWriter::writeBox(ostream& os,
				const Vector3D& pos,
				double height, 
				const Vector3D& col ) const
{
  if (height <= 0.0) {
    return;
  }
  writeBox(os, pos, Vector3D(1.0, 1.0, height), col);
}

void
JVXMatrixWriter::writeOpenBox(ostream& os,
				const Vector3D& pos,
				double height, 
				const Vector3D& col ) const
{
  if (height <= 0.0) {
    return;
  }
  writeOpenBox(os, pos, Vector3D(1.0, 1.0, height), col);
}

void
JVXMatrixWriter::writeBox(ostream& os,
				const Vector3D& pos,
				const Vector3D& size,
				const Vector3D& rotVec,
				double rotAngle,
				const Vector3D& col ) const
{
  if (size <= 0.0) {
    return;
  }
  double xh = 0.5 * size.x();
  double yh = 0.5 * size.y();
  double zh = 0.5 * size.z();

  Vec<Vector3D> points(8);
  points[0] = Vector3D(pos.x() + xh, pos.y() + yh, pos.z() + zh);
  points[1] = Vector3D(pos.x() + xh, pos.y() + yh, pos.z() - zh);
  points[2] = Vector3D(pos.x() + xh, pos.y() - yh, pos.z() + zh);
  points[3] = Vector3D(pos.x() + xh, pos.y() - yh, pos.z() - zh);
  points[4] = Vector3D(pos.x() - xh, pos.y() + yh, pos.z() + zh);
  points[5] = Vector3D(pos.x() - xh, pos.y() + yh, pos.z() - zh);
  points[6] = Vector3D(pos.x() - xh, pos.y() - yh, pos.z() + zh);
  points[7] = Vector3D(pos.x() - xh, pos.y() - yh, pos.z() - zh);
  for (unsigned int i = 0; i < points.size(); ++i) {
    points[i] = Matrix3D::rotate((points[i] - pos), rotVec, rotAngle) + pos; // pos is rotation center
  }
  os << "<geometry>" << endl;
  os << " <pointSet dim=\"3\" point=\"hide\">" << endl;
  os << "  <points>" << endl;
  for (unsigned int i = 0; i < points.size(); ++i) {
    writePoint(os, points[i].x(), points[i].y(), points[i].z());
  }
  os << "  </points>" << endl;
  os << " </pointSet>" << endl;
  os << " <faceSet color=\"hide\" face=\"show\" edge=\"hide\">" << endl;
  os << "  <faces>" << endl;
  os << "   <f> 0 1 3 2 </f>" << endl;
  os << "   <f> 2 3 7 6 </f>" << endl;
  os << "   <f> 4 5 7 6 </f>" << endl;
  os << "   <f> 0 1 5 4 </f>" << endl;
  os << "   <f> 0 2 6 4 </f>" << endl;
  os << "   <f> 1 3 7 5 </f>" << endl;
  primitiveCounter += 6;
  os << "   <color> " << col.x() << " " << col.y() << " " << col.z() << " </color>" << endl;
  os << "  </faces>" << endl;
  os << " </faceSet>" << endl;
  os << "</geometry>" << endl;

}

/** cone: if not rotated, point in z-direction.
 * length: distance from base point (center of disc) to tip
 * radius: radius of disc
 * nCircle: number of devisions of circle 
 */
void
JVXMatrixWriter::writeCone(ostream& os,
			   const Vector3D& pos,
			   double length,
			   double radius,
			   unsigned int nCircle,
			   const Vector3D& rotVec,
			   double rotAngle,
			   const Vector3D& col ) const
{
  PRECOND(nCircle > 1, exception);
  Vec<Vector3D> points(nCircle + 2);
  double angleDelta = 2.0 * PI / nCircle;
  points[0] = Vector3D(pos.x(), pos.y(), pos.z() + length); // first point is "tip" of cone pointing in z-direction
  for (unsigned int i = 0; i < nCircle; ++i) {
    double angle = angleDelta * i;
    double dx = radius * cos(angle);
    double dy = radius * sin(angle);
    points[i+1] = Vector3D(pos.x() + dx, pos.y() + dy, pos.z());
  }
  points[points.size()-1] = pos;
  for (unsigned int i = 0; i < points.size(); ++i) {
    points[i] = Matrix3D::rotate((points[i] - pos), rotVec, rotAngle) + pos; // pos is rotation center
  }
  os << "<geometry>" << endl;
  os << " <pointSet dim=\"3\" point=\"hide\">" << endl;
  os << "  <points>" << endl;
  for (unsigned int i = 0; i < points.size(); ++i) {
    writePoint(os, points[i].x(), points[i].y(), points[i].z());
  }
  os << "  </points>" << endl;
  os << " </pointSet>" << endl;

  os << " <lineSet line=\"show\">" << endl
     << "  <lines>" << endl;
  for (int i = 0; i < static_cast<int>(nCircle)-1; ++i) {
    os << "   <l>" << (i+1) << " " << (i+2) << "</l>" << endl; // connecting "right" face of cube
    ++primitiveCounter;
  }
  os << "   <l>" << nCircle << " 1</l>" << endl;
  ++primitiveCounter;
  // lines going to "tip" of cone:
  for (int i = 1; i <= static_cast<int>(nCircle); ++i) {
    os << "   <l>" << 0 << " " << i << "</l>" << endl; // connecting "right" face of cube
    ++primitiveCounter;
  }
  os << "  </lines>" << endl
     << " </lineSet>" << endl;

  os << " <faceSet color=\"hide\" face=\"show\" edge=\"hide\">" << endl;
  os << "  <faces>" << endl;
  for (unsigned int i = 0; i < (nCircle-1); ++i) {
    unsigned int id1 = i + 1;
    unsigned int id2 = i + 2;
    os << "   <f> 0 " << id1 << " " << id2 << " </f>" << endl;
    os << "   <f> " << (points.size()-1) << " " << id1 << " " << id2 << " </f>" << endl;
    primitiveCounter += 2;
  }
  os << "   <f> 0 " << (nCircle-1) << " 1 </f>" << endl;
  os << "   <f> " << (points.size()-1) << " " << (nCircle-1) << " 1 </f>" << endl;
  primitiveCounter += 2;
  os << "   <color> " << col.x() << " " << col.y() << " " << col.z() << " </color>" << endl;
  os << "  </faces>" << endl;
  os << " </faceSet>" << endl;
  os << "</geometry>" << endl;

}

/** writes box with letter as set of lines.
 The string extra qualifies a potential different image. Defined so far: empty and "r" for 90 right rotated images */
void
JVXMatrixWriter::writeBoxLetterBoth(ostream& os,
				    const Vector3D& pos,
				    double height, 
				    const Vector3D& col,
				    char letterFront,
				    char letterLeft,
				    int letterStyle) const
{
  // cout << "Starting writeBoxLetterBoth!" << endl;
  if (height <= 0.0) {
    return;
  }
  writeBox(os, pos, height, col);
  Vector3D black(0.1, 0.1, 0.1);
  if (height > letterHeight) {
    if (letterFront != ' ') {
      writeLetter(os, letterFront, Vector3D(pos.x() + dxLetterFront, pos.y() + dyLetterFront, pos.z()), Vector3D(letterWidth, 0.0, 0.5 * height), black,
		  Letter3D::FACE_XZ);
    }
    if (letterLeft != ' ') {
      writeLetter(os, letterLeft, Vector3D(pos.x() + dxLetterLeft,
					   pos.y() + dyLetterLeft, pos.z()), Vector3D(letterWidth, 0.0, 0.5 * height), black,
		  Letter3D::FACE_YZ);  
    }
  }
}

/** writes box with letter as set of lines.
 The string extra qualifies a potential different image. Defined so far: empty and "r" for 90 right rotated images */
void
JVXMatrixWriter::writeBoxLetterFlat(ostream& os,
				    const Vector3D& pos,
				    double height, 
				    const Vector3D& col,
				    char letterFront) const
{
  // cout << "Starting writeBoxLetterFlat!" << endl;
  if (height <= 0.0) {
    return;
  }
  writeBox(os, pos, Vector3D(1.0, height, 1.0), col);
  const Vector3D black(0.1, 0.1, 0.1);
  if (height > letterHeight) {
    if (letterFront != ' ') {
      writeLetter(os, letterFront, Vector3D(pos.x() + dxLetterFlat, pos.y() + (0.5 * height) + dyLetterFlat, pos.z() - dyLetterFront), Vector3D(letterWidth, 0.5 * height, 0.0), black,
		  Letter3D::FACE_XY180);
    }
    //     if (letterLeft != ' ') {
    // not properly implemented!
//       writeLetter(os, letterLeft, Vector3D(pos.x() + dxLetterLeft,
// 					   pos.y() + dyLetterLeft, pos.z()), Vector3D(0.9, 0.0, 0.5 * height), black,
// 		  Letter3D::FACE_YZ);  
    // }
  }
}

/** writes box with letter as set of lines.
 The string extra qualifies a potential different image. Defined so far: empty and "r" for 90 right rotated images */
void
JVXMatrixWriter::writeBoxLetterFlat2(ostream& os,
				     const Vector3D& pos,
				     double height, 
				     const Vector3D& col,
				     char letterFront) const
{
  // cout << "Starting writeBoxLetter!" << endl;
  if (height <= 0.0) {
    return;
  }
  writeBox(os, pos, Vector3D(height, 1.0, 1.0), col);
  Vector3D black(0.1, 0.1, 0.1);
  if (height > letterHeight) {
    if (letterFront != ' ') {
      writeLetter(os, letterFront, Vector3D(pos.x() + (0.5*height) + dxLetterFlat2, pos.y() + dyLetterFlat2, pos.z() - dyLetterFront), 
		  Vector3D(0.5*height, letterWidth, 0.0), black,
		  Letter3D::FACE_XY90);
    }
    // not properly implemented!
//     if (letterLeft != ' ') {
//       writeLetter(os, letterLeft, Vector3D(pos.x() + dxLetterLeft,
// 					   pos.y() + dyLetterLeft, pos.z()), Vector3D(0.9, 0.0, 0.5 * height), black,
// 		  Letter3D::FACE_YZ);  
//     }
  }
}


/** writes box with letter as texture map.
 The string extra qualifies a potential different image. Defined so far: empty and "r" for 90 right rotated images */
void
JVXMatrixWriter::writeBoxLetterBoth2(ostream& os,
				     const Vector3D& pos,
				     const Vector3D& dimensions,
				     const Vector3D& col,
				     char letterFront,
				     char letterLeft) const
{
  // cout << "Starting writeBoxLetter!" << endl;
  // double height = dimensions.z();

  double xh = 0.5 * dimensions.x();
  double yh = 0.5 * dimensions.y();
  double zh = 0.5 * dimensions.z();

  if ((xh <= 0.0) || (yh <= 0.0) || (zh <= 0.0)) {
    return;
  }

  string imageUrlFront = getImageUrl(letterFront);
  string imageUrlLeft = getImageUrl(letterLeft);

  os << "<geometry>" << endl;
  os << " <pointSet dim=\"3\" point=\"hide\">" << endl;
  os << "  <points>" << endl;
  writePoint(os, pos.x() + xh, pos.y() + yh, pos.z() + zh);
  writePoint(os, pos.x() + xh, pos.y() + yh, pos.z() - zh);
  writePoint(os, pos.x() + xh, pos.y() - yh, pos.z() + zh);
  writePoint(os, pos.x() + xh, pos.y() - yh, pos.z() - zh);
  writePoint(os, pos.x() - xh, pos.y() + yh, pos.z() + zh);
  writePoint(os, pos.x() - xh, pos.y() + yh, pos.z() - zh);
  writePoint(os, pos.x() - xh, pos.y() - yh, pos.z() + zh);
  writePoint(os, pos.x() - xh, pos.y() - yh, pos.z() - zh);
  os << "  </points>" << endl;
  os << " </pointSet>" << endl;
  os << " <faceSet color=\"hide\" face=\"show\" edge=\"hide\">" << endl;
  os << "  <faces>" << endl;
  os << "   <f> 0 1 3 2 </f>" << endl;
  ++primitiveCounter;
  if (imageUrlFront.size() == 0) {
    os << "   <f> 2 3 7 6 </f>" << endl;
    ++primitiveCounter;
  }
  if (imageUrlLeft.size() == 0) {
    os << "   <f> 4 5 7 6 </f>" << endl;
    ++primitiveCounter;
  }
  os << "   <f> 0 1 5 4 </f>" << endl;
  os << "   <f> 0 2 6 4 </f>" << endl;
  os << "   <f> 1 3 7 5 </f>" << endl;
  primitiveCounter += 3;
  os << "   <color> " << col.x() << " " << col.y() << " " << col.z() << " </color>" << endl;
  os << "  </faces>" << endl;
  os << " </faceSet>" << endl;
  os << "</geometry>" << endl;

  // write single face with texture map on it:
  if (imageUrlFront.size() > 0) {
    os << "<geometry>" << endl;
    os << " <pointSet dim=\"3\" point=\"hide\">" << endl;
    os << "  <points>" << endl;
    writePoint(os, pos.x() - xh, pos.y() - yh, pos.z() - zh);
    writePoint(os, pos.x() + xh, pos.y() - yh, pos.z() - zh);
    writePoint(os, pos.x() + xh, pos.y() - yh, pos.z() + zh);
    writePoint(os, pos.x() - xh, pos.y() - yh, pos.z() + zh);
    os << "  </points>" << endl;
    os << " </pointSet>" << endl;
    os << " <faceSet color=\"hide\" face=\"show\" edge=\"hide\">" << endl;
    os << "  <faces>" << endl;
    os << "   <f>0 1 2 3</f>" << endl;
    ++primitiveCounter;
    os << "   <color> " << col.x() << " " << col.y() << " " << col.z() << " </color>" << endl;
    os << "  </faces>" << endl;
    os << "  <textures dim=\"2\" blend=\"replace\">" << endl;
    os << "    <t> 0.0 0.0 </t> " << endl;
    os << "    <t> 1.0 0.0 </t> " << endl;
    os << "    <t> 1.0 1.0 </t> " << endl;
    os << "    <t> 0.0 1.0 </t> " << endl;
    os << "    <image> <url>" << imageUrlFront << "</url> </image>" << endl;
    os << "  </textures>" << endl;
    os << " </faceSet>" << endl;
    os << "</geometry>" << endl;
  }
  // write single face with texture map on it:
  if (imageUrlLeft.size() > 0) {
    os << "<geometry>" << endl;
    os << " <pointSet dim=\"3\" point=\"hide\">" << endl;
    os << "  <points>" << endl;
    writePoint(os, pos.x() - xh, pos.y() + yh, pos.z() - zh);
    writePoint(os, pos.x() - xh, pos.y() - yh, pos.z() - zh);
    writePoint(os, pos.x() - xh, pos.y() - yh, pos.z() + zh);
    writePoint(os, pos.x() - xh, pos.y() + yh, pos.z() + zh);
    os << "  </points>" << endl;
    os << " </pointSet>" << endl;
    os << " <faceSet color=\"hide\" face=\"show\" edge=\"hide\">" << endl;
    os << "  <faces>" << endl;
    os << "   <f>0 1 2 3</f>" << endl;
    ++primitiveCounter;
    os << "   <color> " << col.x() << " " << col.y() << " " << col.z() << " </color>" << endl;
    os << "  </faces>" << endl;
    os << "  <textures dim=\"2\" blend=\"replace\">" << endl;
    os << "    <t> 0.0 0.0 </t> " << endl;
    os << "    <t> 1.0 0.0 </t> " << endl;
    os << "    <t> 1.0 1.0 </t> " << endl;
    os << "    <t> 0.0 1.0 </t> " << endl;
    os << "    <image> <url>" << imageUrlLeft << "</url> </image>" << endl;
    os << "  </textures>" << endl;
    os << " </faceSet>" << endl;
    os << "</geometry>" << endl;
  }
}

/** writes box with letter as texture map.
 The string extra qualifies a potential different image. Defined so far: empty and "r" for 90 right rotated images */
void
JVXMatrixWriter::writeOpenBoxLetterBoth(ostream& os,
					const Vector3D& pos,
					double height, 
					const Vector3D& col,
					char letterFront,
					char letterLeft,
					int letterStyle) const
{
  if (height <= 0.0) {
    return;
  }
  writeOpenBox(os, pos, height, col);
  Vector3D black(0.1, 0.1, 0.1);
  if (height >= letterHeight) {
    if (letterFront != ' ') {
      writeLetter(os, letterFront, Vector3D(pos.x() + dxLetterFront, pos.y() + dyLetterFront, pos.z()), Vector3D(letterWidth, 0.0, 0.5 * height), black,
		  Letter3D::FACE_XZ);  
    }
    if (letterLeft != ' ') {
      writeLetter(os, letterLeft, Vector3D(pos.x() + dxLetterLeft, pos.y() + dyLetterLeft, pos.z()), Vector3D(letterWidth, 0.0, 0.5 * height), black,
		  Letter3D::FACE_YZ);  
    }
  }
}

void
JVXMatrixWriter::writeBoxStack(ostream& os,
				     const Vector3D& pos,
				     double height, 
				     const Vec<double>& frequencies,
				     const Vec<Vector3D>& colors ) const
{
  PRECOND(colors.size()==frequencies.size(), exception);
  // cout << "starting writeBoxStack!" << endl;
  double currentHeight = 0.0;
  string comment = "starting writeBoxStack " + dtos(height);
  writeComment(os, comment);
  unsigned int boxCount = 0;
  // deterime ordering: 
  if (height <= 0.0) {
    cout << "height smaller zero: " << height << endl;
    return;
  }
  double dz = 0.01; // little bit of space
  Vec<unsigned int> order = documentedOrder(frequencies);
  // reverse(order.begin(), order.end()); // smallest frequency first
  for (unsigned int ii = 0; ii < order.size(); ++ii) {
    unsigned int i = order[ii];
    ERROR_IF(frequencies[i] > 1.0, "Internal error in line 235!",
	     exception);
    if (frequencies[i] > frequencyLimit) {
      ++boxCount;
      if (ii > 0) {
	writeBox(os, Vector3D(pos.x(), pos.y(), 
			      currentHeight + (0.5 * height * frequencies[i])),
		 (height*frequencies[i]) - dz, colors[i]);
      }
      else {
	writeBox(os, Vector3D(pos.x(), pos.y(), 
			      currentHeight + (0.5 * height * frequencies[i])),
		 height*frequencies[i], colors[i]);
      }
    }
    //     else {
    //       cout << "Frequency too small: " << i << " " << frequencies[i] << endl;
    //     }
    currentHeight += height * frequencies[i];
//     if (boxCount > 1) {
//       return;
//     }
  }

}

void
JVXMatrixWriter::writeBoxStackFlat(ostream& os,
				   const Vector3D& pos,
				   double height, 
				   double errorHeight,
				   const Vec<double>& frequencies,
				   const Vec<Vector3D>& colors,
				   const string& lettersFront,
				   int faceMode,
				   int letterStyle) const
  // const string& lettersLeft) const
{
  PRECOND(colors.size()==frequencies.size(), exception);
//   if (wwwImageBase.size() == 0) { // no texture mapping
//     writeBoxStack(os, pos, height, frequencies, colors);
//     return;
//   }
  string comment = "start writeBoxStack with letters " + dtos(height);
  writeComment(os, comment);
  if (verboseLevel > 2) {
    cout << comment << endl;
  }
  double currentHeight = pos.y();
  // deterime ordering: 
  if (height <= 0.0) {
    // cout << "height smaller zero: " << height << endl;
    return;
  }
  Vector3D errorColor(0.7, 0.7, 0.7);
  Vec<unsigned int> order = documentedOrder(frequencies);
  // reverse(order.begin(), order.end()); smallest first
  for (unsigned int ii = 0; ii < colors.size(); ++ii) {
    unsigned int i = order[ii];
    ERROR_IF(frequencies[i] > 1.0, "Internal error in line 262!",
	     exception);
    if (frequencies[i] > frequencyLimit) {
      if (ii == (colors.size()-1)) {
	// if (letterMode2 == AlignColorConstants::LOGO2_CHAR_TEXTURE) {
	writeBoxLetterFlat(os, Vector3D(pos.x(), currentHeight - (0.5 * height * frequencies[i]), pos.z()),
			   height*frequencies[i], colors[i], lettersFront[i]); // , lettersLeft[i]);
	  /* not yet working properly:
	    writeBoxLetterBoth2(os, Vector3D(pos.x(), pos.y(), 
	    currentHeight + (0.5 * height * frequencies[i])),
	    Vector3D(1.0, 1.0, height*frequencies[i]), colors[i], lettersFront[i], lettersLeft[i]);
	  */
	  /*
	    }
	    else if (letterMode2 == AlignColorConstants::LOGO2_CHAR_DEFAULT) {
	    writeBoxLetterBoth(os, Vector3D(pos.x(), pos.y(), 
	    currentHeight + (0.5 * height * frequencies[i])),
	    height*frequencies[i], colors[i], lettersFront[i], lettersLeft[i]);
	    }
	    else { // no letters
	    writeBoxLetterBoth(os, Vector3D(pos.x(), pos.y(), 
	    currentHeight + (0.5 * height * frequencies[i])),
	    height*frequencies[i], colors[i], ' ', ' ');
	    }
	  */
      }
      else { // not topmost block
	writeBoxLetterFlat(os, Vector3D(pos.x(), 
					currentHeight - (0.5 * height * frequencies[i]), pos.z()),
	   height*frequencies[i], colors[i], lettersFront[i]);
	
	// if (letterMode2 == AlignColorConstants::LOGO2_CHAR_TEXTURE) {
	// currently no "open" version, still implement!
	// writeBoxLetterBoth2(os, Vector3D(pos.x(), pos.y(), 
	// 					   currentHeight + (0.5 * height * frequencies[i])),
	// 			      Vector3D(1.0, 1.0, height*frequencies[i]), colors[i], lettersFront[i], lettersLeft[i]);
	  /*
	    }
	    else if (letterMode2 == AlignColorConstants::LOGO2_CHAR_DEFAULT) {
	    writeOpenBoxLetterBoth(os, Vector3D(pos.x(), pos.y(), 
	    currentHeight + (0.5 * height * frequencies[i])),
	    height*frequencies[i], colors[i], lettersFront[i], lettersLeft[i]);
	    }
	    else { // no letters
	    writeOpenBoxLetterBoth(os, Vector3D(pos.x(), pos.y(), 
	    currentHeight + (0.5 * height * frequencies[i])),
	    height*frequencies[i], colors[i], ' ', ' ');
	    }
	  */
      }
    }
    currentHeight -= height * frequencies[i];
  }
  if (errorMode) {
    if (errorHeight < 0.1) {
      errorHeight = 0.1;
    }
    double topWidth = 0.1; // height of top "hat plate"
    writeBox(os, 
	     Vector3D(pos.x(), currentHeight - (0.5 * errorHeight) + (0.5*topWidth), pos.z()),
	     Vector3D(0.5, errorHeight - topWidth, 0.5), errorColor, true);
    writeBox(os, 
 	     Vector3D(pos.x(), currentHeight - errorHeight - (0.5*topWidth), pos.z()),
 	     Vector3D(1.0, topWidth, 1.0), errorColor, true);
  }
}

void
JVXMatrixWriter::writeBoxStackFlat2(ostream& os,
				    const Vector3D& pos,
				    double height, 
				    double errorHeight,
				    const Vec<double>& frequencies,
				    const Vec<Vector3D>& colors,
				    const string& lettersFront,
				    int faceMode,
				    int letterStyle) const
  // const string& lettersLeft) const
{
  PRECOND(colors.size()==frequencies.size(), exception);
//   if (wwwImageBase.size() == 0) { // no texture mapping
//     writeBoxStack(os, pos, height, frequencies, colors);
//     return;
//   }
  string comment = "start writeBoxStack with letters " + dtos(height);
  writeComment(os, comment);
  double currentHeight = pos.x();
  // deterime ordering: 
  if (height <= 0.0) {
    // cout << "height smaller zero: " << height << endl;
    return;
  }
  Vector3D errorColor(0.7, 0.7, 0.7);
  Vec<unsigned int> order = documentedOrder(frequencies);
  // reverse(order.begin(), order.end()); smallest first
  for (unsigned int ii = 0; ii < colors.size(); ++ii) {
    unsigned int i = order[ii];
    ERROR_IF(frequencies[i] > 1.0, "Internal error in line 262!",
	     exception);
    if (frequencies[i] > frequencyLimit) {
      if (ii == (colors.size()-1)) {
	// if (letterMode2 == AlignColorConstants::LOGO2_CHAR_TEXTURE) {
	writeBoxLetterFlat2(os, 
         Vector3D(currentHeight - (0.5*height*frequencies[i]),pos.y(),pos.z()),
				height*frequencies[i], colors[i], lettersFront[i]); // , lettersLeft[i]);     
      }
      else { // not topmost block
	writeBoxLetterFlat2(os, 
         Vector3D(currentHeight - (0.5*height*frequencies[i]),pos.y(),pos.z()),
			    height*frequencies[i], colors[i], lettersFront[i]); // , lettersLeft[i]);     
      }
    }
    currentHeight -= height * frequencies[i];
  }
  if (errorMode) {
    if (errorHeight < 0.1) {
      errorHeight = 0.1;
    }
    double topWidth = 0.1; // height of top "hat plate"
    writeBox(os, 
	     Vector3D(currentHeight - (0.5 * errorHeight) + (0.5 *topWidth), pos.y(), pos.z()),
	     Vector3D(errorHeight - topWidth, 0.5, 0.5), errorColor, ' ');
     writeBox(os, 
 	     Vector3D(currentHeight - errorHeight - (0.5*topWidth), pos.y(), pos.z()),
 	     Vector3D(topWidth, 1.0, 1.0), errorColor, true);
  }

}


void
JVXMatrixWriter::writeReadingFrame(ostream& os,
				   const Alignment& ali,
				   int pos) const
{
  string slice = ali.getSliceString(pos);
  Vec<double> frequencies = getFrequencies( slice, alphabet);
  Vec<Vector3D> colors(3);
  colors[0] = Vector3D(0.99, 0, 0);
  colors[1] = Vector3D(0, 0.99, 0);
  colors[2] = Vector3D(0, 0, 0.99);
  // double inf = scorer.singleEntropy6(slice);
  // cout << "Writing logo position " << pos << " " << colors.size() << endl;
  int rf = pos % 3;
  double height = (1 + rf);
  writeBox(os, Vector3D(pos, -5.0 - (0.5*height) - 0.5, 
			0.0), Vector3D(1.0, height, 1.0), colors[rf]);
  writeBox(os, Vector3D(-5 - (0.5*height) - 0.5, pos, 0.0), 
	       Vector3D(height, 1, 1.0), colors[rf]);
}


void
JVXMatrixWriter::writeReadingFrame(ostream& os, const Alignment& ali) const
{
#ifdef VRML2MATRIXWRITER_VERBOSE
  cout << "starting writeReadingFrame!" << endl;
#endif
  string comment = "starting to write reading frame";
  writeComment(os, comment);
  for (int i = 0; i < static_cast<int>(ali.getLength()); ++i) {
    writeReadingFrame(os, ali, i);
  }
}


void
JVXMatrixWriter::writeMatching(ostream& os,
			       const Alignment& ali,
			       int pos) const
{
  string slice = ali.getSliceString(pos);
  ERROR_IF(slice.size() == 0, "Internal error in line 681!", exception);
  unsigned int gapCount = 0;
  for (unsigned int i = 0; i < slice.size(); ++i) {
    if (isGap(slice[i])) {
      ++gapCount;
    }
  }
  double frac = gapCount / static_cast<double>(slice.size());
  const Vector3D brown = Vector3D(102.0/256.0, 51.0/256.0, 0.0);
  double height = frac * stretchZ;
  if (height >= gapLimit) {
    writeBox(os, Vector3D(pos, ali.getLength() + 5.0, 0.5* height), 
		 height, brown);
    writeBox(os, Vector3D(ali.getLength() + 5.0, pos, 0.5* height), 
		 height, brown);
  }
}

void
JVXMatrixWriter::writeMatching(ostream& os,
			       const Alignment& ali) const
{
  double szd = ali.getLength();
  const Vector3D black = Vector3D(0.0, 0.0, 0.0);
  double bw = 0.25;
  string comment = "starting to write fraction of gaps";
  writeComment(os, comment);
  writeBox(os, 
	       Vector3D(szd + 5, 0.5 * szd, stretchZ + 0.5),
	       Vector3D(bw, szd, bw),
	       black);
  writeBox(os, 
	       Vector3D(szd + 5, 0.5 * szd, - 0.5),
	       Vector3D(bw, szd, bw),
	       black);
  writeBox(os, 
	       Vector3D(0.5 * szd, szd + 5, stretchZ + 0.5),
	       Vector3D(szd, bw, bw),
	       black);
  writeBox(os, 
	       Vector3D(0.5 * szd, szd + 5 , - 0.5),
	       Vector3D(szd, bw, bw),
	       black);
	       
  for (int i = 0; i < static_cast<int>(ali.getLength()); ++i) {
    writeMatching(os, ali, i);
  }
}

/**
   initialize grid in JVX
*/
void
JVXMatrixWriter::init(ostream& os, unsigned int len) const
{
  Vector3D groundCol(0.3, 0.3, 0.3);
  primitiveCounter = 0;
  os << "<?xml version=\"1.0\" encoding=\"ISO-8859-1\" standalone=\"no\"?>" 
     << endl;
  os << "<!DOCTYPE jvx-model SYSTEM \"http://www.javaview.de/rsrc/jvx.dtd\">" 
     << endl;
  os << "<jvx-model>" << endl;
  os << "<meta generator=\"CorreLogo RNA 3D MODEL Generator v.1.00\"/>" << endl;

  os << "<geometries>" << endl;

  double arrowLen = 10.0;
  double arrowTipLen = 3.0;
  double arrowTipRadius = 1;
  unsigned int arrowN = 15;
  double arrowHalf = arrowLen * 0.5;
  double arrowSep = 10.0;
  if (logoMode == AlignColorConstants::LOGO_FLAT) {
    arrowSep = 3.0;
  }
  double step = 10.0;
  double borderLen = 10;
  double stepSum = 0.0;
  // double length = len;
  double step2 = step;
  double step2len = 9.0;
  // unsigned int counter = 0;
  double gw = 1.0; // 0.5;
  Vector3D gridColBorder(0.1, 0.1, 0.1);
  Vector3D gridColHigh(0.99, 0.99, 0);
  Vector3D gridColNormal(0.9, 0.9, 0.9);
  Vector3D gridCol(0.9, 0.9, 0.9);
  // Vector3D gridCol2(102.0/256, 1.0, 1.0);
  Vector3D tickColBorder(0.1, 0.1, 0.1);
  Vector3D tickColNormal(0.5, 0.5, 0.5);
  Vector3D tickColHigh(0.99, 0.99, 0);
  Vector3D zeroCol(0.9, 0.9, 0.9);
  Vector3D tickCol = tickColNormal;
  int numLegend = 10;
  Vec<double> legendFrequencies(numLegend, 1.0/static_cast<double>(numLegend)); // all 16 base pairs
  Vec<string> legendNames(numLegend);
  legendNames[0] = "AU";
  legendNames[1] = "CG";
  legendNames[2] = "GU";
  legendNames[3] = "AA";
  legendNames[4] = "CC";
  legendNames[5] = "GG";
  legendNames[6] = "UU";
  legendNames[7] = "AC";
  legendNames[8] = "AG";
  legendNames[9] = "CU";
  Vec<Vector3D> legendColors = getPairColorsMode(legendNames, 0.0);
  
  switch (legendMode) {
  case 0: // do nothing, no legend
    break;
  case 1:
    writeBoxStack(os,
		  Vector3D(0, 0, 0.0),
		  40.0, legendFrequencies, legendColors);
    break;
  }

  if (arrowMode) {
    writeBox(os,
	     Vector3D(arrowHalf, -arrowSep, 0.0),
	     Vector3D(arrowLen, 0.25, 0.25) , 
	     zeroCol );
    writeCone(os, Vector3D(arrowLen, -arrowSep, 0.0), arrowTipLen, arrowTipRadius, arrowN, Vector3D(0.0, 1.0, 0.0),
	      90.0*DEG2RAD, zeroCol);
    
    writeBox(os,
	     Vector3D(-arrowSep, arrowHalf, 0.0),
	     Vector3D(0.25, arrowLen, 0.25) , 
	     zeroCol );
    writeCone(os, Vector3D(-arrowSep, arrowLen, 0.0), arrowTipLen, arrowTipRadius, arrowN, Vector3D(1.0, 0.0, 0.0),
	    -90.0*DEG2RAD, zeroCol);
  }

  // x-axis:
  // was : 0.0 instead of -gw
  writeBox(os, Vector3D(0.5*len, -gw, -0.5*gw),
	   Vector3D(len, gw, gw), gridColBorder, true);

  // y-axis:
  // was : 0.0 instead of -gw
  writeBox(os, Vector3D(-gw, 0.5*len, -0.5*gw),
	   Vector3D(gw, len, gw), gridColBorder, true);

  if (groundPlateMode) {
    writeBox(os, Vector3D(0.5*len - 0.5*gw, 0.5 * len -0.5*gw, -1.5*gw),
	     Vector3D(len, len, 1.0*gw) , groundPlateColor, true);
  }

  // diagonal stripe:
//   writeBox(os,
// 	   Vector3D(0.5*len, 0.5 * len, -0.5*gw),
// 	   Vector3D(sqrt(2.0) * len, gw, 1.0*gw) , 
// 	   Vector3D(0.0, 0.0, 1.0),
// 	   45.0 * DEG2RAD,
// 	   zeroCol );
  // loop over x:
  stepSum = - step - 1;
  while (stepSum < len) {
    double stepSum2 = -step2 - 1;
    double lenPiece;
    // diagonal stripe:
    if (((stepSum+1) >= -0.01) && ((stepSum + 0.5*step) <= len)) {
      writeBox(os,
	       Vector3D(stepSum + (0.5*step), stepSum + (0.5*step), -0.5*gw),
	       Vector3D((sqrt(2.0) * step) - (3*gw), gw, gw) , 
	       Vector3D(0.0, 0.0, 1.0), // rotation around z-axis
	       45.0 * DEG2RAD,
	       zeroCol );
    }
    // loopy of y:
    while (stepSum2 < len) {
      int iy = static_cast<int>((stepSum2+step2)+0.1);
      if (iy  == 0) {
	stepSum2 += step2;
	continue;
	tickCol = tickColBorder;
	gridCol = gridColBorder;
	lenPiece = borderLen;
      }
      else if ((iy  % 100) != 0) {
	tickCol = tickColNormal;
	gridCol = gridColNormal;
	lenPiece = step2len;
      }
      else {
	tickCol = tickColHigh;
	gridCol = gridColHigh;
	lenPiece = step2len;
      }
      if (((stepSum + 0.5*step) <= len) && (stepSum2 + step2 <= len)
	  && ((stepSum + 0.5*step) >= 0)) {
	// write cube parallel to x - axis
	writeBox(os,
		 Vector3D(stepSum + 0.5*step, stepSum2 + step2, -0.5*gw - 0.01),
		 Vector3D(lenPiece, 1.0, gw), gridCol);
      }
      int ix = static_cast<int>((stepSum+step)+0.1);
      if (ix == 0) {
	stepSum2 += step2;
	continue;
	tickCol = tickColBorder;
	gridCol = gridColBorder;
	lenPiece = borderLen;
      }
      else if ((ix  % 100) != 0) {
	tickCol = tickColNormal;
	gridCol = gridColNormal;
	lenPiece = step2len;
      }
      else {
	tickCol = tickColHigh;
	gridCol = gridColHigh;
	lenPiece = step2len;
      }
      // write cube parallel to y - axis
      if (((stepSum + step) <= len) && (stepSum2 + 0.5 * step2 <= len)
	  && ((stepSum2 + 0.5*step2) >= 0)) {
	writeBox(os,
		 Vector3D(stepSum + step, stepSum2 + 0.5 * step2, -0.5*gw - 0.01),
		 Vector3D(1.0, lenPiece, gw), gridCol);
      }
      stepSum2 += step2;
    }
    stepSum += step;
  }
}

void
JVXMatrixWriter::finalize(ostream& os) const
{
  os << endl << "</geometries>" << endl;
  os << "</jvx-model>" << endl;
}

/** writes complete document with possible reference structures and given matrix */
void
JVXMatrixWriter::writeVrml(ostream& os,
			   const Alignment& ali,
			   const Vec<Vec<double> >& matrix,
			   const Vec<Vec<double> >& errorMatrix,
			   const Vec<double>& infVec,
			   const Vec<double>& infErrorVec,
			   const Vec<Stem>& stems) const
{
  // cout << "Starting writeVrml(2) from JVXMatrixWriter!" << endl;
  // cout << "Defined scorer alphabets with JVXMatrixWriter::writeVrml(2): " 
  // << alphabet << endl;
  init(os, ali.getLength());
  switch (logoMode) {
  case AlignColorConstants::LOGO_NONE:
    break; // do nothing
  case AlignColorConstants::LOGO_DEFAULT:
    writeLogos(os, ali, infVec, infErrorVec);
    break;
  case AlignColorConstants::LOGO_FLAT:
    writeLogosFlat(os, ali, infVec, infErrorVec);
    break;
  default:
    ERROR("Unknown logo mode!", exception);
  }

  if (gapMode != AlignColorConstants::LOGO_NONE) {
    writeMatching(os, ali);
  }

  if (readingFrameMode && (logoMode != AlignColorConstants::LOGO_FLAT)) {
    writeReadingFrame(os, ali);
  }

  unsigned int counter = 0;
  string comment = "starting to write matrix data";
  writeComment(os, comment);
  for (unsigned int i = 0; i < ali.getLength(); ++i) {
    for (unsigned int j = i + 1; j < ali.getLength(); ++j) {
      string s1 = ali.getSliceString(i);
      string s2 = ali.getSliceString(j);
      double mutInf = matrix[i][j];
      if (mutInf <= 0.0) {
	mutInf = 0.01; // apply minimum thickness
      }
      if (writePosStack(os, ali, stems, i, j, mutInf, errorMatrix[i][j], 
			letterMode2)) {
 	++counter;
      }
      mutInf= matrix[j][i];
      if (mutInf <= 0.0) {
	mutInf = 0.01; // apply minimum thickness
      }
      if (writePosStack(os, ali, stems, 
			j, i, mutInf, errorMatrix[j][i], letterMode2)) {
 	++counter;
      }
    }
  }
  
  Vector3D refColor(0.0, 0.0, 0.0);

  // draw reference structure on top:
  //   writeComment(os, "Writing reference structure!");
  //   for (int j = 0; j < static_cast<int>(stems.size()); ++j) {
  //     for (int k = 0; k < (stems[j]).getLength(); ++k) {
  //       int posStart = stems[j].getStart()+k;
  //       int posStop = stems[j].getStop()-k;
  //       double matrixHeight = matrix[posStart][posStop] + refHeight;
  //       writeBox(os, Vector3D(static_cast<double>(posStart), static_cast<double>(posStop), matrixHeight),
  // 		   refHeight, refColor);
  //       writeBox(os, Vector3D(static_cast<double>(posStop), static_cast<double>(posStart), matrixHeight),
  // 		   refHeight, refColor);
  //       ++counter;	
  //     }
  //   }
  //   writeComment(os, "End of writing reference structure!");  
  
  // os << "# end of writing matrix data" << endl;
  finalize(os);
}

/* MODIFIERS */

/* copy method */
void 
JVXMatrixWriter::copy(const JVXMatrixWriter& other)
{
  AbstractMatrixWriter::copy(other);
  letters3D = other.letters3D;
  groundPlateMode = other.groundPlateMode;
  readingFrameMode = other.readingFrameMode;
  boxDivideNum = other.boxDivideNum;
  boxZMin = other.boxZMin;
  dxLetterFlat = other.dxLetterFlat; // = 0.1;
  dyLetterFlat = other.dyLetterFlat; // = -0.6;
  dxLetterFlat2 = other.dxLetterFlat2; // = 0.1;
  dyLetterFlat2 = other.dyLetterFlat2; // = -0.6;
  dxLetterLeft = other.dxLetterLeft; // = -0.2;
  dyLetterLeft = other.dyLetterLeft; //  = 0.3;
  gapLimit = other.gapLimit;
  infLimit = other.infLimit;
  letterHeight = other.letterHeight;
  letterWidth = other.letterWidth;
  referenceGap = other.referenceGap;
  stretchZ = other.stretchZ;
  sequenceType = other.sequenceType;
  writerParameters = other.writerParameters;
  resources = other.resources;
}

