// --*- C++ -*------x---------------------------------------------------------
// $Id: Vrml2MatrixWriter.cc,v 1.18 2005/11/10 16:11:25 bindewae Exp $
//
// Class:           Vrml2MatrixWriter
// 
// Base class:      VrmlMatrixWriterBase
//
// Derived classes: - 
//
// Author:          Eckart Bindewald
//
// Project name:    -
//
// Date:            $Date: 2005/11/10 16:11:25 $
//
// Description:     - 
// -----------------x-------------------x-------------------x-----------------

#include <Vrml2MatrixWriter.h>
// #include <vectornumerics.h>
#include <aligncolor_help.h>
#include <stemhelp.h>
#include <StringTools.h>
#include <colorFunctions.h>
#include <Letter3D.h>
#include <AlignColorConstants.h>

// #define VRML2MATRIXWRITER_VERBOSE

// ---------------------------------------------------------------------------
//                                   Vrml2MatrixWriter
// -----------------x-------------------x-------------------x-----------------

/* CONSTRUCTORS */

/* default constructor */
Vrml2MatrixWriter::Vrml2MatrixWriter() : readingFrameMode(false),  
					 
					 dxLetterFlat(0.25),dxLetterFlat2(0.4), dxLetterFront(-0.25), dxLetterLeft(-0.51),
					 dyLetterFlat(0.4),dyLetterFlat2(-0.25), dyLetterFront(-0.51),  dyLetterLeft(0.3), dzLetter(-0.4),
					 gapLimit(1),
					 letterHeight(0.5), letterLineThickness(0.1), sequenceType("RNA")
  // colorMode(2), logoMode(AlignColorConstants::LOGO_NONE),frequencyLimit(0.05), infLimit(0.5), stretchZ(10.0), alphabet("ACGU"), 
{
}

/* copy constructor */
Vrml2MatrixWriter::Vrml2MatrixWriter(const Vrml2MatrixWriter& other)
{
  copy(other);
}

/* destructor */
Vrml2MatrixWriter::~Vrml2MatrixWriter() { }


/* OPERATORS */

/** Assigment operator. */
Vrml2MatrixWriter& 
Vrml2MatrixWriter::operator = (const Vrml2MatrixWriter& orig)
{
  if ((&orig) != this) {
    copy(orig);
  }
  return *this;
}

ostream& 
operator << (ostream& os, const Vrml2MatrixWriter& rval)
{
  ERROR("Ouput operator not yet implemented!", exception);
  return os;
}

istream& 
operator >> (istream& is, Vrml2MatrixWriter& rval)
{
  ERROR("Input operator not yet implemented!", exception);
  return is;
}

/* PREDICATES */

/*
Vector3D
Vrml2MatrixWriter::getResourceColor(const string& name) const
{
  Vector3D result(0.0, 0.0, 0.0);
  string s;
  double  x = 0;
  double  y = 0;
  double  z = 0;
  string trans=resources.getString(name);
  
  vector<string> words = getTokens(trans);
  // cout << "Found number strings: " << name << endl;
//   for (unsigned int i = 0; i < words.size(); ++i) {
//     cout << words[i] << " ";
//   }
  if (words.size() != 3) {
    cout << name << " " << trans << " " << words.size() << endl;
    ERROR_IF(words.size() != 3,
	     "Three values for red green blue expected in color definition!",
	     exception);
  }
  x = stod(words[0]); // convert to double
  y = stod(words[1]); // convert to double
  z = stod(words[2]); // convert to double

//   cout << "The resource int is: " << " "
//        << x << " " << y << " " << z << endl;
  result.x(x);
  result.y(y);
  result.z(z);

  // cout << "The resource color is: " << result << endl;
  return result; 
}
*/

/** central method for writing single box
 faceMode 0: write in x-z plane, 1: write in y-z plane */
void
Vrml2MatrixWriter::writeLetterText(ostream& os,
				   char c,
				   const Vector3D& pos,
				   const Vector3D sizeOrig,
				   const Vector3D& col,
				   int faceMode) const
{
  string comment = "Starting writeLetterText: " + c;
  writeComment(os, comment);
  // change with faceMode:
  Vector3D rot(1.0, 0.0, 0.0);
  double angle = 90.0 * DEG2RAD;
  double maxExtent = 0.0;
  if (faceMode == Letter3D::FACE_YZ) {
    rot = Vector3D(0.58, -0.58, -0.58);
    angle = 2.09;
    maxExtent = sizeOrig.y();
    if (sizeOrig.z() < maxExtent) {
      maxExtent = sizeOrig.z();
    }
  }
  else if (faceMode == Letter3D::FACE_XZ) {
    maxExtent = sizeOrig.x();
    if (sizeOrig.z() < maxExtent) {
      maxExtent = sizeOrig.z();
    }    
  }
  else if (faceMode == Letter3D::FACE_XY180) {
    rot = Vector3D(0, 0, 1);
    angle = 180.0 * DEG2RAD;
    maxExtent = sizeOrig.y();
    if (sizeOrig.z() < maxExtent) {
      maxExtent = sizeOrig.z();
    }
  }
  else if (faceMode == Letter3D::FACE_XY90) {
    rot = Vector3D(0, 0, 1);
    angle = 90.0 * DEG2RAD;
    maxExtent = sizeOrig.y();
    if (sizeOrig.z() < maxExtent) {
      maxExtent = sizeOrig.z();
    }
  }

  else {
    ERROR("Unknown face!", exception);
  }
  int fontSize = static_cast<int>(maxExtent);
  if (fontSize <= 0) {
    fontSize = 1;
  }
  os << "," << endl
     << "\tTransform {" << endl
     << "\t\ttranslation " << pos.x() << " " << pos.y() << " " << pos.z()
     << "\t\trotation " << rot.x() << " " << rot.y() << " " << rot.z() 
     << " " << angle << endl
    // << "\tscale " << sizeOrig.x() << " " << sizeOrig.y() << " " << sizeOrig.z() << endl
     << "\t\tchildren Shape {" << endl
     << "\t\t\tappearance Appearance {" << endl
     << "\t\t\t\tmaterial Material { diffuseColor " << col.x() << " "
     << col.y() << " " << col.z() << " }" << endl
     << "}" << endl
     << "\t\t\tgeometry Text {" << endl
     << "\t\t\t\tstring [ \"" << c << "\" ]" << endl
     << "\t\t\t\tmaxExtent " << maxExtent << endl
    // << "\t\t\t\tlength  [ " << sizeOrig.x() << ", " << sizeOrig.z() << " ]" << endl
     << "fontStyle FontStyle { size " << fontSize << " }" << endl
     << "\t\t\t}" << endl
     << "\t\t}" << endl
     << "\t}" << endl;  
  ++primitiveCounter;
  /*
    for (Vec<Letter3D::point_type>::size_type i = 0; i < points.size(); ++i) {
    switch (faceMode) {
    case Letter3D::FACE_XZ: // write in x-z plane
    writePoint(os, pos.x() + (points[i].first * dx), pos.y(), 
    pos.z() + points[i].second * dz);
    break;
    case Letter3D::FACE_YZ:  // write in y-z plane
    writePoint(os, pos.x(), pos.y() - (points[i].first * dx), 
    pos.z() + points[i].second * dz);
    break;
    default: ERROR("Internal error: Unknown face mode!", exception);
    }
  }
  */

}

/** write coordinates of 3D point */
void
Vrml2MatrixWriter::writeCoord(ostream& os, const Vector3D& p) const {
  os << p.x() << " " << p.y() << " " << p.z();
}

void
Vrml2MatrixWriter::computeLineBorderPointsXZ(const Vector3D& p1, 
					     const Vector3D& p2,
					     Vector3D& pp0, Vector3D& pp1, Vector3D& pp2, Vector3D& pp3,
					     double thickness) const
{
  PRECOND(thickness > 0.0, exception);
  Vector3D dp = p1 - p2;
  double dtz = dp.z();
  double dtx = dp.x();
  double dx, dz;
  if (dtz == 0) {
    dx = 0;
    dz = thickness;
  }
  else if (dtx == 0) {
    dx = thickness;
    dz = 0;
  }
  else {
    dx = thickness / sqrt(1.0 + ((dtx*dtx)/ (dtz*dtz) ) );
    dz = dx * dtx / dtz;
  }
  pp0.x(p1.x() + dx);
  pp0.y(p1.y());
  pp0.z(p1.z() - dz);

  pp1.x(p1.x() - dx);
  pp1.y(p1.y());
  pp1.z(p1.z() + dz);

  pp2.x(p2.x() - dx);
  pp2.y(p2.y());
  pp2.z(p2.z() + dz);

  pp3.x(p2.x() + dx);
  pp3.y(p2.y());
  pp3.z(p2.z() - dz);
  
}

void
Vrml2MatrixWriter::computeLineBorderPointsYZ(const Vector3D& p1, 
					     const Vector3D& p2,
					     Vector3D& pp0, Vector3D& pp1, Vector3D& pp2, Vector3D& pp3,
					     double thickness) const
{
  Vector3D dp = p1 - p2;
  double dtz = dp.z();
  double dty = dp.y();
  double dy, dz;
  if (dtz == 0.0) {
    dy = 0.0;
    dz = thickness;
  }
  else if (dty == 0) {
    dy = thickness;
    dz = 0;
  }
  else {
    dy = thickness / sqrt(1.0 + ((dty*dty)/ (dtz*dtz) ) );
    dz = dy * dty / dtz;
  }
  pp0.x(p1.x());
  pp0.y(p1.y() + dy);
  pp0.z(p1.z() - dz);

  pp1.x(p1.x());
  pp1.y(p1.y() - dy);
  pp1.z(p1.z() + dz);

  pp2.x(p2.x());
  pp2.y(p2.y() - dy);
  pp2.z(p2.z() + dz);

  pp3.x(p2.x());
  pp3.y(pp2.y() + dy);
  pp3.z(pp2.z() - dz);
}

/** approximates lines with single face
  @see http://www.cs.vu.nl/~eliens/documents/vrml/reference/BOOK.HTM 
*/
void
Vrml2MatrixWriter::writeLine(ostream& os, const Vector3D& p1, const Vector3D& p2, const Vector3D& col,
			     double thickness, int faceMode) const
{
  // compute 4 points:
  Vector3D pp0, pp1, pp2, pp3;
  if (faceMode == Letter3D::FACE_XZ) {
    computeLineBorderPointsXZ(p1, p2, pp0, pp1, pp2, pp3, thickness);
  }
  else if (faceMode == Letter3D::FACE_YZ) {
    computeLineBorderPointsYZ(p1, p2, pp0, pp1, pp2, pp3, thickness);
  }
  else {
    ERROR("Undefined face mode!", exception);
  }
  os << "," << endl 
     << "Transform {" << endl
    // translation 1.5 -1.5 0
     << "children Shape {" << endl
     << "appearance Appearance { material Material { } }" << endl
    // appearance USE A  # Use same dflt material as dodecahedron
     << "geometry IndexedFaceSet {" << endl
     << "coord Coordinate {" << endl
     << " point [" << endl;
  writeCoord(os, pp0);
  os << ", "; // wri"1 1 1, 1 -1 -1, -1 1 -1, -1 -1 1," << endl
  writeCoord(os, pp1);
  os << ", "; 
  writeCoord(os, pp2);
  os << ", "; 
  writeCoord(os, pp3);
  primitiveCounter += 4;
  // os << ", "; 
  os << "]" << endl
     << "}" << endl
     << "coordIndex [" << endl
     << "0 1 2 3" << endl
     << "]" << endl
     << " color Color {" << endl
     << "  color [ ";
  writeCoord(os, col);
  os << " ]" << endl
     << "}" << endl;
  os << "colorPerVertex FALSE" << endl;
  // os << "convex FALSE" << endl;
  os << "solid TRUE" << endl;
    // # Leave colorPerVertex field set to TRUE.
    // # And no indices are needed, either-- each coordinate point
    // # is assigned a color (or, to think of it another way, the same
    // # indices are used for both coordinates and colors).
  os << "}" << endl /* IndexedFaceSet */
     << "}" << endl /* shape */
     << "}" << endl; // Transform
}

/** approximates lines with single face
  @see http://www.cs.vu.nl/~eliens/documents/vrml/reference/BOOK.HTM 
*/
void
Vrml2MatrixWriter::writeLine(ostream& os, const Vec<Vector3D>& points, const Vector3D& col,
			     double thickness, int faceMode) const
{
  // compute 4 points:
  Vec<Vector3D>::size_type nn = points.size();
  Vec<Vector3D> pp0(nn); // stores left and reight neighbor points for each pont
  Vec<Vector3D> pp1(nn); //, pp2, pp3;
  Vector3D help;
  for (Vec<Vector3D>::size_type i = 0; (i+1) < points.size(); ++i) {
    if (faceMode == Letter3D::FACE_XZ) {
      computeLineBorderPointsXZ(points[i], points[i+1], pp0[i], pp1[i], pp0[i+1], pp1[i+1], thickness);
      help = pp0[i+1]; // swap
      pp0[i+1] = pp1[i+1];
      pp1[i+1] = help;
    }
    else if (faceMode == Letter3D::FACE_YZ) {
      computeLineBorderPointsYZ(points[i], points[+1], pp0[i], pp1[i], pp0[i+1], pp1[i+1], thickness);
      help = pp0[i+1];
      pp0[i+1] = pp1[i+1];
      pp1[i+1] = help;
    }
    else {
      ERROR("Undefined face mode!", exception);
    }
  }
  os << "," << endl 
     << "Transform {" << endl
    // translation 1.5 -1.5 0
     << "children Shape {" << endl
     << "appearance Appearance { " << endl
     << "  material Material { } " << endl;
  os << "}" << endl  
     << "geometry IndexedFaceSet {" << endl
     << "coord Coordinate {" << endl
     << " point [" << endl;
  for (Vec<Vector3D>::size_type i = 0; i < pp0.size(); ++i) {
    writeCoord(os, pp0[i]);
    os << ", "; // wri"1 1 1, 1 -1 -1, -1 1 -1, -1 -1 1," << endl
    ++primitiveCounter;
  }
  // write other side backwards:
  for (Vec<Vector3D>::size_type i = 0; i < pp0.size(); ++i) {
    writeCoord(os, pp1[pp1.size() - i -1]);
    os << ", "; // wri"1 1 1, 1 -1 -1, -1 1 -1, -1 -1 1," << endl
    ++primitiveCounter;
  }
  // os << ", "; 
  os << "]" << endl
     << "}" << endl
     << "coordIndex [" << endl;
  unsigned int lineSize =  2 * pp1.size();
  for (unsigned int i = 0; i < lineSize; ++i) {
    os << i << " ";
  }
  os << "]" << endl
     << " color Color {" << endl
     << "  color [ ";
  writeCoord(os, col);
  os << " ]" << endl
     << "}" << endl;
  os << "colorPerVertex FALSE" << endl;
  // os << "convex FALSE" << endl;
  os << "solid TRUE" << endl;
    // # Leave colorPerVertex field set to TRUE.
    // # And no indices are needed, either-- each coordinate point
    // # is assigned a color (or, to think of it another way, the same
    // # indices are used for both coordinates and colors).
  os << "}" << endl /* IndexedFaceSet */
     << "}" << endl /* shape */
     << "}" << endl; // Transform
}

/** approximates lines with single face
  @see http://www.cs.vu.nl/~eliens/documents/vrml/reference/BOOK.HTM 
*/
void
Vrml2MatrixWriter::writeFace(ostream& os, const Vec<Vector3D>& points, 
			     const Vector3D& col,
			     string imageUrl, int faceMode) const
{
  // compute 4 points:
  os << "," << endl 
     << "Transform {" << endl
    // translation 1.5 -1.5 0
     << "children Shape {" << endl
     << "appearance Appearance { " << endl
     << "  material Material { } " << endl;
  if (imageUrl.size() > 0) {
    os << "  texture ImageTexture { url [ \"" << imageUrl << "\" ] " << endl
       << "   repeatS FALSE" << endl
       << "   repeatT FALSE" << endl
       << "  }" << endl;
    os << "  textureTransform    NULL" << endl;
  }
  os << "}" << endl  
     << "geometry IndexedFaceSet {" << endl
     << "coord Coordinate {" << endl
     << " point [" << endl;
  for (Vec<Vector3D>::size_type i = 0; i < points.size(); ++i) {
    writeCoord(os, points[i]);
    os << ", "; // wri"1 1 1, 1 -1 -1, -1 1 -1, -1 -1 1," << endl
    ++primitiveCounter;
  }
  // os << ", "; 
  os << "]" << endl
     << "}" << endl
     << "coordIndex [" << endl;
  for (Vec<Vector3D>::size_type i = 0; i < points.size(); ++i) {
    os << i << " ";
  }
  os << "]" << endl
     << " color Color {" << endl
     << "  color [ ";
  writeCoord(os, col);
  os << " ]" << endl
     << "}" << endl;
  os << "colorPerVertex FALSE" << endl;
  // os << "convex FALSE" << endl;
  os << "solid TRUE" << endl;
    // # Leave colorPerVertex field set to TRUE.
    // # And no indices are needed, either-- each coordinate point
    // # is assigned a color (or, to think of it another way, the same
    // # indices are used for both coordinates and colors).
  os << "}" << endl /* IndexedFaceSet */
     << "}" << endl /* shape */
     << "}" << endl; // Transform
}

/** central method for writing single box
 faceMode 0: write in x-z plane, 1: write in y-z plane */
void
Vrml2MatrixWriter::writeLetterLines(ostream& os,
				    char c,
				    const Vector3D& posOrig,
				    const Vector3D sizeOrig,
				    const Vector3D& col,
				    double thickness,
				    int faceMode) const
{
  string comment  = "Starting writeLetterLines: " + c;
  writeComment(os, comment);
  // first try to find letter in read alphabet:
  Vec<Letter3D>::size_type idx = letters3D.size();
  for (Vec<Letter3D>::size_type i = 0; i < letters3D.size(); ++i) {
    if (c == letters3D[i].getLetter()) {
      idx = i;
      break;
    }
  }
  if (idx >= letters3D.size()) {
    cout << "Warning: could not find bitmap for letter " << c << endl;
    return; // letter not found
  }
  const Letter3D& l3d = letters3D[idx];
  // cout << "Using letter: " << l3d << endl;
  // const Vec<Vec<int> >& bitMap = l3d.getMatrix();
  Vector3D size = sizeOrig; // Vector3D(1.0, 1.0, 1.0);
  // Vector3D pos = posOrig - (size * 0.5); // orig: middle, pos: lower left corner
  double dx = size.x() / l3d.getNumberColumns();
  // double dy = size.y() * 0.5;
  double dz = size.z() / l3d.getNumberRows();
  Vec<Letter3D::point_type> points = l3d.getPoints();
  Vec<Letter3D::line_type> lines = l3d.getLines();

  ERROR_IF(dx < 0.001, "Internal error in line 448!", exception);
  ERROR_IF(dz < 0.001, "Internal error in line 449!", exception);

//     case Letter3D::FACE_XZ: // write in x-z plane
//       writePoint(os, pos.x() + (points[i].first * dx), pos.y(), 
// 		 pos.z() + points[i].second * dz);
//       break;
//     case Letter3D::FACE_YZ:  // write in y-z plane
//       writePoint(os, pos.x(), pos.y() - (points[i].first * dx), 
// 		 pos.z() + points[i].second * dz);
  for (Vec<Letter3D::line_type>::size_type i = 0; i < lines.size(); ++i) {
    Vec<Vector3D> pvec(lines[i].size());
    for (Letter3D::line_type::size_type j = 0; j < lines[i].size(); ++j) {
      // os << "  <l>" << lines[i][j] << " " << lines[i][j+1] << "</l>" << endl;
      const Letter3D::point_type p1 = points[lines[i][j]];
      // const Letter3D::point_type p2 = points[lines[i][j+1]];
      Vector3D pos1 = posOrig; //
      // Vector3D pos2 = posOrig; // points[lines[i][j]];
      switch (faceMode) {
      case Letter3D::FACE_XZ:
	pos1.x(pos1.x() + p1.first * dx);
	pos1.z(pos1.z() + p1.second * dz);
	// pos2.x(pos2.x() + p2.first * dx);
	// pos2.z(pos2.z() + p2.second * dz);
	break;
      case Letter3D::FACE_YZ:
	pos1.y(pos1.y() - p1.first * dx);
	pos1.z(pos1.z() + p1.second * dz);
	// pos2.y(pos2.y() - p2.first * dx);
	// pos2.z(pos2.z() + p2.second * dz);
	break;
      default:
	ERROR("Unknown face mode!", exception);
      }
      // ERROR_IF((pos1-pos2).length() < 0.01, "Line between equal points!", exception);
      pvec[j] = pos1;
    }
    writeLine(os, pvec, col , thickness, faceMode);
  }
//   os << "   <color>" << col.x() << " " << col.y() << " " << col.z() 
//      << "</color>" << endl;
//   os << "  <thickness> 4 </thickness>" << endl;
//   os << "  </lines>" << endl;
//   os << " </lineSet>" << endl;
//   os << "</geometry>" << endl;
  writeComment(os, "Ending writeLetterLines!");
}

/** returns url of image */
string
Vrml2MatrixWriter::getImageUrl(char letter, const string& extra) const
{
  string result = wwwImageBase + letter + extra + ".jpg";
  //   case 'T': os << "url	[ \"" << wwwImageBase << "T" << extra << ".jpg\" ]" << endl;
  //     break;
  return result;
}

/** central method for writing single box
 faceMode 0: write in x-z plane, 1: write in y-z plane */
void
Vrml2MatrixWriter::writeLetterTexture(ostream& os,
				      char letter,
				      const Vector3D& posOrig,
				      const Vector3D sizeOrig,
				      const Vector3D& col,
				      int faceMode) const
{
  string comment  = "Starting writeLetterTexture: " + letter;
  writeComment(os, comment);
  // compute 4 points:
  Vector3D half = sizeOrig * 0.5;
  Vec<Vector3D> points(4, posOrig);
  string extra;
  Vector3D newSize(sizeOrig);
  if (faceMode == Letter3D::FACE_XZ) {
    points[3] += Vector3D(-half.x(), 0.0, -half.z());  // lower left
    points[2] += Vector3D(-half.x(), 0.0, half.z());  // upper left
    points[1] += Vector3D(half.x(), 0.0, half.z());  // upper right
    points[0] += Vector3D(half.x(), 0.0, -half.z());  // lower right
    newSize.y(0.05);
  }
  else if (faceMode == Letter3D::FACE_YZ) {
    points[3] += Vector3D(0.0, +half.y(), -half.z());  // lower left
    points[2] += Vector3D(0.0, +half.y(), half.z());  // upper left
    points[1] += Vector3D(0.0, -half.y(), half.z());  // upper right
    points[0] += Vector3D(0.0, -half.y(), -half.z());  // lower right
    extra = "r"; // workaround to address different image
    newSize.x(0.05);
  }
  string imageUrl = getImageUrl(letter, extra);
  // writeFace(os, points, col , imageUrl, faceMode);
  writeVrmlBox(os, posOrig, newSize, col, imageUrl);
  writeComment(os, "Ending writeLetterTexture!");
}


/** central method for writing single box
 faceMode 0: write in x-z plane, 1: write in y-z plane */
void
Vrml2MatrixWriter::writeLetter(ostream& os,
			       char c,
			       const Vector3D& pos,
			       const Vector3D sizeOrig,
			       const Vector3D& col,
			       int faceMode,
			       int letterStyle) const
{
  writeComment(os, "Starting writeLetter!");
  if (c == ' ') {
    return;
  }
  switch (letterStyle) {
  case LETTER_MODE_TEXT:
    writeLetterText(os, c, pos, sizeOrig, col, faceMode);    break;
  case LETTER_MODE_LINES:
    writeLetterLines(os, c, pos, sizeOrig, col, letterLineThickness, faceMode);
    break;
  case LETTER_MODE_TEXTURE:
    writeLetterTexture(os, c, pos, sizeOrig, col, faceMode);
    break;
  case LETTER_MODE_NONE:
    break;
  default: ERROR("Unknown letter mode!", exception);
  }
  writeComment(os, "Ending writeLetter!");
}

/** writes box with letter as texture map.
 The string extra qualifies a potential different image. Defined so far: empty and "r" for 90 right rotated images */
void
Vrml2MatrixWriter::writeBoxLetterBoth(ostream& os,
				      const Vector3D& pos,
				      double height, 
				      const Vector3D& col,
				      char letterFront,
				      char letterLeft,
				      int letterStyle) const
{
  string comment = "Starting writeBoxLetterBoth!";
  writeComment(os, comment);
  if (height <= 0.0) {
    return;
  }
  writeVrmlBox(os, pos, height, col);
  Vector3D black(0.1, 0.1, 0.1);
  if (height > letterHeight) {
    if (letterFront != ' ') {
      writeLetter(os, letterFront, 
		  Vector3D(pos.x() + dxLetterFront, 
			   pos.y() + dyLetterFront, 
			   pos.z() + dzLetter), 
		  Vector3D(1.0, 1.0, 0.5 * height), black,
		  Letter3D::FACE_XZ, letterStyle);
    }
    if (letterLeft != ' ') {
      writeLetter(os, letterLeft, 
		  Vector3D(pos.x() + dxLetterLeft,
			   pos.y() + dyLetterLeft, 
			   pos.z() + dzLetter), 
		  Vector3D(1.0, 1.0, 0.5 * height), black,
		  Letter3D::FACE_YZ, letterStyle);  
    }
  }
  else {
    comment = "did not write letter because too small: " 
      + dtos(height) + " " + dtos(letterHeight);
    writeComment(os, comment);
  }
}

/** writes cone used for arrow in initialization */
void
Vrml2MatrixWriter::writeVrmlCone(ostream& os,
				 const Vector3D& pos,
				 const Vector3D& rot,
				 double angle,
				 const Vector3D& col ) const
{
  os << "," << endl
     << "\tTransform {" << endl
     << "\t\ttranslation " << pos.x() << " " << pos.y() << " " << pos.z()
     << "\t\trotation " << rot.x() << " " << rot.y() << " " << rot.z() 
     << " " << angle
     << endl
     << "\t\tchildren Shape {" << endl
     << "\t\t\tappearance Appearance {" << endl
     << "\t\t\t\tmaterial Material { diffuseColor " << col.x() << " "
     << col.y() << " " << col.z() << " }" << endl
     << "}" << endl
     << "\t\t\tgeometry Cone {" << endl
     << "\t\t\t}" << endl
     << "\t\t}" << endl
     << "\t}" << endl;
  ++primitiveCounter;
}

/** writes box with letter as texture map.
 The string extra qualifies a potential different image. Defined so far: empty and "r" for 90 right rotated images */
void
Vrml2MatrixWriter::writeVrmlBoxLetter(ostream& os,
				      const Vector3D& pos,
				      double height, 
				      const Vector3D& col,
				      char letter,
				      int faceMode,
				      int letterStyle) const
{
  
  if (height <= 0.0) { // || (wwwImageBase.size() == 0)) {
    return;
  }
  Vector3D letterCol(0.1, 0.1, 0.1);
  double width = 1.0;
  double width2 = 0.5 * width;
  /* 
     os << "texture        ImageTexture {" << endl;
     switch (letter) {
     case 'A': os << "url	[ \"" << wwwImageBase << "A" << extra << ".jpg\" ]" << endl;
     break;
     case 'C': os << "url	[ \"" << wwwImageBase << "C" << extra << ".jpg\" ]" << endl;
     break;
     case 'G': os << "url	[ \"" << wwwImageBase << "G" << extra << ".jpg\" ]" << endl;
     break;
     case 'U': os << "url	[ \"" << wwwImageBase << "U" << extra << ".jpg\" ]" << endl;
     break;
     case 'T': os << "url	[ \"" << wwwImageBase << "T" << extra << ".jpg\" ]" << endl;
     break;
     default: cerr << "Bad letter found: " << letter << endl;
     ERROR("Letter not supported by texture mapping.", exception);
     }
     os << "}" << endl	
  */

  os << "# start of writing box with letter: " << letter << endl;
  os << "," << endl
     << "\tTransform {" << endl
     << "\t\ttranslation " << pos.x() << " " << pos.y() << " " << pos.z()
     << endl
     << "\t\tchildren Shape {" << endl
     << "\t\t\tappearance Appearance {" << endl
     << "\t\t\t\tmaterial Material { diffuseColor " 
     << col.x() << " " << col.y() << " " << col.z() << " }" << endl
     << "}" << endl;
  os << "\t\t\tgeometry Box {" << endl
     << "\t\t\tsize " << "1.0 1.0 " << height << endl
     << "\t\t\t}" << endl
     << "\t\t}" << endl
     << "\t}" << endl;
  ++primitiveCounter;
  if (height > letterHeight) {
    if (faceMode == Letter3D::FACE_XZ) {
      writeLetter(os, letter, Vector3D(pos.x(), 
				       pos.y() - width2 - 0.1, 
				       pos.z()), 
		  Vector3D(width, width, height), letterCol, faceMode, letterStyle);
    }
    else if (faceMode == Letter3D::FACE_YZ) {
      writeLetter(os, letter, 
		  Vector3D(pos.x() - width2 - 0.1, pos.y(), pos.z()), 
		  Vector3D(width, width, height), letterCol, faceMode, letterStyle);
    }
  }
  os << "# end of writing box with letter: " << letter << endl;
}

/** writes box with letter as texture map.
 The string extra qualifies a potential different image. Defined so far: empty and "r" for 90 right rotated images
 writes flat box in y direction */
void
Vrml2MatrixWriter::writeVrmlBoxLetterFlat(ostream& os,
					  const Vector3D& pos,
					  double height, 
					  const Vector3D& col,
					  char letter,
					  int faceMode,
					  int letterStyle) const
{
  
  if (height <= 0.0) { // || (wwwImageBase.size() == 0)) {
    return;
  }
  Vector3D letterCol(0.1, 0.1, 0.1);
  double width = 1.0;
  double width2 = 0.5 * width;
  
  os << "# start of writing box with letter: " << letter << endl;
  os << "," << endl
     << "\tTransform {" << endl
     << "\t\ttranslation " << pos.x() << " " << pos.y() << " " << pos.z()
     << endl
     << "\t\tchildren Shape {" << endl
     << "\t\t\tappearance Appearance {" << endl
     << "\t\t\t\tmaterial Material { diffuseColor " 
     << col.x() << " " << col.y() << " " << col.z() << " }" << endl
     << "}" << endl;
  os << "\t\t\tgeometry Box {" << endl
     << "\t\t\tsize " << "1.0 " << height << " 1.0" << endl
     << "\t\t\t}" << endl
     << "\t\t}" << endl
     << "\t}" << endl;
  ++primitiveCounter;
  if (height > letterHeight) {
    if (faceMode == Letter3D::FACE_XY180) {
      writeLetter(os, letter, 
		  Vector3D(pos.x() + dxLetterFlat, pos.y() + dyLetterFlat, pos.z() + width2 + letterLineThickness ), 
		  Vector3D(width, width, height), letterCol, faceMode, letterStyle);
    }
    else if (faceMode == Letter3D::FACE_XY90) {
      writeLetter(os, letter, 
		  Vector3D(pos.x() + dxLetterFlat2 , pos.y() + dyLetterFlat2, pos.z() + width2 + letterLineThickness), 
		  Vector3D(width, width, height), letterCol, faceMode, letterStyle);
    }
  }
  os << "# end of writing box with letter: " << letter << endl;
}

/** writes box with letter as texture map.
 The string extra qualifies a potential different image. Defined so far: empty and "r" for 90 right rotated images */
void
Vrml2MatrixWriter::writeVrmlBoxLetterFlat2(ostream& os,
					   const Vector3D& pos,
					   double height, 
					   const Vector3D& col,
					   char letter,
					   int faceMode,
					   int letterStyle) const
{
  
  if (height <= 0.0) { // || (wwwImageBase.size() == 0)) {
    return;
  }
  Vector3D letterCol(0.1, 0.1, 0.1);
  double width = 1.0;
  double width2 = 0.5 * width;

  os << "# start of writing box with letter: " << letter << endl;
  os << "," << endl
     << "\tTransform {" << endl
     << "\t\ttranslation " << pos.x() << " " << pos.y() << " " << pos.z()
     << endl
     << "\t\tchildren Shape {" << endl
     << "\t\t\tappearance Appearance {" << endl
     << "\t\t\t\tmaterial Material { diffuseColor " 
     << col.x() << " " << col.y() << " " << col.z() << " }" << endl
     << "}" << endl;
  os << "\t\t\tgeometry Box {" << endl
     << "\t\t\tsize " << height << " 1.0 1.0 " << endl
     << "\t\t\t}" << endl
     << "\t\t}" << endl
     << "\t}" << endl;
  ++primitiveCounter;
//   if (height > letterHeight) {
//     if (faceMode == Letter3D::FACE_XZ) {
//       writeLetter(os, letter, Vector3D(pos.x(), 
// 				       pos.y() - width2 - 0.1, 
// 				       pos.z()), 
// 		  Vector3D(width, width, height), letterCol, faceMode, letterStyle);
//     }
//     else if (faceMode == Letter3D::FACE_YZ) {
//       writeLetter(os, letter, 
// 		  Vector3D(pos.x() - width2 - 0.1, pos.y(), pos.z()), 
// 		  Vector3D(width, width, height), letterCol, faceMode, letterStyle);
//     }
//   }
  if (height > letterHeight) {
//     if (faceMode == Letter3D::FACE_XY180) {
//       writeLetter(os, letter, Vector3D(pos.x(), pos.y(), pos.z() + width2 + letterLineThickness ), 
// 		  Vector3D(width, width, height), letterCol, faceMode, letterStyle);
//     }
//     else if (faceMode == Letter3D::FACE_XY90) {
//       writeLetter(os, letter, 
// 		  Vector3D(pos.x() , pos.y(), pos.z() + width2 + letterLineThickness), 
// 		  Vector3D(width, width, height), letterCol, faceMode, letterStyle);
//     }
    if (faceMode == Letter3D::FACE_XY180) {
      writeLetter(os, letter, 
		  Vector3D(pos.x() + dxLetterFlat, pos.y() + dyLetterFlat, pos.z() + width2 + letterLineThickness ), 
		  Vector3D(width, width, height), letterCol, faceMode, letterStyle);
    }
    else if (faceMode == Letter3D::FACE_XY90) {
      writeLetter(os, letter, 
		  Vector3D(pos.x() + dxLetterFlat2 , pos.y() + dyLetterFlat2, pos.z() + width2 + letterLineThickness), 
		  Vector3D(width, width, height), letterCol, faceMode, letterStyle);
    }
  }
  os << "# end of writing box with letter: " << letter << endl;
}

void
Vrml2MatrixWriter::writeVrmlBoxStack(ostream& os,
				     const Vector3D& pos,
				     double height, 
				     const Vec<double>& frequencies,
				     const Vec<Vector3D>& colors ) const
{
  PRECOND(colors.size()==frequencies.size(), exception);
  double currentHeight = 0.0;
  // cout << "start writeVrmlBoxStack " << height << endl;
  // deterime ordering: 
  if (height <= 0.0) {
    cout << "height smaller zero: " << height << endl;
    return;
  }
  os << "# start of writeVrmlBoxStack" << endl;
  os << "Group {" << endl;
  os << "children [" << endl;
  Vec<unsigned int> order = documentedOrder(frequencies);
  // reverse(order.begin(), order.end()); // smallest frequency first
  for (unsigned int ii = 0; ii < colors.size(); ++ii) {
    unsigned int i = order[ii];
    ERROR_IF(frequencies[i] > 1.0, "Internal error in line 235!",
	     exception);
    if (frequencies[i] > frequencyLimit) {
      writeVrmlBox(os, Vector3D(pos.x(), pos.y(), 
			currentHeight + (0.5 * height * frequencies[i])),
		   height*frequencies[i], colors[i]);
    }
//     else {
//       cout << "Frequency too small: " << i << " " << frequencies[i] << endl;
//     }
    currentHeight += height * frequencies[i];
  }
  // os << "}" << endl; // end of "Group" statement
  os << "] # end of children for writeVrmlBoxStack" << endl;
  os << "} # end of writeVrmlBoxStack" << endl;


}

void
Vrml2MatrixWriter::writeVrmlBoxStack(ostream& os,
				     const Vector3D& pos,
				     double height, 
				     const Vec<double>& frequencies,
				     const Vec<Vector3D>& colors,
				     const string& letters,
				     int faceMode,
				     int letterStyle) const
{
  PRECOND(colors.size()==frequencies.size(), exception);
  // cout << "start writeVrmlBoxStack(2) " << height << " "
  // << frequencies << endl;
  // if (wwwImageBase.size() == 0) { // no texture mapping
  if (letterMode == LETTER_MODE_NONE) {
    writeVrmlBoxStack(os, pos, height, frequencies, colors);
    return;
  }
  double currentHeight = 0.0;
  // deterime ordering: 
  if (height <= 0.0) {
    cout << "height smaller zero: " << height << endl;
    return;
  }
  os << "# start of writeVrmlBoxStack" << endl;
  os << "Group {" << endl;
  os << "children [" << endl;
  Vec<unsigned int> order = documentedOrder(frequencies);
  // reverse(order.begin(), order.end()); smallest first
  for (unsigned int ii = 0; ii < colors.size(); ++ii) {
    unsigned int i = order[ii];
    ERROR_IF(frequencies[i] > 1.0, "Internal error in line 262!",
	     exception);
    if (frequencies[i] > frequencyLimit) {
	writeVrmlBoxLetter(os, Vector3D(pos.x(), pos.y(), 
			       currentHeight + (0.5 * height * frequencies[i])),
			   height*frequencies[i], colors[i], letters[i], faceMode, letterStyle);
	
    }
//     else {
//       cout << "write frequency too small: " << frequencies[i] << endl;
//     }
    currentHeight += height * frequencies[i];
  }
  os << "] # end of children for writeVrmlBoxStack" << endl;
  os << "} # end of writeVrmlBoxStack" << endl;

}

void
Vrml2MatrixWriter::writeBoxStackFlat(ostream& os,
				     const Vector3D& pos,
				     double height, 
				     double errorHeight,
				     const Vec<double>& frequencies,
				     const Vec<Vector3D>& colors,
				     const string& letters,
				     int faceMode,
				     int letterStyle) const
{
  PRECOND(colors.size()==frequencies.size(), exception);
  // cout << "start writeVrmlBoxStack(2) " << height << " "
  // << frequencies << endl;
  // if (wwwImageBase.size() == 0) { // no texture mapping
//   if (letterMode == LETTER_MODE_NONE) {
//     writeVrmlBoxStack(os, pos, height, frequencies, colors);
//     return;
//   }
  double currentHeight = pos.y();
  // deterime ordering: 
  if (height <= 0.0) {
    cout << "height smaller zero: " << height << endl;
    return;
  }
  os << "# start of writeVrmlBoxStack" << endl;
  os << "Group {" << endl;
  os << "children [" << endl;
  Vec<unsigned int> order = documentedOrder(frequencies);
  Vector3D errorColor(0.7, 0.7, 0.7);
  // reverse(order.begin(), order.end()); smallest first
  for (unsigned int ii = 0; ii < colors.size(); ++ii) {
    unsigned int i = order[ii];
    ERROR_IF(frequencies[i] > 1.0, "Internal error in line 262!",
	     exception);
    if (frequencies[i] > frequencyLimit) {
	writeVrmlBoxLetterFlat(os, 
			       Vector3D(pos.x(), currentHeight - (0.5 * height * frequencies[i]), pos.z()),
			       height*frequencies[i], colors[i], letters[i], faceMode, letterStyle);
	
    }
//     else {
//       cout << "write frequency too small: " << frequencies[i] << endl;
//     }
    currentHeight -= height * frequencies[i];
  }
  if (errorMode) {
    if (errorHeight < 0.1) {
      errorHeight = 0.1;
    }
    double topWidth = 0.1; // height of top "hat plate"
    writeBox(os, 
	     Vector3D(pos.x(), currentHeight - (0.5 * errorHeight), pos.z()),
	     Vector3D(0.5, errorHeight, 0.5), errorColor, false);
    writeBox(os, 
	     Vector3D(pos.x(), currentHeight - errorHeight - (0.5*topWidth), pos.z()),
	     Vector3D(1.0, topWidth, 1.0), errorColor, false);
    //     writeVrmlBoxLetterFlat(os, 
    // 			   Vector3D(pos.x(), currentHeight - (0.5 * errorHeight), pos.z()),
    // 			   errorHeight, errorColor, ' ', faceMode, letterStyle);
  }
  os << "] # end of children for writeVrmlBoxStack" << endl;
  os << "} # end of writeVrmlBoxStack" << endl;

}

void
Vrml2MatrixWriter::writeBoxStackFlat2(ostream& os,
				      const Vector3D& pos,
				      double height, 
				      double errorHeight,
				      const Vec<double>& frequencies,
				      const Vec<Vector3D>& colors,
				      const string& letters,
				      int faceMode,
				      int letterStyle) const
{
  PRECOND(colors.size()==frequencies.size(), exception);
  // cout << "start writeVrmlBoxStack(2) " << height << " "
  // << frequencies << endl;
  // if (wwwImageBase.size() == 0) { // no texture mapping
//   if (letterMode == LETTER_MODE_NONE) {
//     writeVrmlBoxStack(os, pos, height, frequencies, colors);
//     return;
//   }
  double currentHeight = pos.x();
  // deterime ordering: 
  if (height <= 0.0) {
    cout << "height smaller zero: " << height << endl;
    return;
  }
  os << "# start of writeVrmlBoxStack" << endl;
  os << "Group {" << endl;
  os << "children [" << endl;
  Vec<unsigned int> order = documentedOrder(frequencies);
  Vector3D errorColor(0.7, 0.7, 0.7);
  // reverse(order.begin(), order.end()); smallest first
  for (unsigned int ii = 0; ii < colors.size(); ++ii) {
    unsigned int i = order[ii];
    ERROR_IF(frequencies[i] > 1.0, "Internal error in line 262!",
	     exception);
    if (frequencies[i] > frequencyLimit) {
	writeVrmlBoxLetterFlat2(os, Vector3D(currentHeight - (0.5 * height * frequencies[i]), pos.y(), pos.z()),
				height*frequencies[i], colors[i], letters[i], faceMode, letterStyle);
	
    }
//     else {
//       cout << "write frequency too small: " << frequencies[i] << endl;
//     }
    currentHeight -= height * frequencies[i];
  }
  if (errorMode) {
    if (errorHeight < 0.1) {
      errorHeight = 0.1;
    }
//     writeVrmlBoxLetterFlat(os, 
// 			   Vector3D(currentHeight - (0.5 * errorHeight), pos.y(), pos.z()),
// 			   errorHeight, errorColor, ' ', faceMode, letterStyle);
    double topWidth = 0.1; // height of top "hat plate"
    writeBox(os, 
	     Vector3D(currentHeight - (0.5 * errorHeight), pos.y(), pos.z()),
	     Vector3D(errorHeight, 0.5, 0.5), errorColor, ' ');
    writeBox(os, 
	     Vector3D(currentHeight - errorHeight - (0.5*topWidth), pos.y(), pos.z()),
	     Vector3D(topWidth, 1.0, 1.0), errorColor, false);
  }
  os << "] # end of children for writeVrmlBoxStack" << endl;
  os << "} # end of writeVrmlBoxStack" << endl;

}

/*
void
Vrml2MatrixWriter::writeBoxStackBoth(ostream& os,
				     const Vector3D& pos,
				     double height, 
				     const Vec<double>& frequencies,
				     const Vec<Vector3D>& colors,
				     const string& lettersFront,
				     const string& lettersLeft,
				     int letterStyle) const
{
  PRECOND(colors.size()==frequencies.size(), exception);
  ASSERT(false
//   if (wwwImageBase.size() == 0) { // no texture mapping
//     writeBoxStack(os, pos, height, frequencies, colors);
//     return;
//   }
  // string comment = "start writeBoxStackBoth with letters " + dtos(height);
  // writeComment(os, comment);
  double currentHeight = 0.0;
  // deterime ordering: 
  if (height <= 0.0) {
    cout << "height smaller zero: " << height << endl;
    return;
  }
  Vec<unsigned int> order = documentedOrder(frequencies);
  // reverse(order.begin(), order.end()); smallest first
  for (unsigned int ii = 0; ii < colors.size(); ++ii) {
    unsigned int i = order[ii];
    ERROR_IF(frequencies[i] > 1.0, "Internal error in line 262!",
	     exception);
    if (frequencies[i] > frequencyLimit) {
      if (ii == (colors.size()-1)) {
// 	if (wwwImageBase.size() > 0) {
// 	  writeBoxLetterBoth2(os, Vector3D(pos.x(), pos.y(), 
// 					  currentHeight + (0.5 * height * frequencies[i])),
// 			     Vector3D(1.0, 1.0, height*frequencies[i]), colors[i], lettersFront[i], lettersLeft[i]);
// 	}
// 	else {
	writeBoxLetterBoth(os, Vector3D(pos.x(), pos.y(), 
					  currentHeight + (0.5 * height * frequencies[i])),
			     height*frequencies[i], colors[i], lettersFront[i], lettersLeft[i], letterStyle);
	// }
      }
      else {
// 	if (wwwImageBase.size() > 0) {
// 	  // currently no "open" version, still implement!
// 	  writeBoxLetterBoth2(os, Vector3D(pos.x(), pos.y(), 
// 					  currentHeight + (0.5 * height * frequencies[i])),
// 			     Vector3D(1.0, 1.0, height*frequencies[i]), colors[i], lettersFront[i], lettersLeft[i]);
// 	}
// 	else {
	// potentially writeOpenBoxLetterBox
	writeBoxLetterBoth(os, Vector3D(pos.x(), pos.y(), 
					    currentHeight + (0.5 * height * frequencies[i])),
			   height*frequencies[i], colors[i], lettersFront[i], lettersLeft[i], letterStyle);
	// }
      }
    }
//     else {
//       cout << "write frequency too small: " << frequencies[i] << endl;
//     }
    currentHeight += height * frequencies[i];
  }
  // draw "stencil": needed for 3D printing!
  if (groundPlateMode) {
    cout << "Writing stencil: " << pos << " " << gridWidth << " " << stencilColor << endl;
    writeBoxLetterBoth(os, Vector3D(pos.x(), pos.y(), -0.5*gridWidth), gridWidth, stencilColor, ' ', ' ', letterStyle);
  }
}
*/

void
Vrml2MatrixWriter::writeVrmlReadingFrame(ostream& os,
					 const Alignment& ali,
					 int pos) const
{
  string slice = ali.getSliceString(pos);
  Vec<double> frequencies = getFrequencies( slice, alphabet);
  Vec<Vector3D> colors(3);
  colors[0] = Vector3D(1, 0, 0);
  colors[1] = Vector3D(0, 1, 0);
  colors[2] = Vector3D(0, 0, 1);
  // double inf = scorer.singleEntropy6(slice);
  // cout << "Writing logo position " << pos << " " << colors.size() << endl;
  int rf = pos % 3;
  double height = (1 + rf);
  writeVrmlBox(os, Vector3D(pos, 
			    -5.0 - (0.5*height) - 0.5, 
			    0.0), Vector3D(1.0, height, 1.0),
	       colors[rf]);
  writeVrmlBox(os, Vector3D(-5 - (0.5*height) - 0.5, pos, 0.0), 
	       Vector3D(height, 1, 1.0),
	       colors[rf]);
}

void
Vrml2MatrixWriter::writeVrmlReadingFrame(ostream& os, const Alignment& ali) const
{
#ifdef VRML2MATRIXWRITER_VERBOSE
  cout << "starting writeVrmReadingFrame!" << endl;
#endif
  os << "# start of writeVrmlReadingFram" << endl;
  os << "Transform{   # Group for ReadingFrame" << endl;
  os << "children [" << endl;
  for (int i = 0; i < static_cast<int>(ali.getLength()); ++i) {
    writeVrmlReadingFrame(os, ali, i);
  }
  os << "]  # children of top transform for ReadingFrame" << endl;
  os << "} # end of Group for ReadingFrame" << endl; // end of group
  os << "# end of writeVrmlReadingFrame" << endl;
}

void
Vrml2MatrixWriter::writeVrmlBox(ostream& os,
				const Vector3D& pos,
				double height, 
				const Vector3D& col ) const
{
  if (height <= 0.0) {
    return;
  }
  // os << "," << endl
  os << "\tTransform {" << endl
     << "\t\ttranslation " << pos.x() << " " << pos.y() << " " << pos.z()
     << endl
     << "\t\tchildren Shape {" << endl
     << "\t\t\tappearance Appearance {" << endl
     << "\t\t\t\tmaterial Material { diffuseColor " << col.x() << " "
     << col.y() << " " << col.z() << " }" << endl
     << "}" << endl
     << "\t\t\tgeometry Box {" << endl
     << "\t\t\tsize " << "1.0 1.0 " << height << endl
     << "\t\t\t}" << endl
     << "\t\t}" << endl
     << "\t}" << endl;
}

void
Vrml2MatrixWriter::writeVrmlBox(ostream& os,
				const Vector3D& pos,
				const Vector3D& size, 
				const Vector3D& col ) const
{
  // os << "," << endl
  
  os << "\tTransform {" << endl
     << "\t\ttranslation " << pos.x() << " " << pos.y() << " " << pos.z()
     << endl
     << "\t\tchildren Shape {" << endl
     << "\t\t\tappearance Appearance {" << endl
     << "\t\t\t\tmaterial Material { diffuseColor " << col.x() << " "
     << col.y() << " " << col.z() << " }" << endl
     << "}" << endl
     << "\t\t\tgeometry Box {" << endl
     << "\t\t\tsize " << size.x() << " " << size.y() << " " << size.z() << endl
     << "\t\t\t}" << endl
     << "\t\t}" << endl
     << "\t}" << endl;
}

void
Vrml2MatrixWriter::writeVrmlBox(ostream& os,
				const Vector3D& pos,
				const Vector3D& size, 
				const Vector3D& col,
				const string& imageUrl) const
{
  // os << "," << endl
  
  os << "\tTransform {" << endl
     << "\t\ttranslation " << pos.x() << " " << pos.y() << " " << pos.z()
     << endl
     << "\t\tchildren Shape {" << endl
     << "\t\t\tappearance Appearance {" << endl
     << "\t\t\t\tmaterial Material { diffuseColor " << col.x() << " "
     << col.y() << " " << col.z() << " }" << endl;
  os << "  texture ImageTexture { url [ \"" << imageUrl << "\" ] " << endl
     << "   repeatS FALSE" << endl
     << "   repeatT FALSE" << endl
     << "  }" << endl     
     << "}" << endl;
  os << "\t\t\tgeometry Box {" << endl
     << "\t\t\tsize " << size.x() << " " << size.y() << " " << size.z() << endl
     << "\t\t\t}" << endl
     << "\t\t}" << endl
     << "\t}" << endl;
}

void
Vrml2MatrixWriter::writeVrmlBox(ostream& os,
				const Vector3D& pos,
				const Vector3D& size,
				const Vector3D& rot,
				double angle,
				const Vector3D& col ) const
{
  if (size <= 0.0) {
    return;
  }
  // os << "," << endl
  os << "\tTransform {" << endl
     << "\t\ttranslation " << pos.x() << " " << pos.y() << " " << pos.z()
     << endl
     << "\t\trotation " << rot.x() << " " << rot.y() << " " << rot.z()
     << " " << angle << endl
     << "\t\tchildren Shape {" << endl
     << "\t\t\tappearance Appearance {" << endl
     << "\t\t\t\tmaterial Material { diffuseColor " << col.x() << " "
     << col.y() << " " << col.z() << " }" << endl
     << "}" << endl
     << "\t\t\tgeometry Box {" << endl
     << "\t\t\tsize " << size.x() << " " << size.y() << " " << size.z() << endl
     << "\t\t\t}" << endl
     << "\t\t}" << endl
     << "\t}" << endl;
}

void
Vrml2MatrixWriter::writeVrmlMatching(ostream& os,
				     const Alignment& ali,
				     int pos) const
{
  string slice = ali.getSliceString(pos);
  ERROR_IF(slice.size() == 0, "Internal error in line 681!", exception);
  unsigned int gapCount = 0;
  for (unsigned int i = 0; i < slice.size(); ++i) {
    if (slice[i] == Alignment::GAP_CHAR) {
      ++gapCount;
    }
  }
  double frac = gapCount / static_cast<double>(slice.size());
  const Vector3D brown = Vector3D(102.0/256.0, 51.0/256.0, 0.0);
  double height = frac * stretchZ;
  if (height >= gapLimit) {
    writeVrmlBox(os, Vector3D(pos, ali.getLength() + 5.0, 0.5* height), 
		 height, brown);
    os << "," << endl;
    writeVrmlBox(os, Vector3D(ali.getLength() + 5.0, pos, 0.5* height), 
		 height, brown);
  }
}

void
Vrml2MatrixWriter::writeVrmlMatching(ostream& os,
				     const Alignment& ali) const
{
  double szd = ali.getLength();
  const Vector3D black = Vector3D(0.0, 0.0, 0.0);
  double bw = 0.25;
  os << "# start of writeVrmlMatching" << endl;
  os << "Group {" << endl;
  os << "children [" << endl;
  writeVrmlBox(os, 
	       Vector3D(szd + 5, 0.5 * szd, stretchZ + 0.5),
	       Vector3D(bw, szd, bw),
	       black);
  os << "," << endl;
  writeVrmlBox(os, 
	       Vector3D(szd + 5, 0.5 * szd, - 0.5),
	       Vector3D(bw, szd, bw),
	       black);
  os << "," << endl;
  writeVrmlBox(os, 
	       Vector3D(0.5 * szd, szd + 5, stretchZ + 0.5),
	       Vector3D(szd, bw, bw),
	       black);
  os << "," << endl;
  writeVrmlBox(os, 
	       Vector3D(0.5 * szd, szd + 5 , - 0.5),
	       Vector3D(szd, bw, bw),
	       black);

//   writeVrmlBox(os, 
// 	       Vector3D(szd + 1, 0.5 * szd, stretchZ),
// 	       Vector3D(1.0, szd, 1.0),
// 	       brown);
	       
  for (int i = 0; i < static_cast<int>(ali.getLength()); ++i) {
    writeVrmlMatching(os, ali, i);
    if ((i+1) < static_cast<int>(ali.getLength())) {
      os << "," << endl;
    }
  }
  os << "]  # end of children of writeVrmlMatching" << endl;
  os << "} # end of writeVrmlMatching" << endl;
}

/**
   initialize vrml world
   backround code take from: 
   http://www.dform.com/inquiry/tutorials/3dsmax/background/bgascii.html
*/
void
Vrml2MatrixWriter::initVrml(ostream& os, unsigned int len) const
{
  unsigned int len2 = len/2;
  primitiveCounter = 0;
  double groundThickness = 0.5;
  double gw = 1.0; // 0.5;
  os << "#VRML V2.0 utf8" << endl
    // << "DEF sky Background {" << endl
     << "Background {" << endl
     << "  skyColor [1.0 1.0 1.0 ]" << endl
    // << "  skyAngle [ ]" << endl
     << "  groundColor [1.0 1.0 1.0 ]" << endl
//      << "  skyColor [0.8627 0.8784 0.3961, 0.902 0.8706 0, ]" << endl
//      << "  skyAngle [1.641, ]" << endl
//      << "  groundColor [0.8627 0.8784 0.3961, 0.902 0.8706 0, ]" << endl
    // << "  groundAngle [ ]" << endl
     << "}" << endl;  
  
 os << "Viewpoint {" << endl
    << "position " << len2 << " " << -20 << " " << 2*len 
    << " #the camera positioned to X Y Z" << endl
     << "orientation 1 0 0 0.35	#default orientation" << endl
     << "description \"front\"	#the name of the view" << endl
     << "}" << endl;
 os << "Viewpoint {" << endl
    << "position " << 0 << " " << -20 << " " << 0.5*len 
    << " #the camera positioned to X Y Z" << endl
     << "orientation 1 0 -0.6 1.1	#default orientation" << endl
     << "description \"origin\"	#the name of the view" << endl
     << "}" << endl;
  os << "Group {" << endl
     << "\tchildren [" << endl;
    /*
     << "\tShape {" << endl
     << "\tappearance Appearance {" << endl
     << "\t\tmaterial Material { }" << endl
     << "\t}" << endl;
  os << "\tgeometry Box {" << endl
     << "\t\tsize 1 1 1" << endl
     << "\t}" << endl
     << "\t}," << endl;
    */
  ++primitiveCounter;
  
  // write ground plate:
  if (groundPlateMode) {
   os << "," << endl
      << "\tTransform {" << endl
      << "\t\ttranslation " << ((len/2)-(0.5*gw)) << " " << ((len/2)-(0.5*gw)) << " " << (-gw-(0.5*groundThickness)) << endl
      << "\t\tchildren Shape {" << endl
      << "\t\t\tappearance Appearance {" << endl
      << "\t\t\t\tmaterial Material { diffuseColor " << groundPlateColor.x() << " "
      << groundPlateColor.y() << " " << groundPlateColor.z() << " }" << endl
      << "}" << endl
      << "\t\t\tgeometry Box {" << endl
      << "\t\t\tsize " << len << " " << len << " " << groundThickness << endl
      << "\t\t\t}" << endl
      << "\t\t}" << endl
      << "\t}," << endl;
  }
  //  os << "\t}" << endl
  //   << "\t}";
  double arrowLen = 15.0;
  double arrowHalf = arrowLen * 0.5;
  double arrowSep = 10.0;
  if (logoMode == AlignColorConstants::LOGO_FLAT) {
    arrowSep = 2.5;
  }
  double step = 10.0;
  double step2 = step;
  double step2len = 9.0;
  double borderLen = 10;
  double stepSum = 0.0;
  // double length = len;
  // unsigned int counter = 0;

  Vector3D gridColBorder(0.1, 0.1, 0.1);
  Vector3D gridColHigh(1, 1, 0);
  Vector3D gridColNormal(1.0, 1.0, 1.0);
  Vector3D gridCol(1.0, 1.0, 1.0);
  // Vector3D gridCol2(102.0/256, 1.0, 1.0);
  Vector3D tickColBorder(0.1, 0.1, 0.1);
  Vector3D tickColNormal(0.5, 0.5, 0.5);
  Vector3D tickColHigh(1, 1, 0);
  Vector3D zeroCol(0.9, 0.9, 0.9);
  Vector3D tickCol = tickColNormal;
  // write arrow with cone here:


//   writeVrmlBox(os,
// 	       Vector3D(stepSum, 0.0, 0.0),
// 	       Vector3D(0.25, 0.25, 15.0) , 
// 	       zeroCol );

  int numLegend = 10;
  Vec<double> legendFrequencies(numLegend, 1.0/static_cast<double>(numLegend)); // all 16 base pairs
  Vec<string> legendNames(numLegend);
  legendNames[0] = "AU";
  legendNames[1] = "CG";
  legendNames[2] = "GU";
  legendNames[3] = "AA";
  legendNames[4] = "CC";
  legendNames[5] = "GG";
  legendNames[6] = "UU";
  legendNames[7] = "AC";
  legendNames[8] = "AG";
  legendNames[9] = "CU";
  Vec<Vector3D> legendColors = getPairColorsMode(legendNames, 0.0);

  switch (legendMode) {
  case 0:
    break;
  case 1:
    os << "# writing legend" << endl;
    writeVrmlBoxStack(os,
		      Vector3D(0, 0, 0.0),
 		    40.0, legendFrequencies, legendColors);
    os << "," << endl;
    break;
  }

  switch (arrowMode) {
  case 0:
    break;
  case 1:
    writeVrmlBox(os,
		 Vector3D(arrowHalf, -arrowSep, 0.0),
		 Vector3D(arrowLen, 0.25, 0.25) , 
		 zeroCol );
    os << "," << endl;
    writeVrmlCone(os,
		  Vector3D(arrowLen, -arrowSep, 0.0),
		  Vector3D(0, 0, 1.0),
		  -90.0 * DEG2RAD,
		  zeroCol );
    os << "," << endl;
    writeVrmlBox(os,
		 Vector3D(-arrowSep, arrowHalf, 0.0),
		 Vector3D(0.25, arrowLen, 0.25) , 
		 zeroCol );
    os << "," << endl;
    writeVrmlCone(os,
		  Vector3D(-arrowSep, arrowLen, 0.0),
		  Vector3D(0, 0, 1.0),
		  0.0,
		  zeroCol );
    os << "," << endl;
    break;
  default:
    ERROR("Unknown arrow mode!", exception);
  }

  // diagonal stripe:
//    writeVrmlBox(os,
//  	       Vector3D(0.5*len, 0.5 * len, 0.5*gw),
//  	       Vector3D(sqrt(2.0) * len, gw, 1.0*gw) , 
//  	       Vector3D(0.0, 0.0, 1.0),
//  	       45.0 * DEG2RAD,
//  	       zeroCol );
//    os << "," << endl;

/*
  while (stepSum < len) {
    if (( static_cast<int>(stepSum+0.1) % 100) != 0) {
      tickCol = tickColNormal;
      gridCol = gridColNormal;
    }
    else {
      tickCol = tickColHigh;
      gridCol = gridColHigh;
    }

    writeVrmlBox(os,
		 Vector3D(stepSum, 0.5 * length, 0.5*gw),
		 Vector3D(1.0, length, gw) , 
		 gridCol);
    os << "," << endl;
    writeVrmlBox(os,
 		 Vector3D(0.5 * length , stepSum, 0.5*gw),
 		 Vector3D(length, 1.0, 1.0*gw), 
		 gridCol);
    os << "," << endl;
    stepSum += step;
  }
  */


  // x-axis:
  // was : 0.0 instead of -gw
  writeBox(os, Vector3D((0.5*len)-(0.5*gw), -gw, -0.5*gw),
	   Vector3D(len, gw, gw), gridColBorder, true);
  os << "," << endl;
  // y-axis:
  // was : 0.0 instead of -gw
  writeBox(os, Vector3D(-gw, (0.5*len)-(0.5*gw), -0.5*gw),
	   Vector3D(gw, len, gw), gridColBorder, true);
  os << "," << endl;
  // diagonal stripe:
//   writeBox(os,
// 	   Vector3D(0.5*len, 0.5 * len, -0.5*gw),
// 	   Vector3D(sqrt(2.0) * len, gw, 1.0*gw) , 
// 	   Vector3D(0.0, 0.0, 1.0),
// 	   45.0 * DEG2RAD,
// 	   zeroCol );
  // loop over x:
  stepSum = - step - 1;
  while (stepSum < len) {
    double stepSum2 = -step2 - 1;
    double lenPiece;
    // diagonal stripe:
    if (((stepSum+1) >= -0.01) && ((stepSum + 0.5*step) <= len)) {
      writeVrmlBox(os,
	       Vector3D(stepSum + (0.5*step), stepSum + (0.5*step), -0.5*gw),
	       Vector3D((sqrt(2.0) * step) - (3*gw), gw, gw) , 
	       Vector3D(0.0, 0.0, 1.0), // rotation around z-axis
	       45.0 * DEG2RAD,
	       zeroCol );
      os << "," << endl;
    }
    // loopy of y:
    while (stepSum2 < len) {
      int iy = static_cast<int>((stepSum2+step2)+0.1);
      if (iy  == 0) {
	stepSum2 += step2;
	continue;
	tickCol = tickColBorder;
	gridCol = gridColBorder;
	lenPiece = borderLen;
      }
      else if ((iy  % 100) != 0) {
	tickCol = tickColNormal;
	gridCol = gridColNormal;
	lenPiece = step2len;
      }
      else {
	tickCol = tickColHigh;
	gridCol = gridColHigh;
	lenPiece = step2len;
      }
      if (((stepSum + 0.5*step) <= len) && (stepSum2 + step2 <= len)
	  && ((stepSum + 0.5*step) >= 0)) {
	// write cube parallel to x - axis
	writeBox(os,
		 Vector3D(stepSum + 0.5*step, stepSum2 + step2, -0.5*gw - 0.01),
		 Vector3D(lenPiece, 1.0, gw), gridCol);
	os << "," << endl;
      }
      int ix = static_cast<int>((stepSum+step)+0.1);
      if (ix == 0) {
	stepSum2 += step2;
	continue;
	tickCol = tickColBorder;
	gridCol = gridColBorder;
	lenPiece = borderLen;
      }
      else if ((ix  % 100) != 0) {
	tickCol = tickColNormal;
	gridCol = gridColNormal;
	lenPiece = step2len;
      }
      else {
	tickCol = tickColHigh;
	gridCol = gridColHigh;
	lenPiece = step2len;
      }
      // write cube parallel to y - axis
      if (((stepSum + step) <= len) && (stepSum2 + 0.5 * step2 <= len)
	  && ((stepSum2 + 0.5*step2) >= 0)) {
	writeBox(os,
		 Vector3D(stepSum + step, stepSum2 + 0.5 * step2, -0.5*gw - 0.01),
		 Vector3D(1.0, lenPiece, gw), gridCol);
	os << "," << endl;
      }
      stepSum2 += step2;
    }
    stepSum += step;
  }



}

void
Vrml2MatrixWriter::finalize(ostream& os) const
{
  os << endl
     << "\t]" << endl
     << "}" << endl;
}

/** returns all possible 16 pairs */
Vec<string>
Vrml2MatrixWriter::getPairs() const
{
  unsigned int nn = alphabet.size();
  Vec<string> pairs;
  string s = "xx";
  for (unsigned int i = 0; i < nn; ++i) {
    for (unsigned int j = 0; j < nn; ++j) {
      s[0] = alphabet[i];
      s[1] = alphabet[j];
      pairs.push_back(s);
    }
  }
  return pairs;
}

Vec<double>
Vrml2MatrixWriter::getPairFrequencies(const string& s1, const string& s2, const Vec<string>& pairs) const
{
  Vec<double> result(pairs.size());
  for (unsigned int i = 0; i < pairs.size(); ++i) {
    result[i] = getFrequency(s1, s2, pairs[i]);
  }
  return result;
}

/** calls getPairColorsX according to colorMode */
Vec<Vector3D>
Vrml2MatrixWriter::getPairColorsMode(const Vec<string>& pairs, double mutInf) const
{
  Vec<Vector3D> colors;
  switch (colorMode) {
  case 1: colors = getPairColors(pairs);
    break;
  case 2: colors = getPairColors2(pairs);
    break;
  case 3: colors = getPairColors3(pairs);
    break;
  case 4: colors = getPairColors4(pairs, mutInf); // rainbow encoding: all pairs have same color
    break;
  case 5: colors = getPairColors5(pairs); // another attempt to hand-crafted colors
    break;
  case 6: colors = getPairColorsProtein1(pairs); // another attempt to hand-crafted colors
    break;
  default: ERROR("Undefined color mode!", exception);
  }
  return colors;
}

void
Vrml2MatrixWriter::writeVrml(ostream& os,
			     const Alignment& ali,
			     const Vec<Vec<double> >& matrix,
			     const Vec<Vec<double> >& errorMatrix,
			     const Vec<double>& infVec,
			     const Vec<double>& infErrorVec,
			     const Vec<Stem>& stems) const
{
  // cout << "Starting writeVrml(2) from Vrml2MatrixWriter!" << endl;
  initVrml(os, ali.getLength());
  switch (logoMode) {
  case AlignColorConstants::LOGO_NONE:
    break; // do nothing
  case AlignColorConstants::LOGO_DEFAULT:
    writeLogos(os, ali, infVec, infErrorVec);
    break;
  case AlignColorConstants::LOGO_FLAT:
    writeLogosFlat(os, ali, infVec, infErrorVec);
    break;
  default:
    ERROR("Unknown logo mode!", exception);
  }
  if (gapMode != AlignColorConstants::LOGO_NONE) {
    writeVrmlMatching(os, ali);
  }
  if (readingFrameMode && (logoMode != AlignColorConstants::LOGO_FLAT)) {
    writeVrmlReadingFrame(os, ali);
  }
  unsigned int counter = 0;
  // double refHeight = 1.0;
  // double stackStart = 5.0;

  // cout << "two scorer version!" << endl;
  os << "# start of writing matrix data" << endl;
  for (unsigned int i = 0; i < ali.getLength(); ++i) {
    string s1 = ali.getSliceString(i);
    for (unsigned int j = i + 1; j < ali.getLength(); ++j) {
      string s2 = ali.getSliceString(j);
      double mutInf = matrix[i][j]; // scorer.compensationScore(s1, s2, ali.getWeights());
      double mutInfError = errorMatrix[i][j]; // scorer.compensationScore(s1, s2, ali.getWeights());
      // cout << "Vrml2MatrixWriter:: Writing position " << i + 1<< " " << j + 1 << endl;
      if (writePosStack(os, ali, stems, i, j, mutInf, mutInfError,
			letterMode2)) {
	++counter;
      }
      mutInf = matrix[j][i]; // scorer.compensationScore(s1, s2, ali.getWeights());
      if (writePosStack(os, ali, stems, j, i, mutInf, mutInfError, 
			letterMode2)) {
	++counter;
      }
    }
    //     if (counter > 2000) {
    //       break;
      //     }
  }

  Vector3D refColor(0.0, 0.0, 0.0);
  /*
  if (drawMode & RENDER_MATRIX) {
    for (unsigned int i = 0; i < stems.size(); ++i) {
      for (int k = 0; k < stems[i].getLength(); ++k) {
   	start = stems[i].getStart()+k;
   	stop = stems[i].getStop()-k;
   	// write in empty half
   	writeVrmlBox(os, Vector3D(static_cast<double>(stop),
   				  static_cast<double>(start),
				  0.5 * stretchZ),
		     stretchZ,
		     refColor);
	// 	if (! (drawMode & RENDER_ENTROPY) ) {
  // 	  // write in half that is normaly reserved for mututal information
  // 	  writeVrmlBox(os, Vector3D(static_cast<double>(start),
  // 	    static_cast<double>(stop), 0.5 * stretchZ), stretchZ,  refColor);
  // 	}
  //       }
  //     }
  //   }
  */

  // draw reference structure on top:
//   writeComment(os, "Writing reference structure!");
//   for (int j = 0; j < static_cast<int>(stems.size()); ++j) {
//     for (int k = 0; k < (stems[j]).getLength(); ++k) {
//       int posStart = stems[j].getStart()+k;
//       int posStop = stems[j].getStop()-k;
//       double matrixHeight = matrix[posStart][posStop] + refHeight;
//       writeVrmlBox(os, Vector3D(static_cast<double>(posStart), static_cast<double>(posStop), matrixHeight),
// 		   refHeight, refColor);
//       matrixHeight = matrix[posStop][posStart] + refHeight;
//       writeVrmlBox(os, Vector3D(static_cast<double>(posStop), static_cast<double>(posStart), matrixHeight),
// 		   refHeight, refColor);
//       ++counter;	
//     }
//   }
//   writeComment(os, "End of writing reference structure!");  
  os << "# end of writing matrix data" << endl;
  finalize(os);
}


/* MODIFIERS */

/* copy method */
void 
Vrml2MatrixWriter::copy(const Vrml2MatrixWriter& other)
{
  AbstractMatrixWriter::copy(other);
  letters3D = other.letters3D;
  readingFrameMode = other.readingFrameMode;
  // colorMode = other.colorMode;
  letterMode = other.letterMode;
  // logoMode = other.logoMode;
  dxLetterFront = other.dxLetterFront; // = 0.1;
  dyLetterFront = other.dyLetterFront; // = -0.6;
  dxLetterLeft = other.dxLetterLeft; // = -0.2;
  dyLetterLeft = other.dyLetterLeft; //  = 0.3;
  dzLetter = other.dzLetter;
  // frequencyLimit = other.frequencyLimit;
  gapLimit = other.gapLimit;
  // infLimit = other.infLimit;
  letterHeight = other.letterHeight;
  letterLineThickness = other.letterLineThickness;
  stretchZ = other.stretchZ;
  // alphabet = other.alphabet;
  sequenceType = other.sequenceType;
  // wwwImageBase = other.wwwImageBase;
  // resources = other.resources;
}
