#ifndef _RNA_PROBABILITY_TOOLS_
#define _RNA_PROBABILITY_TOOLS_

#include <Vec.h>
#include <AbstractSecondaryStructure.h>
#include <debug.h>

using namespace std;

class RnaProbabilityTools {

 public:
  typedef AbstractSecondaryStructure::index_type index_type;
  typedef RankedSolution3<RnaTertiaryStructure> ranked_structure;
  typedef Vec<Vec<double> > matrix_type;

 public:

  /** For a container of AbstractRnaSeconaryStructure, compute the Boltman-weighted probabilities of 
   * individual base pairs. FIXIT: untested so far. */
  template<typename T>
  static
  matrix_type computeProbabilityMatrix(T first, T last, double kt) {
    ASSERT(first != last);
    // compute partition function:
    double partSum = 0;
    double eMin = first->getEnergy();
    // find mimum energy
    for (T it = first; it != last; it++) {
      if (it->getEnergy() < eMin) {
	eMin = it->getEnergy();
      }
    }
    index_type len = first->getLength();
    Vec<set<int> > basePairSets(len); // for residue n, store all partners ever found
    for (T it = first; it != last; it++) {
      partSum += exp(-(it->getEnergy() -eMin) / kt); // subtracting eMin should help avoid large numbers
      for (index_type i = 0; i < len; ++i) {
	if (it->getBasePair(i) != AbstractSecondaryStructure::UNPAIRED_ID) {
	  basePairSets[i].insert(it->getBasePair());
	}
      }
    }
    Vec<Vec<double> > result(len, Vec<double>(len, 0.0));
    for (index_type i = 0; i < len; ++i) {
      // loop only over found base pairs:
      for (set<int>::const_iterator sit = basePairSets[i].begin(); sit != basePairSets[i].end(); sit++) {
	index_type j = *sit; // base pair partner
	double probSum = 0.0;
	for (T it = first; it != last; it++) {
	  double pMolecule = exp(-(it->getEnergy() - eMin)) / partSum;
	  ASSERT(pMolecule >= 0.0);
	  ASSERT(pMolecule <= 1.0);
	  if (it->isPaired(i, j)) {
	    probSum += pMolecule;
	  }
	}
	ASSERT(probSum >= 0.0);
	ASSERT(probSum <= 1.1); // allow for rounding errors
	if (probSum > 1.0) {
	  probSum = 1.0;
	}
	result[i][j] = probSum;
      }
    }
    return result;
  }

  /** Same as computeProbabilityMatrix, however assumes that RnaSecondaryStructure is stored in a datastructure with memeber "second"
   * like pair or RankedSolution3 */
  template<typename T>
  static
  matrix_type computeProbabilityMatrixSecond(T first, T last, double kt) {
    ASSERT(first != last);
    // compute partition function:
    double partSum = 0;
    double eMin = first->second.getEnergy();
    // find mimum energy
    for (T it = first; it != last; it++) {
      if (it->second.getEnergy() < eMin) {
	eMin = it->second.getEnergy();
      }
    }
    index_type len = first->second.getLength();
    Vec<set<int> > basePairSets(len); // for residue n, store all partners ever found
    for (T it = first; it != last; it++) {
      partSum += exp(-(it->second.getEnergy() - eMin) / kt); // subtracting eMin should help avoid large numbers
      for (index_type i = 0; i < len; ++i) {
	if (it->second.getBasePair(i) != AbstractSecondaryStructure::UNPAIRED_ID) {
	  basePairSets[i].insert(it->second.getBasePair(i));
	}
      }
    }
    Vec<Vec<double> > result(len, Vec<double>(len, 0.0));
    for (index_type i = 0; i < len; ++i) {
      // loop only over found base pairs:
      for (set<int>::const_iterator sit = basePairSets[i].begin(); sit != basePairSets[i].end(); sit++) {
	index_type j = *sit; // base pair partner
	double probSum = 0.0;
	for (T it = first; it != last; it++) {
	  double pMolecule = exp(-(it->second.getEnergy() - eMin)/kt) / partSum;
	  ASSERT(pMolecule >= 0.0);
	  ASSERT(pMolecule <= 1.0);
	  if (it->second.isPaired(i, j)) {
	    probSum += pMolecule;
	  }
	}
	ASSERT(probSum >= 0.0);
	ASSERT(probSum <= 1.1); // allow for rounding errors
	if (probSum > 1.0) {
	  probSum = 1.0;
	}
	result[i][j] = probSum;
      }
    }
    return result;
  }

};

#endif
