// --*- C++ -*------x---------------------------------------------------------
// $Id: ranksec.cc,v 1.6 2010/11/30 13:45:44 bindewae Exp $
//
// Program:         - 
//
// Author:          Eckart Bindewald
//
// Project name:    -
//
// Date:            $Date: 2010/11/30 13:45:44 $
//
// Description:     reranks read secondary structures
// 
// -----------------x-------------------x-------------------x-----------------

#include <iostream>
#include <fstream>
#include <string>
#include <Vec.h>
#include <debug.h>
#include <GetArg.h>
#include <FileName.h>
#include <Stem.h>
#include <stemhelp.h>
#include <vectornumerics.h>
#include <SequenceAlignment.h>
#include <SimpleSequenceAlignment.h>

void
helpOutput(ostream& os)
{
  os << "reranks secondary structures according to their compatibility with a probability matrix." << endl;
  os << "Usage: ranksec -i stemfile --prob-matrix matrixfile "
     << "--diff-weight weight" << endl
     << "--energy-max value" << endl
     << "--energy-weight value" << endl
     << "--equal" << endl
     << "--free energy" << endl
     << "--if id  : input format id (equivalent --stem-format)" << endl
     << "--kt value" << endl
     << "-n number : maximum number of output solutions" << endl
     << "-o filename  : output filename for solution matrix" << endl
     << "-p filename  : parameter filename (try ~/bin/ranksec.prm)" << endl
     << "-pk filename : file with pseudoknot structures in addition to structures defined by option -i" << endl
     << "--prob-matrix-format n]" << endl
     << "--pseudo pseudocount  : add pseudocount for computing Matthews correlation coefficient" << endl
     << "--ref-format formatid  : format of reference stems" << endl
     << "--ref-stems filename   : filename of reference stems" << endl
     << "--scores filename" << endl
     << "--score-weight w" << endl
     << "--sequence alignmentfile" << endl
     << "--sequencef-format id : 1: FASTA format, 2: Zuker format" << endl
     << "--stem-format formatid : format of stemfile" << endl
     << "--verbose n"
     << endl;
}

/** output of command line parameter with which the program was called. */
void
parameterOutput(ostream& os, int argc, char** argv)
{
  for (int i = 0; i < argc; i++)
    {
      os << argv[i] << " ";
    }
  os << endl;
}

Vec<double>
computeNewStemScores(const Vec<Vec<Stem> >& stemVec, 
		     const Vec<Vec<double> >& probMatrix,
		     const Vec<double>& initialScores, 
		     double initialScoreWeight, 
		     double kt,
		     double cutoff,
		     double penalty)
{
  // cout << "Starting computeNewStemScores!" << endl;
  // cout << stemVec << initialScores << probMatrix << endl;
  Vec<double> scores(stemVec.size(), 0.0);
  for (unsigned int k = 0; k < stemVec.size(); ++k) {
    scores[k] = initialScores[k];
    Vec<Vec<double> > stemMatrix = generateMatrixFromStems(stemVec[k], probMatrix.size());
    for (unsigned int i = 0; i < stemMatrix.size(); ++i) {
      for (unsigned int j = 0; j < stemMatrix[i].size(); ++j) {
	double term = 0.0;
	if (stemMatrix[i][j] > cutoff) { // contact in stemMatrix
	  ERROR_IF((i>= probMatrix.size()),
		   "Internal error in line 66!");
	  ERROR_IF((j>= probMatrix[i].size()),
		   "Internal error in line 68!");
	  if (probMatrix[i][j] < cutoff) { // no contact in probability matrix
	    // misprediction:
	    if (probMatrix[i][j] > 0) {
	      term = - kt * log(probMatrix[i][j]);
	    }
	    else {
	      term = kt * penalty;
	    }
	  }
	}
	else { // no contact in stemMatrix
	  if (probMatrix[i][j] > cutoff) { // contact in probability matrix
	    // misprediction:
	    if ((1.0 - probMatrix[i][j]) > 0.0) {
	      term = - kt * log(1.0 - probMatrix[i][j]);
	    }
	    else {
	      term = kt * penalty;
	    }
	  }
	}
	if (term > kt * penalty) {
	  term = kt * penalty;
	}
	scores[k] += term;
      }
    }
  }
  // cout << "Ending computeNewStemScores!" << endl;
  return scores;
}

Vec<double>
computeNewStemScores2(const Vec<Vec<Stem> >& stemVec, 
		     const Vec<Vec<double> >& probMatrix,
		     const Vec<double>& initialScores, 
		     double initialScoreWeight, 
		     double kt,
		     double cutoff,
		     double penalty)
{
  cout << "Starting computeNewStemScores2!" << endl;
  // cout << stemVec << initialScores << probMatrix << endl;
  Vec<double> scores(stemVec.size(), 0.0);
  for (unsigned int k = 0; k < stemVec.size(); ++k) {
    scores[k] = initialScores[k];
    Vec<Vec<double> > stemMatrix = generateMatrixFromStems(stemVec[k], 
							   probMatrix.size());
    for (unsigned int i = 0; i < stemMatrix.size(); ++i) {
      for (unsigned int j = 0; j < stemMatrix[i].size(); ++j) {
	double term = 0.0;
	// double a = 2 * (probMatrix[i][j] - 0.5);
	// double b = 1.0 - probMatrix[i][j];
	double p = probMatrix[i][j]; // probability of a contact according to prediction
	if (stemMatrix[i][j] > 0.5) { // contact in stemMatrix
	  ERROR_IF((i>= probMatrix.size()),
		   "Internal error in line 66!");
	  ERROR_IF((j>= probMatrix[i].size()),
		    "Internal error in line 68!");
	  // misprediction:
	  if (p > 0) {
	    term = - kt * log(p);
	  }
	  else {
	    term = kt * penalty;
	  }
	}
	else { // no contact in stemMatrix
	  // misprediction:
	  if ((1.0 - p) > 0.0) {
	    term = - kt * log(1.0 - p);
	  }
	  else {
	    term = kt * penalty;
	  }
	}
	if (term > kt * penalty) {
	  term = kt * penalty;
	}
	scores[k] += term;
      }
    }
  }
  cout << "Ending computeNewStemScores!" << endl;
  return scores;
}


Vec<double>
computeNewStemScores3(const Vec<Vec<Stem> >& stemVec, 
		     const Vec<Vec<double> >& probMatrix,
		     const Vec<double>& initialScores, 
		     double initialScoreWeight, 
		     double kt,
		     double cutoff,
		      double penalty,
		      const Vec<double>& probContacts,
		      const Vec<double>& probNoContacts)
{
  cout << "Starting computeNewStemScores2!" << endl;
  // cout << stemVec << initialScores << probMatrix << endl;
  double apriori = 0.7 / probMatrix.size();
  Vec<double> scores(stemVec.size(), 0.0);
  int bin;
  int numBins = probContacts.size();
  double term;
  for (unsigned int k = 0; k < stemVec.size(); ++k) {
    scores[k] = initialScores[k];
    Vec<Vec<double> > stemMatrix = generateMatrixFromStems(stemVec[k], 
							   probMatrix.size());
    for (unsigned int i = 0; i < stemMatrix.size(); ++i) {
      for (unsigned int j = 0; j < stemMatrix[i].size(); ++j) {
	bin = static_cast<int>(probMatrix[i][j] * numBins);
	if (probMatrix[i][j] == 1.0) {
	  --bin;
	}
	// double p = probMatrix[i][j]; // probability of a contact according to prediction
	if (bin >= static_cast<int>(probContacts.size())) {
	  bin = probContacts.size()-1;
	}
	else if (bin < 0) {
	  bin = 0;
	}
	double p = probContacts[bin] * apriori / 
	  (probContacts[bin]*apriori + probNoContacts[bin]*(1.0-apriori));
	if (stemMatrix[i][j] > 0.5) { // contact in stemMatrix
	  ERROR_IF((i>= probMatrix.size()),
		   "Internal error in line 66!");
	  ERROR_IF((j>= probMatrix[i].size()),
		    "Internal error in line 68!");
	  // misprediction:

	  if (p > 0) {
	    term = - kt * log(p);
	  }
	  else {
	    term = kt * penalty;
	  }
	}
	else { // no contact in stemMatrix
	  // misprediction:
	  if ((1.0 - p) > 0.0) {
	    term = - kt * log(1.0 - p);
	  }
	  else {
	    term = kt * penalty;
	  }
	}
	if (term > kt * penalty) {
	  term = kt * penalty;
	}
	scores[k] += term;
      }
    }
  }
  cout << "Ending computeNewStemScores!" << endl;
  return scores;
}

Vec<double>
computeNewStemScores4(const Vec<double>& initialScores)
{
  cout << "Starting computeNewStemScores4!" << endl;
  // cout << stemVec << initialScores << probMatrix << endl;
  Vec<double> scores(initialScores.size(), 0.0);
  for (unsigned int k = 0; k < scores.size(); ++k) {
    scores[k] = initialScores[k] - initialScores[0];
  }
  return scores;
}

unsigned int
outputScores(ostream& os,
	     Vec<double> scores,
	     Vec<Vec<Stem> >& stems,
	     Vec<Stem>& refStems,
	     unsigned int len,
	     unsigned int maxScores,
	     int verboseLevel,
	     double pseudoCount)
{
  Vec<unsigned int> ranks = generateStair(scores.size());
  parallelSort(scores, ranks);
  //   reverse(ranks.begin(), ranks.end());
  //   reverse(scores.begin(), scores.end());
  unsigned int tp, fp, tn, fn;
  cout << "Assuming length: " << len << endl;
  cout << "Reference stems: " << refStems << endl;
  double bestMath = 0.0;
  unsigned int bestId = 0;
  unsigned int avgCount = 0;
  double matthewsAvg = 0;
  double fpAvg = 0;
  double fnAvg = 0;
  for (unsigned int i = 0; i < scores.size(); ++i) {
    if ((maxScores == 0) || (i < maxScores)) {
      os << "Solution: " << i + 1 << "\tOriginal rank: " << ranks[i] + 1 << "\tScore: " << scores[i] << " ";
    }
    if ((refStems.size() > 0) && (len > 0)) {
      tp = 0; fp = 0; tn = 0; fn = 0;
      double mathews = compareStems(len, stems[ranks[i]], refStems, tp, fp, tn, fn); // , pseudoCount);
      double misFrac = static_cast<double>(fp + fn);
      misFrac /= static_cast<double>(len);

      // compute averages:
      matthewsAvg += mathews;
      fpAvg += fp;
      fnAvg += fn;
      ++avgCount;

      if ((maxScores == 0) || (i < maxScores)) {
	os << " Mathews: " << mathews << " fraction wrong bp: " << misFrac << " " 
	   << " true positive base pairs: " << tp << " true negatives: " << tn << " false positives: " << fp 
	   << " false negatives " << fn;
	os << " Stems: " 
	   << stems[ranks[i]];
      }
      if (verboseLevel > 1) {
	Vec<Vec<double> > stemMatrix(len,
				     Vec<double>(len, 0.0));
	addStemsToMatrix(stemMatrix, stems[ranks[i]], 1.0);
	cout << "Detailed comparison: " << endl;
	for (unsigned int j = 0; j < refStems.size(); ++j) {
	  int counter = 0;
	  os << j + 1 << " ) " << refStems[j] << " found bp: ";
	  for (int k = 0; k < refStems[j].getLength(); ++k) {
	    if (stemMatrix[refStems[j].getStart()+k][refStems[j].getStop()-k] > 0.5) {
	      ++counter;
	    }
	  }
	  os << counter << endl;
	}
      }
      if ((fp + fn) == 0) {
	if ((maxScores == 0) || (i < maxScores)) {
	  cout << "Perfect match!";
	}
      }
      if (mathews > bestMath) {
	bestMath = mathews;
	bestId = i;
      }
    }
    if ((maxScores == 0) || (i < maxScores)) {
      os << endl;
    }
  }
  // final output:
  if ((refStems.size() > 0) && (len > 0)) {
    os << "Best solution: " 
       << bestId + 1 << "\tOriginal Rank: " << ranks[bestId] + 1 << "\tScore: " << scores[bestId];
    double mathews = compareStems(len, stems[ranks[bestId]], 
				  refStems, tp, fp, tn, fn); // , pseudoCount);
    double misFrac = static_cast<double>(fp + fn);
    misFrac /= static_cast<double>(len);
    os << " Mathews: " << mathews << " fraction wrong base pairs: " << misFrac << " true positive base pairs: " 
       << tp << " true negatives: " << tn << " false positives: " << fp << " false negatives: " << fn;
    os << " Stems: " 
       << stems[ranks[bestId]];
    if ((fp + fn) == 0) {
      os << "Perfect match!";
    }
    os << endl;
  }
  matthewsAvg /= avgCount;
  double fAvg = (fpAvg + fnAvg) / avgCount;
  fpAvg /= avgCount;
  fnAvg /= avgCount;

  os << "Average properties (<d_bp>, <matthews>, <fp>, <fn> <d_bp>/len): " 
     << fAvg << " " << matthewsAvg << " " 
     << fpAvg << " " << fnAvg << " " << (fAvg/len) << endl;
  return bestId;
}

void
outputMultiCt(ostream& os,
	      Vec<double> scores,
	      Vec<Vec<Stem> >& stems,
	      const string& sequence) {
  // string::size_type len = sequence.size();
  for (unsigned int i = 0; i < scores.size(); ++i) {
    writeCtSingle(os, stems[i], sequence);
  }
}

Vec<Vec<double> >
readListMatrix(istream& is, double defaultValue, bool mirrorMode)
{
  unsigned int x, y, numEntries;
  is >> x >> y >> numEntries; // dimensions
  double val;
  Vec<Vec<double> > result(x, Vec<double>(y, defaultValue));
  for (unsigned int i = 0; i < numEntries; ++i) {
    is >> x >> y >> val;
    ERROR_IF(is.eof(), "Unexpected end of file!");
    ERROR_IF((x < 1) || (y < 1),
	     "Too small indices found in matrix file!");
    --x;
    --y;
    ERROR_IF((x >= result.size()), "Too large first index found in matrix file!");
    ERROR_IF((y >= result[x].size()), "Too large second index found in matrix file!");
    // cout << "Setting " << x << " " << y << " to " << val << endl;
    result[x][y] = val;
    if (mirrorMode) {
      result[y][x] = val;
    }
  }
  return result;
}

Vec<double>
readGAScores(istream& is)
{
  Vec<double> result;
  vector<string> words;
  while (is) {
    string line = getLine(is);
    if (line.size() == 0) {
      continue;
    }
    words = getTokens(line);
    if (words.size() < 2) {
      continue;
    }
    result.push_back(stod(words[1]));
  }
  return result;
}

/** returns Mathews coefficient between reference stems and prediction matrix*/
double 
computeMatrixMathews(const Vec<Vec<double> >& compMatrix,
		     const Vec<Stem>& referenceStems,
		     double scoreLimit,
		     double& accuracy,
		     double& coverage,
		     double& wrongPosFrac,
		     unsigned int& wrongPos,
		     unsigned int borderLim,
		     unsigned int diagLim)
{
  PRECOND(compMatrix.size() > 0);
  Vec<Vec<double> > stemMatrix(compMatrix.size(), 
			       Vec<double>(compMatrix.size(), 0.0));
  addStemsToMatrix(stemMatrix, referenceStems, 1.0);
  unsigned int tp = 0;
  unsigned int fp = 0;
  unsigned int tn = 0;
  unsigned int fn = 0;
  for (unsigned int i = borderLim; 
       i < static_cast<int>(stemMatrix.size()) - borderLim; ++i) {
    for (unsigned int j = borderLim; j + diagLim < i; ++j) {
      if (stemMatrix[i][j] > 0.0) {
	if (compMatrix[i][j] >= scoreLimit) {
	  ++tp;
	}
	else {
	  ++fn;
	}
      }
      else {
	if (compMatrix[i][j] >= scoreLimit) {
	  ++fp;
	}
	else {
	  ++tn;
	}
      }
    }
  }
  if ((tp + fp) == 0) {
    accuracy = 0.0;
  }
  else {
    accuracy = static_cast<double>(tp) / (tp + fp);
  }
  if ((tp + fn) == 0) {
    coverage = 0.0;
  }
  else {
    coverage = static_cast<double>(tp) / (tp + fn);
  }
  wrongPos = fp + fn; // number of wrong positions
  // divide by alignment length
  wrongPosFrac = static_cast<double>(wrongPos) 
    / static_cast<double>(compMatrix.size()); 
  return computeMathews(tp, fp, tn, fn);
}

double
structureSimilarity(const Vec<Stem>& stems,
		    const Vec<Stem>& refStems,
		    unsigned int length)
{
  PRECOND(length > 0);
  // cout << "Starting structureSimilarity!" << endl;
  double accuracy = 0.0;
  double coverage = 0.0;
  double wrongPosFrac = 0.0;
  unsigned int wrongPos = 0;
  unsigned int borderLim = 0;
  unsigned int diagLim = 0;

  /** convert stems to matrix */
  Vec<Vec<double> > m1 = generateMatrixFromStems(stems, length);

  double diff = computeMatrixMathews(m1,
				     refStems,
				     0.5,
				     accuracy,
				     coverage,
				     wrongPosFrac,
				     wrongPos,
				     borderLim,
				     diagLim);
  // cout << "Ending structureSimilarity!" << endl;
  return (1.0 - diff); // 1.0 leads to distance 0. Distance Mathews      
}

void
addBistabilityData(const Vec<Stem>& stems, 
		   const Vec<Stem>& refStems,
		   unsigned int length,
		   double refScore, 
		   double score,
		   double kt,
		   double structCutoff,
		   double& weightThisStructure, 
		   double& weightOtherStructure)
{
  // cout << "Starting addBistabilityData!" << endl;
  double sim = structureSimilarity(stems, refStems, length);
//   cout << sim << " : Structure similarity between "
//        << stems << " " << refStems << endl;
  if (sim < structCutoff) {
    weightThisStructure += exp(-(score-refScore)/kt);
  }
  else {
    weightOtherStructure += exp(-(score-refScore)/kt);
  }
  // cout << "Ending addBistabilityData!" << endl;
}


double
analyzeForBistability(const Vec<Vec<Stem> >& stems,
		      const Vec<Stem>& refStems,
		      double refScore,
		      const Vec<double>& scores,
		      double kt,
		      double structCutoff,
		      unsigned int length)
{
  PRECOND(refStems.size() > 0);
  cout << "Starting analyzeForBistability" << endl;
  double weightThisStructure = 0.0;
  double weightOtherStructure = 0.0;
  for (unsigned int i = 0; i < stems.size(); ++i) {
    addBistabilityData(stems[i], refStems, length, refScore, scores[i],
		       kt, structCutoff,
		       weightThisStructure, weightOtherStructure);
  }
  cout << "Ending analyzeForBistability" << endl;
  return weightThisStructure / (weightThisStructure + weightOtherStructure);
}


/** use measure of Hofacker 2001 */
double
analyzeForBistability2(const Vec<Vec<Stem> >& stems,
		       const Vec<Stem>& refStems,
		       double refScore,
		       const Vec<double>& scores,
		       double kt,
		       double structCutoff,
		       unsigned int length,
		       double equalWeight,
		       double freeEnergy,
		       double maxEnergy,
		       double maxEnergyWeight,
		       double bpDiffWeight,
		       unsigned int& altIdx)
{
  double UNFOLD_PENALTY = 100.0;
  if (refStems.size() == 0) {
    return 1e3; // returns bad high score
  }
  cout << "Starting analyzeForBistability" << endl;
  cout << "Reference stems: " << refScore << " " << endl;
  writeStems(cout, refStems);
  if (stems.size() < 2) {
    cout << "Warning: less than two structures defined!" << endl;
    return 0.0;
  }
//   double partition = 0.0;
//   for (unsigned int i = 0; i < scores.size(); ++i) {
//     partition += exp(-scores[i]/kt);
//   }
  // double probRef = exp(-refScore/kt)/partition;
  // find first not similar structure:
  altIdx = stems.size(); // index of first different structure
  for (unsigned int i = 0; i < stems.size(); ++i) {
    if (structureSimilarity(stems[i], refStems, length) > structCutoff) {
      altIdx = i;
      break;
    }
  }
  if (altIdx >= stems.size()) {
    cout << "No switch found!" << endl;
    return UNFOLD_PENALTY;
  }
  cout << "Found alternative stems: " << altIdx + 1 << " "
       << structureSimilarity(stems[altIdx], refStems, length)
       << " " << refScore << " " << scores[altIdx] << endl;
  writeStems(cout, stems[altIdx]);
  // double probAlt = exp(-scores[altIdx]/kt)/partition;
  double dScore = scores[altIdx] - refScore;
  double energyPenalty = 0;
  if (refScore > maxEnergy) {
    energyPenalty += refScore - maxEnergy;
  }
  if (scores[altIdx] > maxEnergy) {
    energyPenalty += scores[altIdx] - maxEnergy;
  }
  double unfoldPenalty = 0.0;
  if (refScore >= 0) {
    unfoldPenalty += UNFOLD_PENALTY;
  }
  if (scores[altIdx] >= 0) {
    unfoldPenalty += UNFOLD_PENALTY;
  }
  int bpDiff = abs(static_cast<int>(countNumBasePairs(refStems))
		   - static_cast<int>(countNumBasePairs(stems[altIdx])));  
  // see paper of Hofacker, now with modification
  double objectiveFunction = 1.0*(refScore + scores[altIdx]) 
    + ( maxEnergyWeight * energyPenalty )
    + ( bpDiffWeight * bpDiff )
    + unfoldPenalty
   - (2.0 * freeEnergy) + (equalWeight * dScore * dScore);
  return objectiveFunction;
}

string
convertStemsToTopology(const Vec<Stem>& stemsOrig, unsigned int length)
{
  Vec<Stem> stems;
  for (unsigned int i = 0; i < stemsOrig.size(); ++i) {
    if (stemsOrig[i].getLength() >= 2) {
      stems.push_back(stemsOrig[i]);
      stems[stems.size()-1].setLength(1);
    }
  }
  string topologyString = stemsToBracketFasta(stems, length);
  string dashes = ".-";
//   topologyString = trim(topologyString, dashes);
//   topologyString = squeezeRepeats(topologyString, dashes);
  topologyString = removeFromString(topologyString, dashes);
  return topologyString;
}

int
main(int argc, char ** argv)
{
  bool bistabilityMode = false;
  bool helpMode;
  int argcFile = 0;
  char ** argvFile = 0;
  unsigned int maxScores = 0;
  unsigned int minStemLength = 1;
  int verboseLevel = 0;
  double bpDiffWeight = 1.0;
  double cutoff = 0.5;
  double equalWeight = 1.0;
  double energyMax = -1.0;
  double energyMaxWeight = 5.0;
  double freeEnergy = 0.0;
  double initialScoreWeight = 1.0;
  double kt = 1.0;
  double penalty = 10.0;
  double pseudoCount = 0; // used for counting false/true positives/negatives
  double refScore = 0.0;
  // double structCutoff = 0.9;
  unsigned int pkCounter = 0;
  int algorithm = 0; // 2;
  int probMatrixFormat = 3;
  int scoreFormat = 1;
  int sequenceFormat = 2;
  int stemFormat = 1;
  int outputFormat = 4;
  int refStemFormat = 2;
  string commandFileName;
  string inputFileName;
  string logFileName; //  = "mainprogramtemplate.log";
  string outFileName = "ranksec.out.matrix";
  string parameterFileName;
  string pkFileName;
  string probMatrixFileName;
  string rootDir = ".";
  string refStemFileName;
  string sequenceFileName;
  string scoreFileName;
  string sequence;
  
  SimpleSequenceAlignment ali;
  Vec<Stem> referenceStems;
  Vec<Vec<Stem> > stemVec;
  Vec<Vec<Stem> > pkStemVec;
  Vec<double> initialScores(stemVec.size(), 0.0);

  getArg("-help", helpMode, argc, argv);

  if ((argc < 2) || helpMode)  {
    helpOutput(cout);
    exit(0);
  }
  getArg("-root", rootDir, argc, argv, rootDir);
  addSlash(rootDir);
  getArg("-commands", commandFileName, argc, argv, commandFileName);
  addPathIfRelative(commandFileName, rootDir);

  if (commandFileName.size() > 0) {
    ifstream commandFile(commandFileName.c_str());
    if (!commandFile) {
      if (isPresent("-commands", argc, argv)) {
	ERROR_IF(!commandFile, "Error opening command file.");
      }
      else {
	cerr << "Warning: Could not find command file: " + commandFileName 
	     << endl;
      }
    }
    else {
      argvFile = streamToCommands(commandFile, argcFile, 
				  string("mainprogramtemplate"));
    }
    commandFile.close();
  }

  getArg("-algorithm", algorithm, argcFile, argvFile, algorithm);
  getArg("-algorithm", algorithm, argc, argv, algorithm);
  getArg("-cutoff", cutoff, argcFile, argvFile, cutoff);
  getArg("-cutoff", cutoff, argc, argv, cutoff);
  getArg("-diff-weight", bpDiffWeight, argcFile, argvFile, bpDiffWeight);
  getArg("-diff-weight", bpDiffWeight, argc, argv, bpDiffWeight);
  getArg("-energy-max", energyMax, argcFile, argvFile, energyMax);
  getArg("-energy-max", energyMax, argc, argv, energyMax);
  getArg("-energy-weight", energyMaxWeight, argcFile, argvFile, energyMaxWeight);
  getArg("-energy-weight", energyMaxWeight, argc, argv, energyMaxWeight);
  getArg("-equal", equalWeight, argcFile, argvFile, equalWeight);
  getArg("-equal", equalWeight, argc, argv, equalWeight);
  getArg("-free", freeEnergy, argcFile, argvFile, freeEnergy);
  getArg("-free", freeEnergy, argc, argv, freeEnergy);
  getArg("i", inputFileName, argcFile, argvFile, inputFileName);
  getArg("i", inputFileName, argc, argv, inputFileName);
  getArg("-if", stemFormat, argcFile, argvFile, stemFormat);
  getArg("-if", stemFormat, argc, argv, stemFormat);
  getArg("-kt", kt, argcFile, argvFile, kt);
  getArg("-kt", kt, argc, argv, kt);
  getArg("-log", logFileName, argc, argv, logFileName);
  getArg("-log", logFileName, argcFile, argvFile, logFileName);
  addPathIfRelative(logFileName, rootDir);
  getArg("n", maxScores, argcFile, argvFile, maxScores);
  getArg("n", maxScores, argc, argv, maxScores);
  getArg("o", outFileName, argcFile, argvFile, outFileName);
  getArg("o", outFileName, argc, argv, outFileName);
  getArg("-of", outputFormat, argcFile, argvFile, outputFormat);
  getArg("-of", outputFormat, argc, argv, outputFormat);
  getArg("p", parameterFileName, argcFile, argvFile, parameterFileName);
  getArg("p", parameterFileName, argc, argv, parameterFileName);
  getArg("-penalty", penalty, argcFile, argvFile, penalty);
  getArg("-penalty", penalty, argc, argv, penalty);

  getArg("-pk", pkFileName, argcFile, argvFile, pkFileName);
  getArg("-pk", pkFileName, argc, argv, pkFileName);

  getArg("-prob-matrix", probMatrixFileName, argcFile, argvFile, probMatrixFileName);
  getArg("-prob-matrix", probMatrixFileName, argc, argv, probMatrixFileName);
  getArg("-prob-matrix-format", probMatrixFormat, argcFile, argvFile, probMatrixFormat);
  getArg("-prob-matrix-format", probMatrixFormat, argc, argv, probMatrixFormat);
  getArg("-pseudo", pseudoCount, argcFile, argvFile, pseudoCount);
  getArg("-pseudo", pseudoCount, argc, argv, pseudoCount);
  getArg("-scores", scoreFileName, argcFile, argvFile, scoreFileName);
  getArg("-scores", scoreFileName, argc, argv, scoreFileName);
  getArg("-score-weight", initialScoreWeight, argcFile, argvFile, initialScoreWeight);
  getArg("-score-weight", initialScoreWeight, argc, argv, initialScoreWeight);
  getArg("-score-format", scoreFormat, argcFile, argvFile, scoreFormat);
  getArg("-score-format", scoreFormat, argc, argv, scoreFormat);
  getArg("-ref-stems", refStemFileName, argcFile, argvFile, refStemFileName);
  getArg("-ref-stems", refStemFileName, argc, argv, refStemFileName);
  getArg("-stem-format", stemFormat, argcFile, argvFile, stemFormat);
  getArg("-stem-format", stemFormat, argc, argv, stemFormat);
  getArg("-ref-format", refStemFormat, argcFile, argvFile, refStemFormat);
  getArg("-ref-format", refStemFormat, argc, argv, refStemFormat);
  getArg("-sequence", sequenceFileName, argcFile, argvFile, sequenceFileName);
  getArg("-sequence", sequenceFileName, argc, argv, sequenceFileName);
  getArg("-sequence-format", sequenceFormat, argcFile, argvFile, sequenceFormat);
  getArg("-sequence-format", sequenceFormat, argc, argv, sequenceFormat);
  getArg("-verbose", verboseLevel, argcFile, argvFile, verboseLevel);
  getArg("-verbose", verboseLevel, argc, argv, verboseLevel);

  if (logFileName.size() > 0) {
    ofstream logFile(logFileName.c_str(), ios::app);
    parameterOutput(logFile, argc, argv);
    if (argcFile > 1) {
      logFile << "Parameters from command file: ";
      parameterOutput(logFile, argcFile, argvFile);
    }
    logFile.close();
  }

  /***************** MAIN PROGRAM *****************************/
  if (verboseLevel > 0) {
    cout << "ranksec called with following parameters: " << endl;
    parameterOutput(cout, argc, argv);  
    if (argcFile > 0) {
      cout << "parameters read from file:" << endl;
      parameterOutput(cout, argcFile, argvFile);
    }
  }
  
  Vec<double> probContacts, probNoContacts;
  if (parameterFileName.size() > 0) {
    if (verboseLevel > 0) {
      cout << "Reading parameter file: " << parameterFileName
	   << endl;
    }
    ifstream parameterFile(parameterFileName.c_str());
    ERROR_IF(!parameterFile, "Error opening parameter file!");
    parameterFile >> probContacts >> probNoContacts;
    parameterFile.close();
  }

  ifstream inputFile(inputFileName.c_str());
  ERROR_IF(!inputFile, "Error opening input file!");

  switch (stemFormat) {
  case 1:
    // region file
    stemVec = readMultiStems(inputFile);
    break;
  case 2:
    // bracket notation: not implemented for multiple entries!
    // stemVec = stemsFromBracketFasta(inputFile);
    ERROR("stem-format 2 (bracket notation) currently not implemented for solution files!");
    break;
  case 3: 
    stemVec = readCtCombined(inputFile, sequence, minStemLength);
    break;
  case 4:
    if (verboseLevel > 0) {
      cout << "starting readRNAsubopt!" << endl;
    }
    stemVec = readRNAsubopt(inputFile, initialScores, sequence);
    break;
  case 5:
    // TODO : implement!
//     cout << "starting readRNAsubopt (-p option)!" << endl;
//     stemVec = readRNAsuboptP(inputFile);
    break;
  default:
    ERROR("Unknown stem file format!");
  }

  if (verboseLevel > 0) {
    cout << "Number of read secondary structures: " << stemVec.size() << endl;
    for (unsigned int i = 0; i < stemVec.size(); ++i) {
      cout << i + 1 << " " << stemVec[i];
      if (i < initialScores.size()) {
	cout << " " << initialScores[i] << endl;
      }
    }
  }

  ERROR_IF(stemVec.size() == 0,
	   "No query stems read!");

  if (pkFileName.size() > 0) {
    ifstream pkFile(pkFileName.c_str());
    ERROR_IF(!pkFile, "Error opening input file!");    
    switch (stemFormat) {
    case 1:
      // region file
      pkStemVec = readMultiStems(pkFile);
      break;
    case 2:
      // bracket notation: not implemented for multiple entries!
      // stemVec = stemsFromBracketFasta(inputFile);
      ERROR("Bracked notation not implemented!");
      break;
    case 3: 
      pkStemVec = readCtCombined(pkFile, sequence, minStemLength);
      break;
    case 4:
      if (verboseLevel > 0) {
	cout << "# starting readRNAsubopt!" << endl;
      }
      pkStemVec = readRNAsubopt(pkFile, initialScores, sequence);
      break;
    case 5:
      // TODO
//       cout << "starting readRNAsubopt (-p option)!" << endl;
//       pkStemVec = readRNAsuboptP(pkFile);
      break;
    default:
      ERROR("Unknown stem file format!");
    }
    pkFile.close();
    if (verboseLevel > 1) {
      cout << "# " << pkStemVec.size() << " pseudo-knot structures read!"
	   << endl;
      cout << "Read pseudo-knots: " << endl << pkStemVec << endl;
    }
    ERROR_IF(pkStemVec.size() != stemVec.size(), 
	     "Number of pk structures must equal number of initial structures!");
    for (unsigned int i = 0; i < stemVec.size(); ++i) {
      for (unsigned int j = 0; j < pkStemVec[i].size(); ++j) {
	// ignore stems with length zero:
	if (pkStemVec[i][j].getLength() > 0) {
	  stemVec[i].push_back(pkStemVec[i][j]); // adding to one structure
	}
      }
    }
  }

  if (refStemFileName.size() > 0) {
    if (verboseLevel > 0) {
      cout << "# Reading reference stems from: " << refStemFileName << endl;
    }
    ifstream stemFile(refStemFileName.c_str());
    ERROR_IF(!stemFile, "Error opening stem file!");
    switch (refStemFormat) {
    case 1:
      // region file
      referenceStems = readStems(stemFile);
      break;
    case 2:
      // bracket notation
      referenceStems = stemsFromBracketFasta(stemFile, pkCounter);
      break;
    default:
      ERROR("Unknown stem file format for reference stems!");
    }
    stemFile.close();
  }
  if(referenceStems.size() == 0) {
    if (verboseLevel > 0) {
      cout << "# Warning: No reference stems defined! Using first structure." << endl;
    }
    referenceStems = stemVec[0];
    if (verboseLevel > 0) {
      writeStems(cout, referenceStems);
    }
    if (initialScores.size() > 0) {
      refScore = initialScores[0];
    }
  }
  else if (verboseLevel > 0) {
    cout << "Defined reference stems: " << endl;
    writeStems(cout, referenceStems);
  }
  
  if (sequenceFileName.size() > 0) {
    ifstream sequenceFile(sequenceFileName.c_str());
    ERROR_IF(!sequenceFile, "Error opening sequence file!");
    switch (sequenceFormat) {
    case 1:
      if (verboseLevel > 0) {
	cout << "# Reading sequence in Fasta format!" << endl;
      }
      ali.readFasta(sequenceFile);
      break;
    case 2:
      if (verboseLevel > 0) {
	cout << "# Reading sequence in Zuker format!" << endl;
      }
      sequence = readZuker(sequenceFile);
      ali.addSequence(sequence, "sequence");
      break;
    default:
      ERROR("Unknown sequence file format!");
    }
    sequenceFile.close();
  }
  ERROR_IF(stemVec.size() == 0, "No secondary structures defined!");
  if (verboseLevel > 0) {
    cout << "# " <<  stemVec.size() << " structures read" << endl;
  }
  if (ali.size() > 0) {
    if (verboseLevel > 0) {
      cout << "# Length of read sequence: " << ali.getLength() << endl;
      cout << "# Used sequence: " << endl
	   << ali.getSequence(0) << endl;
    }
  }
  else if (sequence.size() > 0) {
    ali.addSequence(sequence, "sequence");
  }
  unsigned int length = ali.getLength();
  ERROR_IF(length == 0, "No sequence length specified!");
  string refTopology = convertStemsToTopology(referenceStems, length);
  if (scoreFileName.size() > 0) {
    if (verboseLevel > 0) {
      cout << "# Reading score file: " << scoreFileName << endl;
    }
    ifstream scoreFile(scoreFileName.c_str());
    ERROR_IF(!scoreFile, "Error opening score file!");
    switch (scoreFormat) {
    case 1:
      scoreFile >> initialScores;
      ERROR_IF(initialScores.size() != stemVec.size(), "Number of initial scores does not match number of structures!");
      break;
    case 2: // used by GA
      initialScores = readGAScores(scoreFile);
      break;
    default:
      ERROR("Unknown score file format!");
      scoreFile.close();
    }
    if (verboseLevel > 1) {
      cout << "# Read scores: " << initialScores << endl;
    }
  }
  if (initialScores.size() != stemVec.size()) {
    if (verboseLevel > 0) {
      cout << "# Setting initial scores to zero!" << endl;
    }
    initialScores = Vec<double>(stemVec.size(), 0.0);
  }
  if (initialScoreWeight != 1.0) {
    for (unsigned int i = 0; i < initialScores.size(); ++i) {
      initialScores[i] *= initialScoreWeight;
    }
  }

  if (bistabilityMode && (initialScores.size() > 0)) {
    for (int i = 1; i < 10; ++i) {
      double lStructCutoff = i * 0.1;
      if (verboseLevel > 0) {
	cout << "Working on Mathews coefficient cutoff: " << lStructCutoff << " ";
      }
      unsigned int altIndex = 0;
      double result = analyzeForBistability2(stemVec, referenceStems,
	       refScore, initialScores, kt,
	     lStructCutoff, length,
	       equalWeight, freeEnergy, energyMax, 
	       energyMaxWeight, bpDiffWeight, altIndex);
      if (altIndex < stemVec.size()) {
	if (verboseLevel > 0) {
	  cout << "Mathews cutoff: " << lStructCutoff << " Weight of top structure versus weight of other structures: " 
	       << result << " index: " << altIndex + 1 << " topologies: " ; //  << endl;
	  cout << refTopology << " " << convertStemsToTopology(stemVec[altIndex], length)
	       << endl;
	}
      }
      else {
	if (verboseLevel > 0) {
	  cout << "No switch found!" << endl;
	}
      }
    }
  }
  else {
    if (verboseLevel > 0) {
      cout << "Not analyzing for bistability." << endl;
    }
  }

  // generate consensus probability matrix:
  Vec<Vec<double> > consProbMatrix;
  if (probMatrixFileName.size() > 0) {
    if (verboseLevel > 0) {
      cout << "Reading probability matrix file: " << probMatrixFileName 
	   << endl;
    }
    ifstream probMatrixFile(probMatrixFileName.c_str());
    ERROR_IF(!probMatrixFile, "Error opening probability matrix file!");
    switch (probMatrixFormat) {
    case 1: 
      probMatrixFile >> consProbMatrix;
      break;
    case 2:
      // true: use mirror image too
      consProbMatrix = readListMatrix(probMatrixFile, 0.0, true); 
      break;
    case 3:
      if (verboseLevel > 0) {
	cout << "Reading plain matrix!" << endl;
      }
      consProbMatrix = readPlainMatrix(probMatrixFile);
      break;
    default:
      ERROR("Unknown probability matrix file format!");
    }
    probMatrixFile.close();
  }
  else {
    ERROR_IF((ali.size() == 0),
	     "No probability matrix or aligmment specified");
    if (referenceStems.size() == 0) {
      if (verboseLevel > 0) {
	cout << "Warning: No probability matrix and reference stems specified!"
	     << endl;
      }
    }
    consProbMatrix = Vec<Vec<double> >(ali.getLength(),
				       Vec<double>(ali.getLength(), 0.0));
    addStemsToMatrix(consProbMatrix, referenceStems, 1);    
  }
  inputFile.close();
  Vec<double> newScores;
  switch (algorithm) {
  case 0: newScores = initialScores;
    break;
  case 1: newScores = computeNewStemScores(stemVec, consProbMatrix, 
					   initialScores, 
					   initialScoreWeight, kt,
					   cutoff, penalty);  
    break;
  case 2: newScores = computeNewStemScores2(stemVec, consProbMatrix, 
					    initialScores, 
					    initialScoreWeight, kt,
					    cutoff, penalty);  
    break;
  case 3: newScores = computeNewStemScores3(stemVec, consProbMatrix, 
					    initialScores, 
					    initialScoreWeight, kt,
					    cutoff, penalty,
					    probContacts,
					    probNoContacts);  
    break;
  case 4: newScores = computeNewStemScores4(initialScores);
    break;
  default:
    ERROR("Unknown algorithm id!");
  }
  unsigned int bestId = 0;
  cout << "# using output format " << outputFormat << endl;
  switch (outputFormat) {
  case 0:
    bestId = outputScores(cout, newScores, stemVec, referenceStems, ali.getLength(),
			  maxScores, verboseLevel, pseudoCount );
    break;
  case 4:
    outputMultiCt(cout, newScores, stemVec, sequence);
    break;
  default:
    ERROR("Unknown output format!");
  }
  if (ali.getLength() > 0) {
    ofstream outfile(outFileName.c_str());
    ERROR_IF(!outfile, "Error opening output file!");
    
    writeMatrix(outfile, generateMatrixFromStems(stemVec[bestId],
						 ali.getLength()));
    outfile.close();
  }

  ERROR_IF(ali.size() == 0,
	   "No sequence data specified! Use option --sequence");

  return 0;
}
