// --*- C++ -*------x---------------------------------------------------------
// $Id: stemconvert.cc,v 1.33 2012-01-04 14:48:32 eckart Exp $
//
// Program:         - 
//
// Author:          Eckart Bindewald
//
// Project name:    -
//
// Date:            $Date: 2012-01-04 14:48:32 $
//
// Description:     
// 
// -----------------x-------------------x-------------------x-----------------

#include <iostream>
#include <fstream>
#include <string>
#include <Vec.h>
#include <debug.h>
#include <GetArg.h>
#include <FileName.h>
#include <stemhelp.h>
#include <Stem.h>
#include <SequenceAlignment.h>
#include <SimpleSequenceAlignment.h>
#include <SequenceAlignmentTools.h>
#include <ctype.h>
#include <compass_help.h>
#include <CompensationScorer.h>
#include <vectornumerics.h>
#include <StemTools.h>
#include <ProbeStemEnergyModel.h>
#include <RnaSecondaryStructure.h>
#include <RnaSecondaryStructureTools.h>
// #include <MatchFoldTools.h>
#include <RnaFormats.h>

using namespace std;

typedef unsigned int size_type;
typedef string sequence_type;

#define STEMCONVERT_VERSION "0.8.5"

/** version history 
 * 0.8.1 fixed issue with reading BPL format file 
 * 0.8.2 improved output of multiple sequences in FASTA format
 * 0.8.3 added RNAalifold pairs output
 * 0.8.4 added option --starts-alignment
 * 0.8.5 added option --stem-energy VALUE for default stem energies.
 */

void
helpOutput(ostream& os)
{
  os << "stemconvert version " << STEMCONVERT_VERSION << endl;
  os << "The program converts between different secondary structure file formats." << endl;
  os << "usage: stemconvert -i inputfile --if 1|2|3|4 --of 1|2|3|4 " << endl
     << " additional non-mandatory options: " << endl
     << "-a sequencefile : read sequence (alignment) in FASTA format" << endl
     << "--collapse id  : remove gaps with respect to n'th sequence" << endl
     << "--consensus fraction : remove base pairs for which fraction of complementarity is lower than given threshold" << endl
     << "--expand expandid Expanding gaps in secondary structure with respect to sequence " << endl
     << "--gap-frac-alpha alpha : sets all matrix elements to zero that correspond to alignment columns with less than fraction non-gap characters" << endl
     << "--matrix filename : external matrix for structure comparison"
     << "--offset indexoffset : defines index offset for output"
     << "-l threshold : threshold converting matrix to base pairs" << endl
     << "--length length : specify length of secondary structure (necessary for some conversions, for example region-file-format (which does not contain lengths))" << endl
     << "--stem-energy energy     : default energy given to each stem." << endl
     << "--stem-min minstemlength : filter out all stems that are shorter than specified minimum stem length" << endl
     << "--subset positions : use only specified postions. Example --subset 120-140,156-158" << endl
     << "--wc sequenceId : remove all non-complementary base-pairs of secondary structure according to alignment and sequence id" << endl;
  os << "Input Format ids: (mandatory for options --if inputformat )" << endl;
  os << "1: region file" << endl
     << "2: bracket notation" << endl
     << "3: square contact matrix" << endl
     << "4: ct file format" << endl
     << "6: BPL file format." << endl
     << "8: RNAfold output format." << endl
     << "9: WMatch output format." << endl
     << "10: length (output only)" << endl
     << "11: Stockholm format (output only)" << endl
     << "17: RNAeval format (header, sequence, bracket notation in 3 lines)" << endl
     << "experimental format options: " << endl
     << "18: pseudoviewer format (output only)" << endl;
  os << "Output Format ids: (mandatory for options --of outputformat )" << endl;
  os << "1: region file" << endl
     << "2: bracket notation" << endl
     << "3: square contact matrix" << endl
     << "4: ct file format" << endl
     << "5: topology" << endl
     << "6: BPL file format." << endl
     << "7: stem statistics." << endl
     << "8: RNAfold output format." << endl
     << "9: Individuals." << endl
     << "10: Individuals (not using grow step)" << endl
     << "11: Stockholm format (output only)" << endl
     << "12: length" << endl
     << "15: another statistics" << endl
     << "16: MCSym" << endl
     << "17: RNAeval format (header, sequence, bracket notation in 3 lines)" << endl
     << "experimental format options: " << endl
     << "18: pseudoviewer format (output only)" << endl
     << "there is more: up to --of 31! To do." << endl
     << "31: RNAalifold pairs" << endl;
}

/** output of command line parameter with which the program was called. */
void
parameterOutput(ostream& os, int argc, char** argv)
{
  for (int i = 0; i < argc; i++)
    {
      os << argv[i] << " ";
    }
  os << endl;
}

/** removes gap at position col of sequence with number collapseId 
    in alignment,
    adjusts alignment, probability matrix and reference stems. */
void
collapseAlignment(SequenceAlignment& ali, 
		  unsigned int collapseId, 
		  unsigned int col,
		  Vec<Stem>& referenceStems)
{
  if ((collapseId >= ali.size()) || (col >= ali.getLength())
      || (ali.getLength() < 2)) {
    ERROR("Internal error in line 2584!");
    return;
  }
  deletePositionInStems(referenceStems, ali.getSequence(collapseId), col); // order important!
  ali.deleteColumn(col);
}

/** removes gap at position col of sequence 
    adjusts sequence, probability matrix and reference stems. */
void
collapseSequence(string& seq,
		 unsigned int col,
		 Vec<Stem>& referenceStems)
{
  deletePositionInStems(referenceStems, seq, col); // order important!
  seq.erase(col, 1);
}

/** removes all gaps of sequence with number collapseId in alignment,
    adjusts alignment, probability matrix and reference stems. */
void
collapseAlignment(SequenceAlignment& ali, 
		  unsigned int collapseId, 
		  Vec<Stem>& referenceStems)
{
  if (collapseId >= ali.size()) {
    return;
  }
  for (int i = ali.getLength()-1; i >= 0; --i) {
    if ((ali.getSequence(collapseId)[i] == '.')
	|| (ali.getSequence(collapseId)[i] == '-')) {
      collapseAlignment(ali, collapseId, 
	static_cast<unsigned int>(i), referenceStems);
    }
  }
}

/** removes all gaps of sequence with number collapseId in alignment,
    adjusts alignment, probability matrix and reference stems. */
void
collapseSequence(string& seq,
		 Vec<Stem>& referenceStems)
{
  for (int i = seq.size()-1; i >= 0; --i) {
    if ((seq[i] == '.') || (seq[i] == '-')) {
      collapseSequence(seq,
		       static_cast<unsigned int>(i), referenceStems);
    }
  }
}

Vec<unsigned int>
getGapSubset(const string& s)
{
  Vec<unsigned int> result;
  for (unsigned int i = 0; i < s.size(); ++i) {
    if ((s[i] == '.') || (s[i] == '-')) {
      result.push_back(i);
    }
  }
  return result;
}

Vec<unsigned int>
getNoGapSubset(const string& s)
{
  Vec<unsigned int> result;
  for (unsigned int i = 0; i < s.size(); ++i) {
    if (!((s[i] == '.') || (s[i] == '-'))) {
      result.push_back(i);
    }
  }
  return result;
}

/** removes all gaps of sequence and matrix
 * with respect to sequence */
Vec<Vec<double>  >
collapseAlignment(const string& seq, 
		  const Vec<Vec<double> >& matrix)
{
  Vec<unsigned int> noGapSubset = getNoGapSubset(seq);
  string seqNoGap = getSubset(seq, noGapSubset); // gap free string
  // collapse matrix according to gaps:
  return getMatrixSubset(matrix, noGapSubset);
}


/** removes gap at position col of sequence with number collapseId 
    in alignment,
    adjusts alignment, probability matrix and reference stems. */
void
expandSecondary(const SequenceAlignment& ali, 
		unsigned int collapseId, 
		unsigned int col,
		string& currentSequence,
		Vec<Stem>& referenceStems)
{
  if ((collapseId >= ali.size()) || (col >= ali.getLength())
      || (ali.getLength() < 2)) {
    ERROR("Internal error in line 2584!");
    return;
  }
  insertGapInStems(referenceStems, currentSequence, col); // order important!
}

/** removes all gaps of sequence with number collapseId in alignment,
    adjusts alignment, probability matrix and reference stems. */
string
expandSecondary(const SequenceAlignment& ali, 
		unsigned int expandId, 
		Vec<Stem>& referenceStems)
{
  ERROR_IF(expandId >= ali.size(), "Internal error in line 108!");
  string currentSequence = ali.getSequence(expandId);
  currentSequence = removeFromString(currentSequence, '-');
  currentSequence = removeFromString(currentSequence, '.');
  for (int i = 0; i < static_cast<int>(ali.getLength()); ++i) {
    if ((ali.getSequence(expandId)[i] == '.') || (ali.getSequence(expandId)[i] == '-')) {
      expandSecondary(ali, expandId, static_cast<unsigned int>(i), 
		      currentSequence, referenceStems);
    }
  }
  return currentSequence;
}

void
insertGapInMatrix(Vec<Vec<double> >& matrix,
		  unsigned int pos,
		  double val)
{
  // insert new columns:
  for (unsigned int i = 0; i < matrix.size(); ++i) {
    matrix[i].insert(matrix[i].begin()+pos, val);
  }
  // insert whole new row: length one element greater ! 
  matrix.insert(matrix.begin()+pos, Vec<double>(matrix.size()+1, val));
}

/** expands matrix in accordance to alignment sequence expandId */
void
expandMatrix(const string& seq,
	     Vec<Vec<double> >& matrix)
{
  int nn = seq.size();
  for (int i = 0; i < nn; ++i) {
    // TODO : cleaner implementation of gap characters
    if ((seq[i] == '.') || (seq[i] == '-')) {
      insertGapInMatrix(matrix, i, 0.0);
    }
  }
}

int
countConflictingBasePairs(const Vec<Stem>& stems,
			  unsigned int len,
			  double thresh)
{
  Vec<Vec<double> > matrix = generateMatrixFromStems(stems, len);
  unsigned int counter = 0;
  for (unsigned int i = 0; i < matrix.size(); ++i) {
    unsigned int threshCount = 0;
    for (unsigned int j = 0; j < matrix.size(); ++j) {
      if (i == j) {
	continue;
      }
      if (matrix[i][j] >= thresh) {
	++threshCount;
      }
    }
    if (threshCount > 1) {
      ++counter;
    }
  }
  return counter;
}

/** reads matrix in format that is also used by program "R" */
Vec<Vec<double> >
readListMatrix(istream& is, unsigned int length, double thresh)
{
  Vec<Vec<double > > result(length, Vec<double>(length, 0.0));
  Vec<string> words;
  while (is) {
    string line = getLine(is);
    if ((line.size() == 0) || (line[0] == '#')) {
      continue;
    }
    words = getTokens(line);
    if (words.size() >= 3) {
      unsigned int id1 = stoui(words[0])-1;
      unsigned int id2 = stoui(words[1])-1;
      double val = stod(words[2]);
      if (val >= thresh) {
	result[id1][id2] = 1.0;
      }
    }
  }
  return result;
}

string
convertStemsToTopology(const Vec<Stem>& stemsOrig, unsigned int length)
{
  Vec<Stem> stems;
  for (unsigned int i = 0; i < stemsOrig.size(); ++i) {
    if (stemsOrig[i].getLength() >= 2) {
      stems.push_back(stemsOrig[i]);
      stems[stems.size()-1].setLength(1);
    }
  }
  string topologyString = stemsToBracketFasta(stems, length);
  string dashes = ".-";
//   topologyString = trim(topologyString, dashes);
//   topologyString = squeezeRepeats(topologyString, dashes);
  topologyString = removeFromString(topologyString, dashes);
  return topologyString;
}

void
writeStemStatistics(ostream& os, 
		    const Vec<Stem>& stems, 
		    const Vec<Vec<double> >& extMatrix, 
		    int stemStatLen)
{
  for (Vec<Stem>::size_type i = 0; i < stems.size(); ++i) {
    if (stems[i].getLength() == stemStatLen) {
      for (int j = 0; j < stems[i].getLength(); ++j) {
	ERROR_IF((stems[i].getStart() + j >= static_cast<int>(extMatrix.size())) 
		 || ((stems[i].getStop() -j) >= static_cast<int>(extMatrix.size())),
		 "Unsufficient matrix definition!");
	os << extMatrix[stems[i].getStart() + j][stems[i].getStop()-j] << " ";
      }
      os << endl;
    }
  }

}


/** returns fraction of complementary base pairs */
double
computeComplementaryFraction(const SequenceAlignment& ali, 
			     unsigned int i, 
			     unsigned int j)
{
  string s1 = ali.getColumn(i);
  string s2 = ali.getColumn(j);
  unsigned int complementCount = 0;
  for (unsigned int k = 0; k < s1.size(); ++k) {
    if (isComplementary(s1[k], s2[k])) {
      ++complementCount;
    }
  }
  return static_cast<double>(complementCount) / static_cast<double>(s1.size());
}

/** removes all non-complementary base pairs from stem */
Vec<Stem>
removeNonComplementary(const string& seq, const Vec<Stem>& stems)
{
  unsigned int nn = seq.size();
  Vec<Vec<double> > matrix = generateMatrixFromStems(stems, nn);
  for (unsigned int i = 0; i < nn; ++i) {
    for (unsigned int j = 0; j < i; ++j) {
      if (!isComplementary(seq[i], seq[j])) {
	matrix[i][j] = 0.0;
	matrix[j][i] = 0.0;
      }
    }
  }
  return generateStemsFromMatrix(matrix, 1, 0.5, seq);
}

/** removes all non-complementary base pairs from stem */
Vec<Stem>
removeNonConsensus(double consensusFrac, 
		   const SequenceAlignment& ali, 
		   const Vec<Stem>& stems)
{
  unsigned int nn = ali.getLength();
  Vec<Vec<double> > matrix = generateMatrixFromStems(stems, nn);
  for (unsigned int i = 0; i < nn; ++i) {
    for (unsigned int j = 0; j < i; ++j) {
      if ((matrix[i][j] > 0.0) 
	  && (computeComplementaryFraction(ali, i, j) < consensusFrac)) {
	matrix[i][j] = 0.0;
	matrix[j][i] = 0.0;
      }
    }
  }
  return generateStemsFromMatrix(matrix, 1, 0.5, ali.getSequence(0));
}

/** returns true, if position x,y is conflicting already with
 * another position given matrix and threshold 
 */
bool
isNonConflictingPair(const Vec<Vec<double> >& matrix,
		  int x, 
		  int y, 
		  double thresh)
{
  unsigned int nn = matrix.size();
  // check all x values:
  for (unsigned int i = 0; i < nn; ++i) {
    if (static_cast<int>(i) == x) {
      continue;
    }
    if (matrix[i][y] >= thresh) {
      return false;
    }
  }
  // check all other y values:
  for (unsigned int i = 0; i < nn; ++i) {
    if (static_cast<int>(i) == y) {
      continue;
    }
    if (matrix[x][i] >= thresh) {
      return false;
    }
  }
  return true;
}

/** similar to simpleGrow, but in this case extending stem as long as nucleotides of 
    reference sequence are matching
    only grow if now conflicts
*/
int
simpleGrow3(Vec<Vec<double> >& compMatrix, 
	    const string& sequence,
	    double simpleGrowLimit,
	    const CompensationScorer& scorer)
{
  int x, y;
  int nn = compMatrix.size();
  int numChanged = 0;
  for (int i = 0; i < static_cast<int>(compMatrix.size()); ++i) {
    for (int j = i + 3; j < static_cast<int>(compMatrix[i].size()); ++j) {
      if (compMatrix[i][j] > simpleGrowLimit) {
	for (int k = 1; ; ++k) {
	  x = i + k;
	  y = j - k;
	  if ((x >= nn) || (y < 0)) {
	    break;
	  }
	  if ((compMatrix[x][y] < simpleGrowLimit)
	      && (scorer.isAllowedPair(sequence[x], sequence[y]))
	      && isNonConflictingPair(compMatrix, x, y, simpleGrowLimit)) {
	    compMatrix[x][y] = compMatrix[i][j];
	    compMatrix[y][x] = compMatrix[x][y];
	    ++numChanged;
	  }
	  else {
// 	    cout << "Breaking up search at : " << i +1  << " " << j + 1 << " "
// 		 << compMatrix[i][j] << " " << k << " " << x << " " << y << " " << compMatrix[x][y]
// 		 << fracMatching(ali, static_cast<unsigned int>(x), static_cast<unsigned int>(y), 
// 				 scorer, Alignment::GAP_CHAR) << endl;
	    break;
	  }
	}
	for (int k = -1; ; --k) {
	  x = i + k;
	  y = j - k;
	  if ((x < 0) || (y >= nn)) {
	    break;
	  }
	  if ((compMatrix[x][y] < simpleGrowLimit)
	      && (scorer.isAllowedPair(sequence[x], sequence[y]))
	      && isNonConflictingPair(compMatrix, x, y, simpleGrowLimit)) {
	    compMatrix[x][y] = compMatrix[i][j];
	    compMatrix[y][x] = compMatrix[x][y];
	    ++numChanged;
	  }
	  else {
// 	    cout << "Breaking up search at : " << i +1  << " " << j + 1 << " "
// 		 << compMatrix[i][j] << " " << k << " " << x << " " << y << " " << compMatrix[x][y]
// 		 << fracMatching(ali, static_cast<unsigned int>(x), static_cast<unsigned int>(y), 
// 				 scorer, Alignment::GAP_CHAR) << endl;
	    break;
	  }
	}
      }
    }
  }
  return numChanged;
}

void
removeNonMatching(Vec<Vec<double> >& consMatrix, 
		  const string& seq, 
		  const CompensationScorer& scorer)
{
  for (unsigned int i = 0; i < seq.size(); ++i) {
    for (unsigned int j = 0; j < seq.size(); ++j) {
      if (!isComplementary(seq[i],seq[j])) {
	consMatrix[i][j] = 0.0;
      }
    }
  }
}

/** uses full copy of matrix representing consensus secondary structure */
void
writeIndividual(ostream& os, const Vec<Vec<double> >& consMatrix,
		const string& seq,
		const CompensationScorer& scorer,
		int outputFormat)
{
  // grow matrix: if complementary base pair next to exisiting stem 
  Vec<unsigned int> noGapSubset = getNoGapSubset(seq);
  string seqNoGap = getSubset(seq, noGapSubset); // gap free string
  // collapse matrix according to gaps:
  Vec<Vec<double> > collMatrix = getMatrixSubset(consMatrix, noGapSubset);
  // and not conflicting to existing contact, "grow"
  simpleGrow3(collMatrix, seqNoGap, 0.5, scorer);
  // remove bad contacts:
  removeNonMatching(collMatrix, seqNoGap, scorer);
  // expand again:
  expandMatrix(seq, collMatrix);
  // convert matrix to stems:
  Vec<Stem> stems = generateStemsFromMatrix(collMatrix, 1, 0.5, seq);
  // cout << "Here is the individual sec structure of " << endl << seq
  // << endl << stems << endl;
  // write output format:
  writeCtSingle(cout, stems, seq);
}

/** uses full copy of matrix representing consensus secondary structure */
void
writeIndividualNoGrow(ostream& os, Vec<Vec<double> > consMatrix,
		      const string& seqOrig,
		      const CompensationScorer& scorer,
		      double thresh,
		      int outputFormat,
		      int collapseMode)
{
  // grow matrix: if complementary base pair next to exisiting stem 
  Vec<unsigned int> noGapSubset;
  // remove bad contacts:
  removeNonMatching(consMatrix, seqOrig, scorer);
  if (collapseMode) {
    noGapSubset = getNoGapSubset(seqOrig);
    consMatrix = getMatrixSubset(consMatrix, noGapSubset);
  }
  else {
    noGapSubset = generateStair(seqOrig.size());
  }
  string seqNoGap = getSubset(seqOrig, noGapSubset); // gap free string
  // convert matrix to stems:
  Vec<Stem> stems = generateStemsFromMatrix(consMatrix,
					    1, thresh, seqNoGap);
  // cout << "Here is the individual sec structure of " << endl << seq
  // << endl << stems << endl;
  // write output format:
  string ssstruct;
  switch (outputFormat) {
  case 2: // bracket notation   
    ssstruct = stemsToBracketFasta(stems, seqNoGap);
    os << seqNoGap << endl;
    os << ssstruct << endl;
    break;
  case 4: // ct file
    writeCtSingle(cout, stems, seqNoGap);
    break;
  default:
    ERROR("Output format not suported!");
  }      
}


void
writeIndividuals(ostream& os, 
		 const Vec<Stem>& stems, 
		 const SequenceAlignment& ali,
		 const CompensationScorer& scorer,
		 int outputFormat)
{
  //  generate matrix:
  // cout << "Here is the consensus structure: " << stems << endl;
  Vec<Vec<double> > matrix = generateMatrixFromStems(stems, ali.getLength());
  // cout << "Here is the consensus matrix: " << matrix << endl;
  for (SequenceAlignment::size_type i = 0; i < ali.size(); ++i) {
    writeIndividual(os, matrix, ali.getSequence(i), scorer, outputFormat);
  }
}

void
writeIndividualsNoGrow(ostream& os, 
		       const Vec<Stem>& stems, 
		       const SequenceAlignment& ali,
		       const CompensationScorer& scorer,
		       int outputFormat,
		       int collapseMode)
{
  //  generate matrix:
  // cout << "Here is the consensus structure: " << stems << endl;
  Vec<Vec<double> > matrix = generateMatrixFromStems(stems, ali.getLength());
  // cout << "Here is the consensus matrix: " << matrix << endl;
  double thresh = 0.5;
  for (unsigned int i = 0; i < ali.size(); ++i) {
    writeIndividualNoGrow(os, matrix, ali.getSequence(i), scorer, thresh, outputFormat, collapseMode);
  }
}

// adds potential spaces
string
conformName(const string& s, int len) 
{
  if (static_cast<int>(s.size()) == len) {
    return s;
  }
  else if (static_cast<int>(s.size()) > len) {
    return s.substr(0, len);
  }
  int dc = len - static_cast<int>(s.size());
  string result = s;
  for (int i = 0; i < dc; ++i) {
    result = result + " ";
  }
  return result;
}

/** writes single base pair (entry of "pair" structure). See McSym UserManual.pdb */
void
writeMcSymBasePair(ostream& os, Stem::index_type start, Stem::index_type stop, 
		   const sequence_type& sequence)
{
  os << /* sequence[start] << */ (start+1) << " " << /* sequence[stop] << */ (stop+1) << " { wct } 100%" << endl;
}

/** writes McSym input file from secondary structure */
void
writeMcSym(ostream& os,  const Vec<Stem>& stems, const sequence_type& sequence)
{
  ERROR_IF(sequence.size() == 0,
	   "Sequence has to be specified for writing McSym format!");
  RnaSecondaryStructure structure(stems, sequence);
  os << "sequence (r " /* << sequence[0] */ << "1 " << sequence << ")" << endl;
  os << "residue (" << endl;
  for (size_type i = 0; i < sequence.size(); ++i) {
    os << " " << (i+1) << " { ";
    if (structure.isPaired(i)) {
      os << "helix }  100%" << endl;
    }
    else {
      os << " }   5" << endl;
    }
  }
  os << ")" << endl << endl;

  os << "pair (" << endl;
  for (size_type i = 0; i < stems.size(); ++i) {
    for (Stem::index_type j = 0; j < stems[i].getLength(); ++j) {
      writeMcSymBasePair(os, stems[i].getStart() + j, stems[i].getStop()-j, sequence);
    }
  } 
  os << ")" << endl << endl;

  os << "connect (" << endl;
  for (RnaSecondaryStructure::index_type i = 1; (i+1) < structure.getLength(); ++i) {
    os << " " << (i+1) << " " << (i+2) << " { ";
    if (structure.isPaired(i) && structure.isPaired(i+1)) {
      os << " helix }  100%" << endl;
    }
    else {
      os << " connect }  5"  << endl;
    }
  }
  

  os << ")" << endl << endl;

  os << "exploreLV (" << endl;
  os << " unnamed" << endl
     << " rmsd (1 align base_only no_hydrogen)" << endl
     << " file_pdb (\"unnamed-%.pdb\")" << endl
     << " time_limit (1 days)" << endl
     << ")" << endl;
}

void
writeStockholm(ostream& os, const Vec<Stem>& stems, const SequenceAlignment& ali)
{
  int col1Len = 19; // this many characters in first column
  os << "STOCKHOLM 1.0" << endl;
  for (unsigned int i = 0; i < ali.size(); ++i) {
    string name = ali.getName(i);
    vector<string> words = getTokens(name);
    ERROR_IF(words.size() == 0, 
	     "writeStockholm: No sequence name specified!");
    name = words[0]; // only get first word
    name = conformName(name, col1Len);
    os << name << " " << ali.getSequence(i) << endl;
  }
  string item = "#= GC SS_cons";
  item = conformName(item, col1Len);
  string ssstruct = stemsToBracketFasta(stems, ali.getLength());
  os << item << " " << ssstruct << endl;
  os << "//" << endl; // end of alignment
}

/** writes number of stems of certain lengths: */
void
writeStatistics1(ostream& os, const Vec<Stem>& stems, unsigned int seqLen) 
{
  Vec<int> histogram(20, 0);
  for (unsigned int i = 0; i < stems.size(); ++i) {
    int len = stems[i].getLength();
    // loop over smaller stem lengths that fit into this stem:
    for (int j = 1; j <= len; ++j) {
      if (j >= static_cast<int>(histogram.size())) {
	break; // undefined
      }
      int nl = len - j + 1;
      ASSERT(nl >= 0);
      ++histogram[j];
    }
  }
  for (unsigned int i = 1; i < histogram.size(); ++i) {
    os << i << "\t" << histogram[i] << "\t" << (100.0*static_cast<double>(histogram[i])/seqLen) << "\t" << seqLen << endl;
  }
}

void
writePseudoviewerStem(ostream& os,
		      const Stem& stem) {
  os << (stem.getStart()+1) << "-" << (stem.getStart() + stem.getLength() - 1 + 1) 
     << ";"
     << (stem.getStop()-stem.getLength()+1 + 1) << "-" << (stem.getStop() + 1);
}

/** writes output compatible with Pseudoviewer web application */
void
writePseudoviewerStems(ostream& os, const string& name,
		       const Vec<Stem>& stems,
		       const string& sequence) {
  os << "# " << name << endl;
  os << "1" << endl;
  os << sequence << endl << endl;
  os << "Positions paired." << endl;
  for (Vec<Stem>::size_type i = 0; i < stems.size(); ++i) {
    writePseudoviewerStem(os, stems[i]);
    os << endl;
  }
}

void
writeStemWithSeparation(ostream& os,
			  const Stem& stem)
{
  os << (stem.getStart() + 1) << " " << (stem.getStop() + 1) << " "
     << stem.getLength() << " " << (stem.getStop() - stem.getStart()) << endl;
}

/** writes list of stems with sequence separation between start and stop nucleotide */
void
writeStemsWithSeparations(ostream& os,
			  const Vec<Stem>& stems)
{
  for (Vec<Stem>::size_type i = 0; i < stems.size(); ++i) {
    writeStemWithSeparation(os, stems[i]);
  }
}

string::size_type
computeAlphabetId(char c1, 
		  char c2,
		  const string& alphabet) 
{
  string::size_type n2 = alphabet.size() * alphabet.size();
  string::size_type p1 = alphabet.find(c1);
  if (p1 >= alphabet.size()) {
    return n2;
  }
  string::size_type p2 = alphabet.find(c2);
  if (p1 >= alphabet.size()) {
    return n2;
  }
  string::size_type id = p1 * alphabet.size() + p2;
  return id;
}

string::size_type
computeAlphabetId(char c1, 
		  const string& alphabet) 
{
  string::size_type n = alphabet.size();
  string::size_type p1 = alphabet.find(c1);
  if (p1 >= alphabet.size()) {
    return n;
  }
  return p1;
}

void
writeDinucleotideStatistics(ostream& os,
			    char c1,
			    char c2,
			    unsigned int structuredCounter,
			    unsigned int unstructuredCounter,
			    double fracStructured)
{
  double ltr = static_cast<double>(structuredCounter) / unstructuredCounter;
  if (fracStructured <= 0.0) {
    fracStructured = 0.1;
  }
  double ltr2 = ltr / fracStructured;
  unsigned int sum = structuredCounter + unstructuredCounter;
  os << c1 << c2 << " " 
     << ltr << " " << log(ltr) << " " 
     <<  ltr2 << " " << log(ltr2) << " "
     << structuredCounter << " " << unstructuredCounter << " " << sum;
}

/** compares dinucleotides found in and out of structure */
void
writeDinucleotideStatistics(ostream& os,
			    const Vec<Stem>& stems,
			    const SequenceAlignment& ali,
			    const string& alphabet)
{
  string::size_type n2 = alphabet.size() * alphabet.size();
  Vec<Vec<double> > matrix = generateMatrixFromStems(stems, ali.getLength());
  Vec<double> rowSums(matrix.size(), 0);
  unsigned int structCount = 0;
  for (Vec<Vec<double> >::size_type i = 0; i < matrix.size(); ++i) {
    rowSums[i] = elementSum(matrix[i]);
    if (rowSums[i] > 0.5) {
      ++structCount;
    }
  }
  double fracStructured = static_cast<double>(structCount) 
    / (matrix.size() - structCount);
  double thresh = 0.5;
  Vec<unsigned int> structuredCounters(n2, 1U); // count using pseudo count!
  Vec<unsigned int> unstructuredCounters(n2, 1U);
  string seq = ali.getSequence(0); // only use first sequences
  for (string::size_type i = 0; (i+1) < seq.size(); ++i) {
    char c1 = seq[i];
    char c2 = seq[i+1];
    string::size_type id = computeAlphabetId(c1, c2, alphabet);
    if (id >= n2) {
      continue; // gap or unknown residue encountered
    }
    // check if both structured:
    if ((rowSums[i] > thresh) && (rowSums[i+1] > thresh)) {
      structuredCounters[id]++;
    }
    else {
      unstructuredCounters[id]++;
    }

  }
  for (string::size_type i = 0; i < alphabet.size(); ++i) {
    char c1 = alphabet[i];
    for (string::size_type j = 0; j < alphabet.size(); ++j) {
      char c2 = alphabet[j];
      string::size_type id = computeAlphabetId(c1, c2, alphabet);
      writeDinucleotideStatistics(os, c1, c2, structuredCounters[id], unstructuredCounters[id],
				  fracStructured);
      os << " " << fracStructured << " " << seq.size() << endl;
    }
  }
}

void
writeNucleotideStatistics(ostream& os,
			    char c1,
			    unsigned int structuredCounter,
			    unsigned int unstructuredCounter,
			    double fracStructured)
{
  double ltr = static_cast<double>(structuredCounter) / unstructuredCounter;
  if (fracStructured <= 0.0) {
    fracStructured = 0.1;
  }
  double ltr2 = ltr / fracStructured;
  unsigned int sum = structuredCounter + unstructuredCounter;
  os << c1 << " " 
     << ltr << " " << log(ltr) << " " 
     <<  ltr2 << " " << log(ltr2) << " "
     << structuredCounter << " " << unstructuredCounter << " " << sum;
}

/** compares dinucleotides found in and out of structure */
void
writeNucleotideStatistics(ostream& os,
			    const Vec<Stem>& stems,
			    const SequenceAlignment& ali,
			    const string& alphabet)
{
  string::size_type nn = alphabet.size();
  Vec<Vec<double> > matrix = generateMatrixFromStems(stems, ali.getLength());
  Vec<double> rowSums(matrix.size(), 0);
  unsigned int structCount = 0;
  for (Vec<Vec<double> >::size_type i = 0; i < matrix.size(); ++i) {
    rowSums[i] = elementSum(matrix[i]);
    if (rowSums[i] > 0.5) {
      ++structCount;
    }
  }
  double fracStructured = static_cast<double>(structCount) 
    / (matrix.size() - structCount);
  double thresh = 0.5;
  Vec<unsigned int> structuredCounters(nn, 1U); // count using pseudo count!
  Vec<unsigned int> unstructuredCounters(nn, 1U);
  string seq = ali.getSequence(0); // only use first sequences
  for (string::size_type i = 0; (i+1) < seq.size(); ++i) {
    char c1 = seq[i];
    string::size_type id = computeAlphabetId(c1, alphabet);
    if (id >= nn) {
      continue; // gap or unknown residue encountered
    }
    // check if both structured:
    if ((rowSums[i] > thresh) && (rowSums[i+1] > thresh)) {
      structuredCounters[id]++;
    }
    else {
      unstructuredCounters[id]++;
    }
    
  }
  for (string::size_type i = 0; i < alphabet.size(); ++i) {
    char c = alphabet[i];
      string::size_type id = computeAlphabetId(c, alphabet);
      writeNucleotideStatistics(os, c, structuredCounters[id], unstructuredCounters[id],
				fracStructured);
      os << " " << fracStructured << " " << seq.size() << endl;
  }
}

void
writeSequenceWithPairednessConstraints(ostream& os,
				       const Vec<Stem>& stems,
				       const SequenceAlignment& ali) {
  ASSERT(ali.size() > 0);
  os << ali.getSequence(0) << endl;
  for (SequenceAlignment::sequence_size_type i = 0; i < ali.getLength(); ++i) {
    bool found = false;
    for (Vec<Stem>::size_type j = 0; j < stems.size(); ++j) {
      if (stems[j].usesBase(static_cast<Stem::index_type>(i))) {
	found = true;
	break;
      }
    }
    if (found) {
      os << "|";
    } else {
      os << "x";
    }
  }
  os << endl;
}

/** returns those stems from origStems that have an equivalent in filterStems */
Vec<Stem> filterStems(const Vec<Stem>& origStems, 
		      const Vec<Stem>& filterStems) {
  Vec<Stem> result;
  for (Vec<Stem>::size_type i = 0; i < origStems.size(); ++i) {
    for (Vec<Stem>::size_type j = 0; j < filterStems.size(); ++j) {
      if (origStems[i] == filterStems[j]) {
	result.push_back(origStems[i]);
	break;
      }
    }
  }
  return result;
}

/** returns those stems from origStems that have no equivalent in filterStems */
Vec<Stem> filterMissingStems(const Vec<Stem>& origStems, 
			     const Vec<Stem>& filterStems) {
  Vec<Stem> result;
  for (Vec<Stem>::size_type i = 0; i < origStems.size(); ++i) {
    bool found = false;
    for (Vec<Stem>::size_type j = 0; j < filterStems.size(); ++j) {
      if (origStems[i] == filterStems[j]) {
	found = true;
	break;
      }
    }
    if (!found) {
      result.push_back(origStems[i]);
    }
  }
  return result;
}

void
writeRNAFoldProbingConstraints(ostream& os, const string& sequence, const Vec<Stem>& stems) {
  Vec<Vec<double> > matrix = Vec<Vec<double> >(sequence.size(), Vec<double>(sequence.size(), 0.0));
  addStemsToMatrix(matrix, stems, 1.0);
  os << sequence << endl;
  for (Vec<Vec<double> >::size_type i = 0; i < matrix.size(); ++i) {
    if (elementSum(matrix[i]) > 0.5) {
      os << "|"; // base is paired to an unspecified partner
    } else {
      os << "x"; // the base is unpaired
    }
  } 
  os << endl;
}

/********************************* MAIN **********************************/
int
main(int argc, char ** argv)
{
  bool aliChangedFlag = false;
  bool helpMode;
  //  bool guAllowed= true;
  int argcFile = 0;
  char ** argvFile = 0;
  unsigned int length = 0;
  unsigned int minStemLength = 1;
  unsigned int verboseLevel = 0;
  int collapseId = -1;
  int complementaryFilter = 0;
  int expandId = -1;
  int gapFracRescaleMode = 0;
  int indexOffset = 0;
  int inputFormat = 1;
  int leaveMatrixMode = 0;
  int outputFormat = 1;
  int stemStatLen = 3;
  int watsonCrickId = -1;
  int winnerDiagBorder = 1;
  double gapFracAlpha = 1.0; // linear rescaling of result matrix with fraction of gaps
  double consensusFrac = 0.0;
  double stemEnergy = 0.0;
  double thresh = 0.5;
  double winnerCutoff = 0.01; // used for speedup of winner takes all 
  string aliFileName;
  string alphabet = "ACGU";
  string bracket;
  string commandFileName;
  string inputFileName;
  string logFileName; //  = "mainprogramtemplate.log";
  string matrixFileName;
  string probeFileName;
  string rootDir = ".";
  string sequence;
  string sequenceName;
  string startsAlignment; // alignment file name that is specified for the sole purpose of specifying sequence lengths
  string subsetString;
  SimpleSequenceAlignment ali;
  CompensationScorer scorer;
  Vec<int> starts; // contains one entry for first residue of only sequence; one based indices
  Vec<Stem> stems;
  Vec<Vec<double> > matrix;
  Vec<Vec<double> > extMatrix;
  Vec<unsigned int> subset;
  Vec<double> designCosts;
  Vec<double> probeData; // probability of a base being base paired!
  getArg("-help", helpMode, argc, argv);

  if ((argc < 2) || helpMode)  {
    helpOutput(cout);
    exit(0);
  }

  // cout << "mark 0!" << endl;

  getArg("-root", rootDir, argc, argv, rootDir);
  addSlash(rootDir);

  getArg("-commands", commandFileName, argc, argv, commandFileName);
  addPathIfRelative(commandFileName, rootDir);

  if (commandFileName.size() > 0) {
    ifstream commandFile(commandFileName.c_str());
    if (!commandFile) {
      if (isPresent("-commands", argc, argv)) {
	ERROR_IF(!commandFile, "Error opening command file.");
      }
      else {
	cerr << "Warning: Could not find command file: " + commandFileName 
	     << endl;
      }
    }
    else {
      argvFile = streamToCommands(commandFile, argcFile, 
				  string("mainprogramtemplate"));
    }
    commandFile.close();
  }

  getArg("a", aliFileName, argc, argv, aliFileName);
  getArg("-alphabet", alphabet, argcFile, argvFile, alphabet);
  getArg("-collapse", collapseId, argc, argv, collapseId);
  --collapseId; // convert to internal counting
  getArg("-consensus", consensusFrac, argcFile, argvFile, consensusFrac);
  getArg("-consensus", consensusFrac, argc, argv, consensusFrac);
  getArg("-complementary", complementaryFilter, argc, argv, 
	 complementaryFilter);
  getArg("-expand", expandId, argc, argv, expandId);
  --expandId; // convert to internal counting
  getArg("-gap-frac-alpha", gapFracAlpha, argcFile, argvFile, gapFracAlpha);
  getArg("-gap-frac-alpha", gapFracAlpha, argc, argv, gapFracAlpha);
  if (isPresent("-gap-frac-alpha", argcFile, argvFile)
      || isPresent("-gap-frac-alpha", argc, argv)) {
    gapFracRescaleMode = 1;
  }
  getArg("i", inputFileName, argc, argv, inputFileName);
  getArg("-if", inputFormat, argcFile, argvFile, inputFormat);
  getArg("-if", inputFormat, argc, argv, inputFormat);
  getArg("l", thresh, argcFile, argvFile, thresh);
  getArg("l", thresh, argc, argv, thresh);
  getArg("-length", length, argcFile, argvFile, length);
  getArg("-length", length, argc, argv, length);
  getArg("-log", logFileName, argc, argv, logFileName);
  getArg("-log", logFileName, argcFile, argvFile, logFileName);
  addPathIfRelative(logFileName, rootDir);
  getArg("-matrix", matrixFileName, argcFile, argvFile, matrixFileName);
  getArg("-matrix", matrixFileName, argc, argv, matrixFileName);
  getArg("-of", outputFormat, argcFile, argvFile, outputFormat);
  getArg("-of", outputFormat, argc, argv, outputFormat);
  getArg("-offset", indexOffset, argc, argv, indexOffset);
  getArg("-probe", probeFileName, argc, argv, probeFileName);
  getArg("-starts", starts, argc, argv);
  if (starts.size() == 0) {
    starts.push_back(0);
  } else {
    convert2InternalCounting(starts);
  }
  getArg("-starts-alignment", startsAlignment, argc, argv);
  if (startsAlignment.size() > 0) {
    ifstream startAliFile(startsAlignment.c_str());
    ERROR_IF(!startAliFile, "Error opening alignment file: " + starts[0]);
    SimpleSequenceAlignment startAli;
    startAli.readFasta(startAliFile);
    starts = SequenceAlignmentTools::toStarts(startAli);
    startAliFile.close();
  } 
  getArg("-stem-energy", stemEnergy, argcFile, argvFile, stemEnergy);
  getArg("-stem-energy", stemEnergy, argc, argv, stemEnergy);
  getArg("-stem-length", stemStatLen, argcFile, argvFile, stemStatLen);
  getArg("-stem-length", stemStatLen, argc, argv, stemStatLen);
  getArg("-stem-min", minStemLength, argcFile, argvFile, minStemLength);
  getArg("-stem-min", minStemLength, argc, argv, minStemLength);
  getArg("-subset", subsetString, argcFile, argvFile, subsetString);
  getArg("-subset", subsetString, argc, argv, subsetString);
  subset = parseStringToVector(subsetString);
  convert2InternalCounting(subset);
  getArg("-verbose", verboseLevel, argcFile, argvFile, verboseLevel);
  getArg("-verbose", verboseLevel, argc, argv, verboseLevel);
  getArg("-wc", watsonCrickId, argcFile, argvFile, watsonCrickId);
  getArg("-wc", watsonCrickId, argc, argv, watsonCrickId);
  --watsonCrickId;

  if (logFileName.size() > 0) {
    ofstream logFile(logFileName.c_str(), ios::app);
    parameterOutput(logFile, argc, argv);
    if (argcFile > 1) {
      logFile << "Parameters from command file: ";
      parameterOutput(logFile, argcFile, argvFile);
    }
    logFile.close();
  }

  /***************** MAIN PROGRAM *****************************/

  if (aliFileName.size() > 0) {
    ifstream aliFile(aliFileName.c_str());
    ERROR_IF(!aliFile, "Error opening alignment file!");
    ali.readFasta(aliFile);
    aliFile.close();
    if (verboseLevel > 0) {
      cout << "Alignment length and size: " << ali.getLength() 
	   << " " << ali.size() << endl;
    }
  }

  ifstream inputFile(inputFileName.c_str());
  ERROR_IF(!inputFile, "Error opening input file!");
  unsigned int pkCounter = 0;
  switch(inputFormat) {
  case 1:
    stems = readStems(inputFile);
    ERROR_IF((!isPresent("-length", argc, argv)) && (!isPresent("a", argc, argv)),
	     "Neither length nor alignment specified!");
    break;
  case 2:
    stems = stemsFromBracketFasta(inputFile, pkCounter, length);
    break;
  case 3:
    if (outputFormat == 3) {
      leaveMatrixMode = 1;
    }
    matrix = readPlainMatrix(inputFile);
    winnerTakesAll(matrix, winnerDiagBorder, winnerCutoff);
    // careful: does not work together with collapsing of alignment
    if (gapFracRescaleMode) {
      if (verboseLevel > 0) {
	cout << "Calling gapFracRescale!" << endl;
      }
      gapFracRescale(matrix, ali, gapFracAlpha);
    }
    length = matrix.size();
    stems = generateStemsFromMatrix(matrix, minStemLength, thresh, sequence);
    break;
  case 4:
    readCtSingle(inputFile, stems, sequence, minStemLength);
    break;
  case 5:
    matrix = readPlainMatrix(inputFile);
    break;
  case 6: // read BPL format 
    matrix = readBPL(inputFile, sequence);
    stems = generateStemsFromMatrix(matrix, minStemLength, 0.5, sequence);
    break;
  case 8: // RNAfold output 
    {
      RnaSecondaryStructure rnaSecStruct = StemTools::readRNAfold2(inputFile);
      sequence = rnaSecStruct.getSequence();
      stems = convertSecondaryStructureToStems(rnaSecStruct);
      matrix = convertSecondaryStructureToMatrix(rnaSecStruct);
    }
    break;
  case 9: // WMatch output
    ERROR_IF(ali.size() ==0, "Alignment has to be specified!");
    matrix = StemTools::readWMatch(inputFile, ali.getLength());
    if (verboseLevel > 2) {
      cout << "wmatch file read with size: " << matrix.size() << endl;
    }
    ERROR_IF(matrix.size() != ali.getLength(),
	     "Matrix size does not match alignment size!");
    stems = generateStemsFromMatrix(matrix, minStemLength, 0.5, sequence);
    break;
  case 17:  // read RNAeval format 
    {
      RnaSecondaryStructure rnaSecStruct = StemTools::readRNAeval(inputFile);
      sequence = rnaSecStruct.getSequence();
      stems = convertSecondaryStructureToStems(rnaSecStruct);
      matrix = convertSecondaryStructureToMatrix(rnaSecStruct);

    }
    break;
  case 30: {
    RnaSecondaryStructure rnaSecStruct = RnaSecondaryStructureTools::readFastaSecondaryCosts(inputFile);
    sequence = rnaSecStruct.getSequence();
    stems = convertSecondaryStructureToStems(rnaSecStruct);
    starts = rnaSecStruct.getStarts();
    matrix = convertSecondaryStructureToMatrix(rnaSecStruct);
    designCosts = rnaSecStruct.getDesignCosts();
  }
    break;
  default:
    ERROR("Unknown input format id!");
  }

  inputFile.close();

  if (matrixFileName.size() > 0) {
    ifstream matrixFile(matrixFileName.c_str());
    ERROR_IF(!matrixFile, "Error opening matrix file!");
    extMatrix = readPlainMatrix(matrixFile);
    matrixFile.close();
    // cout << "Matrix with dimension " << matrix.size() << " successfully read!" << endl;
  }

  if (probeFileName.size() > 0) {
    ifstream probeFile(probeFileName.c_str());
    ERROR_IF(!probeFile, "Error opening probe data file: " + probeFileName);
    probeFile >> probeData;
    if (verboseLevel > 1) {
      cout << " Probe data: " << probeData << endl;
    }
    probeFile.close();
  }

  // if no alignment specified but sequence was read as part of secondary structure:
  if ((ali.size() == 0) && (sequence.size() > 0)) {
    ali.addSequence(sequence, inputFileName);
  }

  if (ali.size() > 0) {
    length = ali.getLength();
    ali.upperCaseSequences(); // transform to upper case
  }

  // remove all gaps of a reference sequence:
  if (collapseId >= 0) {
    if (verboseLevel > 0) {
      cout << "Collapsing gaps with respect to sequence "
	   << collapseId + 1 << endl;
    }
    if (leaveMatrixMode) {
      matrix = collapseAlignment(ali.getSequence(collapseId), matrix);
      sequence = ali.getSequence(collapseId);
      sequenceName = ali.getName(collapseId);
    }
    else {
      collapseAlignment(ali, collapseId, stems);
      sequence = ali.getSequence(collapseId);
      sequenceName = ali.getName(collapseId);
    }
    aliChangedFlag = true;
    if (verboseLevel > 0) {
      cout << "New length of alignment after collapsing: " << ali.getLength() << endl;
    }
  }
  else if (expandId >= 0) {
    if (verboseLevel > 0) {
      cout << "Expanding gaps in secondary structure with respect to sequence "
	   << expandId + 1 << endl;
    }
    if (leaveMatrixMode) {
      unsigned int mSize = matrix.size();
      expandMatrix(ali.getSequence(expandId), matrix);
      sequence = ali.getSequence(expandId);
      if (verboseLevel > 0) {
	cout << "Expanding matrix! " << mSize << " " << matrix.size() << endl;
      }
    }
    else {
      string currentSequence = expandSecondary(ali, static_cast<unsigned int>(expandId), stems);
      if (verboseLevel > 0) {
	cout << "New length of secondary structure after collapsing: " << currentSequence.size() << endl;
      }
      sequence = currentSequence; // TODO : verify if correct!
      sequenceName = ali.getName(expandId);
    }
    aliChangedFlag = false;
  }

  // remove all pairs that are not AU, CG, (or GU) (also T instead of U)
  if (watsonCrickId >= 0) {
    if (verboseLevel > 0) {
      cout << "removing all non-complementary base-pairs of secondary structure according to alignment and Watson-Crick sequence id" << endl;
    }
    ERROR_IF(ali.size() == 0, 
	     "Alignment has to be specified in order to remove non-complementary base pairs");
    stems = removeNonComplementary(ali.getSequence(watsonCrickId), stems);
  }

  if (consensusFrac > 0.0) {
    stems = removeNonConsensus(consensusFrac, ali, stems);
  }

  int conflicts = 0;
  if (ali.size() > 0) {
    conflicts = countConflictingBasePairs(stems, ali.getLength(), 0.5);
    if (verboseLevel > 1) {
      cout << "Number of conflicting bases: " << conflicts;
      if (ali.getLength() > 0) {
	cout << " " << 100.0 * conflicts / static_cast<double>(ali.getLength()) << " %" << endl;
      }
      else {
	cout << endl;
      }
    }
  }

  if ((sequence.size() != ali.getLength()) && (ali.size() > 0)) {
    sequence = ali.getSequence(0);
    sequenceName = ali.getName(0);
  }
  if (sequence.size() > 0) {
    length = sequence.size();
  }
  
  // if only subset of sequence is wanted: convert to matrix then back to stems:
  if (subset.size() > 0) {
    if (verboseLevel > 1) {
      cerr << "Using subset indices: " << externalCounting(subset) << endl;
    }
    matrix = Vec<Vec<double> >(length,
			       Vec<double>(length, 0.0));
    addStemsToMatrix(matrix, stems, 1.0);
    matrix = getMatrixSubset(matrix, subset);
    if (sequence.size() > 0) {
      sequence = getSubset(sequence, subset);
    }
    length = matrix.size();
    stems = generateStemsFromMatrix(matrix, minStemLength, thresh, sequence);
  }
  if (stemEnergy != 0.0) {
    if (verboseLevel > 0) {
      cout << "# changing default stem energy to value " << stemEnergy << endl;
    }
    for (Vec<Stem>::size_type i = 0; i < stems.size(); ++i) {
      stems[i].setEnergy(stemEnergy);
    }
  }
  switch (outputFormat) {
  case RnaFormats::REGION_FORMAT:
    writeStems(cout, stems, indexOffset);
    break;
  case RnaFormats::BRACKET_FORMAT:
    ERROR_IF(length == 0, "No length of sequence specified!");
    cout << ">" << fileNameBase(inputFileName) << endl;
    cout << stemsToBracketFasta(stems, length) << endl;
    break;
  case RnaFormats::MATRIX_FORMAT:
    if (!leaveMatrixMode) {
      matrix = Vec<Vec<double> >(length,
				 Vec<double>(length, 0.0));
      addStemsToMatrix(matrix, stems, 1.0);
    }
    writeMatrix(cout, matrix);
    break;
  case RnaFormats::CT_FORMAT:  // CT file:
    ERROR_IF(ali.size() == 0, 
	     "Error: no alignment or sequence specified. Use option -a");
    writeCtSingle(cout, stems, sequence);
    break;
  case 5: // write out topology:
    cout << convertStemsToTopology(stems, length) << endl;
    break;

  case 6: // write BPL format 
    ERROR_IF(length == 0, "No length of sequence specified!");
    writeBPLStems(cout, fileNameBase(inputFileName),
		  sequence, stems);
    break;
  case 7: // constraints for RNAfold
    ERROR_IF(length == 0, "No length of sequence specified!");
    // cout << ">" << fileNameBase(inputFileName) << endl;
    cout << stemsToBracketFasta(stems, length) << endl;
    break;
  case 8: // write stem statistics:
    writeStemStatistics(cout, stems, extMatrix, stemStatLen);
    break;
  case 9: // write individualized sequences using "grow step":
    writeIndividuals(cout, stems, ali, scorer, outputFormat);
    break;
  case 10: // write individualized sequences NOT using "grow step", collapse individual sequences
    writeIndividualsNoGrow(cout, stems, ali, scorer, 2, true);
    break;
  case 11:
    writeStockholm(cout, stems, ali);
    break;
  case 12: // simply write length of secondary structure (or first sequence including gaps in alignment)
    cout << length << endl;
    break;
  case 13: // write individualized sequences NOT using "grow step". Do not collapse individual sequences
    writeIndividualsNoGrow(cout, stems, ali, scorer, 2, false);
    break;
  case 14: // write individualized sequences NOT using "grow step". Do not collapse individual sequences. write multi ct format
    writeIndividualsNoGrow(cout, stems, ali, scorer, 4, false);
    break;
  case 15:
    writeStatistics1(cout, stems, length);
    break;
  case 16: 
    writeMcSym(cout, stems, sequence);
    break;
  case 17: // RNAeval format:
    ERROR_IF(sequence.size() != length,
	     "Sequence length has to match structure length!");
    if (sequenceName.size() > 0) {
      cout << ">" << sequenceName << endl;
    }
    cout << sequence << endl;
    cout << stemsToBracketFasta(stems, length) << endl;
    break;
  case 18: // output for pseudoviewer:
    writePseudoviewerStems(cout, fileNameBase(inputFileName),
			   stems, sequence);
    break;
  case 19:
    writeStemsWithSeparations(cout, stems);
    break;
  case 20:
    writeDinucleotideStatistics(cout, stems, ali, alphabet);
    break;
  case 21:
    writeNucleotideStatistics(cout, stems, ali, alphabet);
    break;
  case RnaFormats::FASTA_FORMAT: // FASTA format:
    ERROR_IF(sequence.size() != length,
	     "Sequence length has to match structure length!");
    if (starts.size() == 1) {
      if (sequenceName.size() > 0) {
	cout << ">" << sequenceName << endl;
      }
      cout << sequence << endl;
    } else {
      for (size_type i = 0; (i+1) < starts.size(); ++i) {
	cout << ">s" << (i+1) << endl;
	cout << sequence.substr(starts[i], (starts[i+1]-starts[i])) << endl;
      }
      cout << ">s" << starts.size() << endl;
      cout << sequence.substr(starts[starts.size()-1], (sequence.size()-starts[starts.size()-1])) << endl;
    }
    break;
  case 23: // FASTA format with dummy name:
    ERROR_IF(sequence.size() != length,
	     "Sequence length has to match structure length!");
    cout << ">sequence" << endl;
    cout << sequence << endl;
    break;
  case 24:
    writeSequenceWithPairednessConstraints(cout, stems, ali);
    break;
//   case 25: // write ALL stems compatible with structure, even sub-stems using probing scores if available:
//     {
//       matrix = Vec<Vec<double> >(length,
// 				 Vec<double>(length, 0.0));
//       addStemsToMatrix(matrix, stems, 1.0);
//       ASSERT(probeData.size() > 0);
//       ProbeStemEnergyModel probeModel(probeData);
//       Vec<Stem> allStems = MatchFoldTools::generateAllStems(sequence, matrix, starts, minStemLength, 0.5,
// 							    probeModel);
//       writeStems(cout, allStems, indexOffset);
//     }
//     break;
//   case 26: // write those stems compatible with structure (no sub-stems) using probing scores:
//     {
//       matrix = Vec<Vec<double> >(length,
// 				 Vec<double>(length, 0.0));
//       addStemsToMatrix(matrix, stems, 1.0);
//       ASSERT(probeData.size() > 0);
//       ProbeStemEnergyModel probeModel(probeData);
//       Vec<Stem> allStems = MatchFoldTools::generateAllStems(sequence, matrix, starts, minStemLength, 0.5,
// 							    probeModel);
//       writeStems(cout, filterStems(allStems, stems), indexOffset); // only use allStems members that are part of stems
//     }
//     break;
//   case 27: // write ALL stems that are NOT compatible with structure, even sub-stems using probing scores if available:
//     {
//       ASSERT(probeData.size() > 0);
//       matrix = Vec<Vec<double> >(length,
// 				 Vec<double>(length, 0.0));
//       addStemsToMatrix(matrix, stems, 1.0);
//       ASSERT(probeData.size() > 0);
//       ProbeStemEnergyModel probeModel(probeData);
//       Vec<Stem> compatibleStems = MatchFoldTools::generateAllStems(sequence, matrix, starts, minStemLength, 0.5,
// 								   probeModel);
//       Vec<Stem> allStems = MatchFoldTools::generateAllStems(sequence, starts, minStemLength, guAllowed,
// 							    probeModel);
//       Vec<Stem> remainingStems = filterMissingStems(allStems, compatibleStems); // only write stems that were not part of structure:
//       writeStems(cout, remainingStems, indexOffset);
//     }
//     break;
//   case 28:
//     writeRNAFoldProbingConstraints(cout,sequence, stems);
//     break;
//   case 29: {
//     RnaSecondaryStructure rnaStructure(stems, sequence, starts);
//     MatchFoldTools::writeFasta(cout, rnaStructure);
//   }
//     break;
//   case 30: { // output of design costs;
//     cout << designCosts << endl;
//   }
//     break;
//   case 31: // write RNAalifold pairs to be added RNAalifold postscript output
//     writeRNAalifoldPairs(cout, fileNameBase(inputFileName),
// 			 sequence, stems);
//     break;
//   case RnaFormats::MCSOPT_FORMAT: {
//     RnaSecondaryStructure rnaStructure(stems, sequence, starts);
//     MatchFoldTools::writeMcsopt(cout, rnaStructure);
//   }
//     break;
  default:
    ERROR("Unknown output format id!");
    
  }

  
  return 0;
}
