// --*- C++ -*------x---------------------------------------------------------
#ifndef __CORRELATION_FINDER__
#define __CORRELATION_FINDER__

#include <Correlation.h>
#include <debug.h>

#include <Stem.h>
#include <MAFAlignment.h>
#include <SingleLinkage2DProgressiveFilter.h>

using namespace std;

#ifdef COVARNA_CONCURRENT_VECTOR
#include <tbb/blocked_range.h>
using namespace tbb;
#else
#include <ser_blocked_range.h>
#endif

class CorrelationFinder {

 public:
  typedef Correlation::length_type length_type;
  typedef string::size_type size_type;
  typedef string::size_type set_size_type;
  typedef Vec<Correlation> result_container;
#ifdef COVARNA_CONCURRENT_VECTOR
  typedef blocked_range<length_type> range_type;
#else
  typedef ser_blocked_range<length_type> range_type;
#endif
  typedef map<string, double> double_hash_type;

 public:

  virtual ~CorrelationFinder() { }

  /** Computes densities (hits per area) assuming that search is finished. */
  virtual  void augmentDensities(double_hash_type& densities, length_type searchMax) const = 0;

  /** Computes densities (hits per area) assuming that search is finished. */
  virtual  double_hash_type computeDensities(length_type searchMax, bool addEmtpy, ostream * os) const = 0;

  /** Returns natural logarithm of  probability of a particular stem to start at a particular position. Multiply with number of possible positions
   * to obtain E-value (either (N*(N-1))/2 for one MAF alignment (N == totalLength), or N*M for two MAF alignments */ 
  virtual double computeLogPValue(const Stem& stem) const = 0;

  /** Returns natural logarithm of  probability of a particular stem to start at a particular position. Multiply with number of possible positions
   * to obtain E-value (either (N*(N-1))/2 for one MAF alignment (N == totalLength), or N*M for two MAF alignments */ 
  virtual double computeForwardLogPValue(const Stem& stem) const = 0;

  virtual size_type getBasepairTypeMin() const = 0;

  virtual SingleLinkage2DProgressiveFilter& getClusterFilter() const = 0;

  /** Returns pointer to stored MAF alignment */
  virtual MAFAlignment* getMaf() const = 0;

  /** Correlation finder method as "strategy" pattern */
  virtual void run() const = 0;

  /** Runs algorithm in defined index vector set. Here: indices must be consecutive. They are counted as internal column number ,
   * this is different than the external genome position. */ 
  virtual void run(const range_type& range) const = 0;

  /** Returns computed results. Makes sense after calling run method. */
  virtual result_container getResults() const = 0;

  /** Sets base pair type minimum number of different correlated mutations per colum pair */
  virtual void setBasepairTypeMin(size_type n) = 0;

  /** checking of "wrong" diagonals NOT to be complementary. 0: no checking; 1: allow one; 2: none of both may be complementary */
  virtual void setCheckAntiNeighborMode(int mode) = 0;

  /** Sets cutoff of cluster filter */
  virtual void setClusterCutoffAndInit(length_type cutoff) = 0;

  /** Sets the "active" status of the cluster filter. If false, then the filter will simply pass through all input values. */
  virtual void setClusterFilterActive(bool active) = 0; 

  /** Sets the minimum size of clusters that pass the initial cluster filter. */
  virtual void setClusterFilterSizeMin(size_t sizeMin) = 0; 

  /** If true, look for stretches of (reverse) complementary columns. Default: true */
  virtual void setComplementMode(bool mode) = 0;

  virtual void setCorrDistMin(int dist) = 0;

  /** Require this many non-gap characters in a column */
  virtual void setNonGapMin(size_type n) = 0;

  /** Sets output intervall. Typical value: 100000 */
  virtual void setOutIntervall(length_type) = 0;

  /** If true, look for stretches of reverse (complementary) columns. Default: true */
  virtual void setReverseMode(bool mode) = 0;

  /** Sets maximum number of columns to be searched with linear search */ 
  virtual void setSearchColumnMax(size_type value) = 0;

  /** Sets maximum number of columns to be searched with linear search */ 
  virtual void setSearchColumnSplit(size_type value) = 0;

  /** Minimum number of consecutive correlations */
  virtual void setStemLengthMin(Stem::index_type len) = 0;

  /** Sets verbose level. 0: silent; 1: default; >1: more and more verbose */
  virtual void setVerbose(int verbose) = 0;

  /**Writes contests of results datastructure. Further filtering will be applied, but this helps to understand to estimate the density of hits.
   */
  virtual void writeRawResults(ostream& os, length_type searchMax) const = 0;


};

#endif
