// --*- C++ -*------x---------------------------------------------------------
#ifndef _MAF_SEARCH_TABLES3_
#define _MAF_SEARCH_TABLES3_

#include <queue>
#include <MAFAlignment.h>
#include <RankedSolution6.h>
#include <debug.h>
#include <SortedCompressedSequence2.h>
#include <SearchTables.h>

using namespace std;

class MAFSearchTables3 {

 public:

  typedef MAFAlignment::length_type length_type;

  typedef Vec<length_type> set_type;

  // typedef Vec<length_type> compressed_type;

  typedef SortedCompressedSequence2 compressed_type;
  
  // maps hashes like hg18_mm8_A_C to set containing column ids...
  typedef map<string, compressed_type > table_type;

  typedef map<string, compressed_type::const_iterator > iterator_table_type;

  typedef MAFAlignment::size_type size_type;

  enum { NO_AMBIGUITY = 1, COMPLEMENT_AMBIGUITY = 2, MATCH_AMBIGUITY = 3 };

  friend class HashCorrelationFinder3;

 private:

  int ambiguityMode;

  double assemblyPairFraction; // what fraction of all possible assembly pairs should be stored as hash tables? Between 0 and 1

  MAFAlignment *maf;

  table_type positionHashes; // maps hashes like hg18_mm8_pan_A_C_A to set containing column ids...

  iterator_table_type positionHashStarts;

  bool sameNucShortcut; // if true a (not-recommended) shortcut for searches is performed. Example: if false: triplet GGG can be found by CCC, CCU, CUC, UCC, CUU,UCU, UUC, UUU. If true: can only be found by CCC.

  length_type searchRangeMin;

  length_type searchRangeMax;

  int verbose;

 public:

  /** Generates hash tables that are later used by HashCorrelationFinder */
  MAFSearchTables3() : ambiguityMode(NO_AMBIGUITY), assemblyPairFraction(1.0), maf(0), sameNucShortcut(false), searchRangeMin(0), searchRangeMax(0), verbose(1) {
  }

  /** Generates hash tables that are later used by HashCorrelationFinder */
  MAFSearchTables3(MAFAlignment * _maf) : ambiguityMode(NO_AMBIGUITY), assemblyPairFraction(1.0), maf(_maf), sameNucShortcut(false),
	  searchRangeMin(0), verbose(1) {
    searchRangeMax = _maf -> getTotalLength();
  }

  MAFSearchTables3(const MAFSearchTables3& other) {
    copy(other);
  }

  MAFSearchTables3& operator = (const MAFSearchTables3& other) {
    if (this != &other) {
      copy(other);
    }
    return (*this);
  }

  virtual ~MAFSearchTables3() { }

  virtual void copy(const MAFSearchTables3& other) {
    ERROR("Copying of search tables not implemented!"); // would be extremely slow and probably unnecessary
  }

  /** Generates compressed set for internal use */
  // static compressed_type compressSet(const set_type& set);

  static set_type uncompressSet(const compressed_type& set) { 
    return set.toVector(); 
  }

  /** Tests compression and uncompression */
  // static void testCompressSet();

  virtual void run(const string& refAssembly) {
    createSearchHashTable(maf, maf->getAssemblies(), refAssembly);
    ASSERT(validate());
  }

  static string createHashTableHash(const string& assembly1, const string& assembly2, const string& assembly3,
				    char c1, char c2, char c3);

  // static string createHashTableHash(const string& assembly1, char c1);

  static void testCreateHashTableHash();

  virtual void createSearchHashTable(MAFAlignment *maf, const set<string>& assemblies, const string& refAssembly);

  virtual double getAssemblyPairFraction() const { return assemblyPairFraction; } 

  /** Returns uncompressed set with certain hash code. Precondition: hash code must exist */
  virtual const compressed_type& getSet(const string& hash) {
    PRECOND(findPositionHash(hash) != positionHashes.end());
    return findPositionHash(hash) -> second;
  }

  /** Estimates the potential number of hash table entries of two assemblies
   */
  virtual double estimateAssemblyTripleHashSize(const string& assem1, const string& assem2, const string& assem3) const;

  /** Returns iterator to beginning of correct hash table */
  virtual table_type::iterator findPositionHash(const string& hashhash) {
    return positionHashes.find(hashhash);
  }
  
  /** Resets all start positions to beginning of hash tables */
  virtual void resetPositionHashStarts() const {
    ASSERT(false);
/*     for (table_type::const_iterator it = positionHashes.begin(); it != positionHashes.end(); it++) { */
/*       positionHashStarts[it -> first] = it->second.begin(); */
/*     } */
  }

  virtual void setAmbiguityMode(int mode) { ambiguityMode = mode; }
  
  /** Sets parameter that determines what fraction of possible hash tables are actually generated */
  virtual void setAssemblyPairFraction(double value) { assemblyPairFraction = value; }

  virtual void setMAF(MAFAlignment* _maf) {
    maf = _maf;
    searchRangeMax = maf->getTotalLength();
  }

  /** Sets minimum values of column indices that can be part of search result */
  virtual void setSearchRangeMin(length_type n) { searchRangeMin = n; }

  /** Sets minimum values of column indices that can be part of search result */
  virtual void setSearchRangeMax(length_type n) { searchRangeMax = n; }

  /** Sets verbose level */
  virtual void setVerbose(int _verbose) { verbose = _verbose; }

  /** Returns true if properly defined. */
  virtual bool validate() const {
    return (positionHashes.size() > 0); //  && (positionHashes.size() == positionHashStarts.size());
  }
  
};

#endif
