// --*- C++ -*------x---------------------------------------------------------
// $Id: Histogram.h,v 1.1.1.1 2006/07/03 14:43:20 bindewae Exp $
//
// Class:           Histogram
// 
// Base class:      -
//
// Derived classes: - 
//
// Author:          Eckart Bindewald
//
// Description:     This class implements the concept of a histogram.
// 
// -----------------x-------------------x-------------------x-----------------

#ifndef __Histogram_H__
#define __Histogram_H__

// Includes

#include <iostream>

#include <debug.h>
#include <Vec.h>

class Histogram {

public:

  /* CONSTRUCTORS */

  Histogram();

  Histogram(unsigned int n,
	    double _min, double _step) : minx(_min), step(_step) { 
  
    histData = Vec<double>(n, 0.0);
  }

  Histogram(const Histogram& orig);

  virtual ~Histogram();

  /* OPERATORS */

  /** Assigment operator. */
  Histogram& operator = (const Histogram& orig);
  /** comparison operator */
  friend bool operator==(const Histogram& left, const Histogram& right);
  /** output operator */
  friend ostream& operator << (ostream& os, const Histogram& object);
  /** input operator  */
  friend istream& operator >> (istream& is, Histogram& object);

  /* PREDICATES */

  /** return left side x-value of first bin */
  double getMin() const { return minx; }

  /** return RIGHT side x-value of last bin */
  double getMax() const { return minx + step * histData.size(); }

  /** return bin heights */
  Vec<double> getData() const { return histData; }

  /** return width of bins */
  double getStep() const { return step; }

  /** get the bin height which corresponds to the correct bin of value */
  double getValue(double orig) const;

  /** Interpolate between bins. No interpolation, 
      if value is at middle of bin */
  double getInterpolatedValue(double orig) const;

  /** return height of bin # d */
  double getBinValue(unsigned int d) const;

  /** return x-value of left side of bin */
  double getBinStart(unsigned int d) const { return minx + (step * d); }

  /** return x-value of left side of bin */
  double getBinMiddle(unsigned int d) const { 
    return minx + (step * (d+0.5)); }

  bool isDefined(double d) {
    int bin = getBin(d);
    if ((bin >= 0) && (bin < static_cast<int>(histData.size()))) {
      return true;
    }
    return false;
  }

  /** return bin number which corresponds to the value */
  int getBin(double d) const;

  /* MODIFIERS */

  /** set overall start value */
  void setMin(double x) { minx = x; }

  /** add data element */
  void addData(double d) { 
    int bin = getBin(d);
    if ((bin >= 0) && (bin < static_cast<int>(histData.size()))) {
      histData[bin] += 1;
    }
  }


  /** set data vector */
  void setData(const Vec<double>& d) { histData = d; }

  /** set step size */
  void setStep(double orig) { step = orig; }

protected:
  /* OPERATORS  */
  /* PREDICATES */
  virtual bool compare(const Histogram& other) const;
  /* MODIFIERS  */
  virtual void copy(const Histogram& orig);

private:
  /* OPERATORS  */
  /* PREDICATES */
  /* MODIFIERS  */

private:
  /* PRIVATE ATTRIBUTES */
  Vec<double> histData;
  double minx;
  double step;

}; // class Histogram

#endif /* __A_CLASS_H__ */

