// --*- C++ -*------x---------------------------------------------------------
// $Id: Histogram4D.cc,v 1.2 2006/10/18 12:09:02 bindewae Exp $
//
// Class:           Histogram
// 
// Base class:      -
//
// Derived classes: - 
//
// Author:          Eckart Bindewald
//
// Description:     This class implements the concept of a histogram
// 
// -----------------x-------------------x-------------------x-----------------

#include <Histogram4D.h>
#include <vectornumerics.h>
#include <StringTools.h>

// ---------------------------------------------------------------------------
//                                   Histogram
// -----------------x-------------------x-------------------x-----------------

/* CONSTRUCTORS */

/** 
 * default operator
  */
Histogram4D::Histogram4D() : minx1(0.0), stepx1(1.0),
			     minx2(0.0), stepx2(1.0),
			     minx3(0.0), stepx3(1.0),
			     minx4(0.0), stepx4(1.0),
			     x1functor(0.0, 1.0),
			     x2functor(0.0, 1.0),
			     x3functor(0.0, 1.0),
			     x4functor(0.0, 1.0)
{
  // no precondition (except sufficient memory)
  // postcondition depends on class
}

/** copy constructor
  */
Histogram4D::Histogram4D(const Histogram4D& orig)
{
  // no precondition (except sufficient memory)
  copy(orig);
  //   POSTCOND( (orig == *this));
}

/** destructor
  */
Histogram4D::~Histogram4D()
{
  // no precondition (except sufficient memory)
  // postcondition depends on class
}


/* OPERATORS */

/** Assignement operator. */
Histogram4D& Histogram4D::operator = (const Histogram4D& orig)
{
  // no precondition
  if (&orig != this)
    {
      copy(orig);
    }
  POSTCOND( (orig == *this));
  return *this;
}

/** comparison operator. 
*/
bool 
operator == (const Histogram4D& left,const Histogram4D& right)
{
  // no precondition
  return left.compare(right);
}

/** output operator 
*/
ostream& 
operator << (ostream& os, const Histogram4D& object)
{
  PRECOND(os);

  os << object.getDimX1() << " " << object.minx1 << " " << object.stepx1 << endl
     << object.getDimX2() << " " << object.minx2 << " " << object.stepx2 << endl
     << object.getDimX3() << " " << object.minx3 << " " << object.stepx3 << endl
     << object.getDimX4() << " " << object.minx4 << " " << object.stepx4 << endl
     << object.histData;

  POSTCOND(os);
  return os;
}

/** input operator  
*/
istream& 
operator >> (istream& is, Histogram4D& object)
{
  ERROR_IF(!is, "Error reading histogram data file!");
  int nx1, nx2, nx3, nx4; // only dummies
  is >> nx1 >> object.minx1 >> object.stepx1
     >> nx2 >> object.minx2 >> object.stepx2
     >> nx3 >> object.minx3 >> object.stepx3
     >> nx4 >> object.minx4 >> object.stepx4
     >> object.histData;
  object.x1functor = HistogramStandardFunctor(object.minx1, object.stepx1);
  object.x2functor = HistogramStandardFunctor(object.minx2, object.stepx2);
  object.x3functor = HistogramStandardFunctor(object.minx3, object.stepx3);
  object.x4functor = HistogramStandardFunctor(object.minx4, object.stepx4);
  ERROR_IF(!is, "Error reading histogram!");
  ERROR_IF(object.histData.size() == 0,
	   "No histogram data read!");
  return is;
}


/* PREDICATES */


/**
 * returns true, if all members of other and this object 
 * give true with respect to "==" operator.
 */
bool 
Histogram4D::compare(const Histogram4D& other) const
{
  // no precondition

  ERROR("Compare method not yet implemented!");

  return false;
}

/** returns histogram value for x,y,z */
double 
Histogram4D::getValue(double x1, double x2, double x3, double x4) const
{
  PRECOND(isDefined(x1, x2, x3, x4));
  ASSERT(histData.size() > 0);
  return histData[getBinX1(x1)][getBinX2(x2)][getBinX3(x3)][getBinX4(x4)];  
}

/** same as getValue, but in case of out of bounds it uses nearest in-bound values */
double 
Histogram4D::getSafeValue(double x1, double x2, double x3, double x4) const
{
  ASSERT(histData.size() > 0);
  return histData[getSafeBinX1(x1)][getSafeBinX2(x2)][getSafeBinX3(x3)][getSafeBinX4(x4)];  
}

/*
double 
Histogram4D::getInterpolatedValue(double orig) const
{
  // get bin
  ERROR_IF(histData.size() == 0, "Unspecified histogram data!");
  double result = 0.0;
  // check if out of bounds: 
  if (orig < minx) { 
    result = getBinValue(0); 
  }
  else if (orig > getMax()) {
    result = getBinValue(histData.size() - 1);
  }
  else { // "normal" case
    int b = getBin(orig);
    double middle = getBinMiddle(b);
    
    if (orig >= middle) {
      result = getBinValue(b) + 
	( (orig-middle) * (getBinValue(b+1) - getBinValue(b)) / step );
    }
    else {
      if (b <= 0) {
	return getBinValue(0); // no interpolation of left side of first bin
      }
      double middle2 = middle -step;
      ASSERT(orig >= middle2);
      result = getBinValue(b-1) + 
	(orig-middle2) * (getBinValue(b) - getBinValue(b-1)) / step;
    }
  }
  return result;
}
*/

double 
Histogram4D::getBinValue(index_type nx1,
			 index_type nx2,
			 index_type nx3,
			 index_type nx4) const
{
  ERROR_IF(histData.size() == 0, "Unspecified histogram data!");
  if (nx1 < 0) {
    nx1 = 0;
  }
  else if (nx1 >= getDimX1()) {
    nx1 = getDimX1() - 1;
  }
  if (nx2 < 0) {
    nx2 = 0;
  }
  else if (nx2 >= getDimX2()) {
    nx2 = getDimX2() - 1;
  }
  if (nx3 < 0) {
    nx3 = 0;
  }
  else if (nx3 >= getDimX3()) {
    nx3 = getDimX3() - 1;
  }
  if (nx4 < 0) {
    nx4 = 0;
  }
  else if (nx4 >= getDimX4()) {
    nx4 = getDimX4() - 1;
  }

  return histData[nx1][nx2][nx3][nx4];
}

int
Histogram4D::getBinX1(double d) const
{
  // return static_cast<int>((d - minx1) / stepx1);
  return x1functor(d);
}

int
Histogram4D::getBinX2(double d) const
{
  // return static_cast<int>((d - minx2) / stepx2);
  return x2functor(d);
}

int
Histogram4D::getBinX3(double d) const
{
  // return static_cast<int>((d - minx3) / stepx3);
  return x3functor(d);
}

int
Histogram4D::getBinX4(double d) const
{
  // return static_cast<int>((d - minx4) / stepx4);
  return x4functor(d);
}


int
Histogram4D::getSafeBinX1(double d) const
{
  int bin = getBinX1(d);
  if (bin < 0) {
    return 0;
  }
  if (bin >= static_cast<int>(getDimX1())) {
    return getDimX1()-1;
  }
  return bin;
}
int
Histogram4D::getSafeBinX2(double d) const
{
  int bin = getBinX2(d);
  if (bin < 0) {
    return 0;
  }
  if (bin >= static_cast<int>(getDimX2())) {
    return getDimX2()-1;
  }
  return bin;
}
int
Histogram4D::getSafeBinX3(double d) const
{
  int bin = getBinX3(d);
  if (bin < 0) {
    return 0;
  }
  if (bin >= static_cast<int>(getDimX3())) {
    return getDimX3()-1;
  }
  return bin;
}
int
Histogram4D::getSafeBinX4(double d) const
{
  int bin = getBinX4(d);
  if (bin < 0) {
    return 0;
  }
  if (bin >= static_cast<int>(getDimX4())) {
    return getDimX4()-1;
  }
  return bin;
}


/* MODIFIERS */

void
Histogram4D::normalize() {
  double norm = elementSum(histData);
  for (unsigned int i = 0; i < histData.size(); ++i) {
    for (unsigned int j = 0; j < histData[i].size(); ++j) {
      for (unsigned int k = 0; k < histData[i][j].size(); ++k) {
	for (unsigned int m = 0; m < histData[i][j][k].size(); ++m) {
	  histData[i][j][k][m] = histData[i][j][k][m] / norm;
	}
      }
    }
  }
}

void
Histogram4D::normalizeX4() {
  for (unsigned int i = 0; i < histData.size(); ++i) {
    for (unsigned int j = 0; j < histData[i].size(); ++j) {
      for (unsigned int k = 0; k < histData[i][j].size(); ++k) {
	double norm = elementSum(histData[i][j][k]);
	for (unsigned int m = 0; m < histData[i][j][k].size(); ++m) {
	  histData[i][j][k][m] = histData[i][j][k][m] / norm;
	}
      }
    }
  }
}

void
Histogram4D::normalizeX3X4() {
  for (unsigned int i = 0; i < histData.size(); ++i) {
    for (unsigned int j = 0; j < histData[i].size(); ++j) {
      double norm = elementSum(histData[i][j]);
      for (unsigned int k = 0; k < histData[i][j].size(); ++k) {
	for (unsigned int m = 0; m < histData[i][j][k].size(); ++m) {
	  histData[i][j][k][m] = histData[i][j][k][m] / norm;
	}
      }
    }
  }
}



/** copies orig object to this object ("deep copy")
  */
void 
Histogram4D::copy(const Histogram4D& orig)
{
  // no precondition

  histData = orig.histData;
  minx1 = orig.minx1;
  stepx1 = orig.stepx1;
  minx2 = orig.minx2;
  stepx2 = orig.stepx2;
  minx3 = orig.minx3;
  stepx3 = orig.stepx3;
  minx4 = orig.minx4;
  stepx4 = orig.stepx4;
  x1functor = orig.x1functor;
  x2functor = orig.x2functor;
  x3functor = orig.x3functor;
  x4functor = orig.x4functor;
  //   POSTCOND(orig == *this);
}




