#include <iostream>
#include <fstream>
#include <string>
#include <Vec.h>
#include <GetArg.h>
#include <debug.h>
#include <StringTools.h>
#include <vectornumerics.h>

#define AREA_VERSION "0.5.1"

/** Version history 
 *  0.5.0 Initial production version
 *  0.5.1 added option -a with possible modes "enrich" and "contrib"
 */

using namespace std;

typedef string::size_type size_type;


/** output of command line parameter with which the program was called. */
void
parameterOutput(ostream& os, int argc, char** argv)
{
  for (int i = 0; i < argc; i++)
    {
      os << argv[i] << " ";
    }
  os << endl;
}

Vec<Vec<Vec<double> > > 
areaEnrich(istream& is, const Vec<size_type>& colVec1, const Vec<size_type>& colVec2,
	   size_type start1Col, size_type end1Col, size_type start2Col, size_type end2Col,
	   size_type genomeLength, const vector<unsigned int>& featCoverages,
	   const string& enrichMode) {
  size_type n = colVec1.size();
  ERROR_IF(n != featCoverages.size(), "Number of feature base coverages has to match number of specified features. Use option --coverage");
  Vec<Vec<double> > frac1(n, Vec<double>(n, 0.0));
  Vec<Vec<double> > frac2(n, Vec<double>(n, 0.0));
  Vec<Vec<double> > sumProd(n, Vec<double>(n, 0.0));
  Vec<Vec<double> > localArea(n, Vec<double>(n, 0.0));
  Vec<Vec<double> > enrich = sumProd;
  while (is) {
    string line = getLine(is);
    if (line.size() == 0 || line[0] == '#') {
      continue;
    }
    vector<string> tokens=getTokens(line);
    for (size_type i = 0; i < n; ++i) {
      size_type col1 = colVec1[i];
      double val1 = stod(tokens[col1]);
      size_type len1 = stoui(tokens[end1Col]) - stoui(tokens[start1Col]);
      for (size_type j = 0; j < n; ++j) {
	size_type len2 = stoui(tokens[end2Col]) - stoui(tokens[start2Col]);
	size_type col2 = colVec2[j];
	localArea[i][j] += len1 * len2; // local interaction area for this type of interaction
	double val2 = stod(tokens[col2]);
	sumProd[i][j] += val1 * val2;
      }
    }
  }
  // post-processing
  for (size_type i = 0; i < n; ++i) {
    for (size_type j = 0; j < n; ++j) {
      // fraction of combined features given predicted interaction areas
      if ((localArea[i][j] > 0) && (featCoverages[i]) > 0 && (featCoverages[j] > 0)) {
	frac1[i][j] = sumProd[i][j]/localArea[i][j];
	frac2[i][j] = (static_cast<double>(featCoverages[i])/genomeLength) 
	  * (static_cast<double>(featCoverages[j])/genomeLength);
	if (frac2[i][j] > 0) {
	  if (enrichMode == "enrich") {
	    enrich[i][j] = frac1[i][j]/frac2[i][j];
	  } else if (enrichMode == "contrib") {
	    enrich[i][j] = frac1[i][j];
	  } else {
	    ERROR("Unknown enrichment mode! Allowed: enrich|contrib");
	  }
	}
      }
    }
  }
  Vec<Vec<Vec<double> > > result;
  result.push_back(enrich);
  return result;
}

int
main(int argc, char ** argv) {
  cout << "# areaenrich version " << AREA_VERSION << endl;
  cout << "# Program called with parameters: ";
  parameterOutput(cout, argc, argv);
  // tbl = read.table("result_36182h_36183h_36192h_36193h_Pstem_0.01_E2_0.01_stems2_farinter_nointra_annoFBGenefRNAdb_coverages_head.bed")
  string filename;
  Vec<size_type> colIds1;
  Vec<size_type> colIds2;
  string enrichMode = "enrich";
  size_type step = 0;
  size_type nFeat = 0;
  size_type startCol1 = 0;
  size_type startCol2 = 0;
  size_type startRCol1 = 0;
  size_type startRCol2 = 0;
  size_type genomeLength = 0;
  vector<unsigned int> featCoverages;
  string outFileBase = "areaenrich_";
  getArg("a", enrichMode, argc, argv, enrichMode);
  getArg("i", filename, argc, argv, filename);
  getArg("f", nFeat, argc, argv, nFeat); // number of features
  getArg("-c1", startCol1, argc, argv, startCol1);
  getArg("-c2", startCol2, argc, argv, startCol2);
  ERROR_IF(startCol1 < 1, "Start feature column has to be greater zero. Use option --c1");
  ERROR_IF(startCol2 < 1, "Start feature column has to be greater zero. Use option --c2");
  --startCol1;
  --startCol2; // internal counting

  getArg("-coverages", featCoverages, argc, argv);
  getArg("L", genomeLength, argc, argv, genomeLength);
  ERROR_IF(genomeLength < 1, "Genome length has to be greater zero. Use option -L");

  getArg("o", outFileBase, argc, argv, outFileBase);
  getArg("t", step, argc, argv, step);
  getArg("-r1", startRCol1, argc, argv, startRCol1);
  getArg("-r2", startRCol2, argc, argv, startRCol2);
  ERROR_IF(startRCol1 < 1, "Start region column has to be greater zero. Use option --r1");
  ERROR_IF(startRCol2 < 1, "Start region column has to be greater zero. Use option --r2");
  --startRCol1;
  --startRCol2;
  size_type endRCol1 = startRCol1 + 1;
  size_type endRCol2 = startRCol2 + 1;
  ifstream ifile(filename.c_str());
  
  ERROR_IF(!ifile, "Error opening input file " + filename);
  // nFeat = 22;
  // startCol1 = 23;
  // size_type endCol1 = step*(nFeat -1) + startCol1;
  //startCol2 = endCol1 + 2;
  // size_type endCol2 = step*(nFeat -1) + startCol2;
  Vec<size_type> colVec1, colVec2;
  for (size_type i = 0;  i < nFeat; ++i) {
    colVec1.push_back(startCol1 + (i *step));
    colVec2.push_back(startCol2 + (i *step));
  }
  ERROR_IF(colVec1.size() != nFeat, "Internal error in line 111");
  ERROR_IF(colVec2.size() != nFeat, "Internal error in line 111");
  cout << "# using columns (1): " << externalCounting(colVec1) << endl;
  cout << "# using columns (2): " << externalCounting(colVec2) << endl;
  Vec<Vec<Vec<double> > > result = areaEnrich(ifile, colVec1, colVec2,
			     startRCol1, endRCol1, startRCol2, endRCol2, genomeLength,
					      featCoverages, enrichMode);
  ERROR_IF(result.size() < 1, "No result computed; internal error in line 84");
  string enrichFileName = outFileBase + enrichMode + ".mtx";
  ofstream enrichFile(enrichFileName.c_str());
  ERROR_IF(!enrichFile, "Error opening enrichment file: " + enrichFileName);
  writeMatrix(enrichFile, result[0]);
  writeMatrix(cout, result[0]);
  enrichFile.close();
  return 0;
}

