// --*- C++ -*------x---------------------------------------------------------
// $Id:
//
// Author:          Eckart Bindewald
//
// Description:     simple numeric functions. Could be used for
//                  more sophisticated matrix or vector classes.
// 
// -----------------x-------------------x-------------------x-----------------

#ifndef __VECTOR_NUMERICS_H__
#define __VECTOR_NUMERICS_H__

#define __STL_NO_DRAND48 

// Includes

#include <math.h>
#include <limits.h>
#include <debug.h>
#include <list>
#include <set>
#include <Vec.h>
// #include <nrutil.h>
#include <algorithm>
#include <Matrix3D.h>
#include <Vector3D.h>
#include <RankedSolution4.h>
#include <Limits.h>
#include <generalNumerics.h>
#include <StringTools.h>

#ifdef COVARNA_CONCURRENT_VECTOR
#include <tbb/concurrent_vector.h>
using namespace tbb;
#endif

// typedef Vec<Vector3D> Graph;

/** return false if not a number (NaN) */
bool isDefined(const Vector3D& v);

/** return true, if not obscure number */
template <class T>
inline
bool
isReasonable(const Vec<T>& v)
{
  for (unsigned int i = 0; i < v.size(); ++i) {
    if (!isReasonable(v[i])) {
      return false;
    }
  }
  return true;
}


/** returns error function for positive values of z
    approximation according to Abramowitz and Stegun, 1964 
    (http://www.geo.vu.nl/users/hydro/students/hd537/error_function.pdf) */
double errorFunction(double z);

/** returns square of number */
template <class T>
T
square(const T& x)
{
  return x * x;
}

/* translate list into vector */
template<class T>
inline
Vec<T>
listToVector(const list<T>& lst)
{
  Vec<T> result(lst.size());
  unsigned int count = 0;
  for (typename list<T>::const_iterator i = lst.begin(); i != lst.end(); ++i) {
    ASSERT(count < result.size());
    result[count++] = *i;
  }
  return result;
}

/* translate list into vector */
/*
// template<class T>
inline
Vec<unsigned int>
listToVector(const list<unsigned int>& lst)
{
  Vec<unsigned int> result(lst.size());
  unsigned int count = 0;
  for (list<unsigned int>::const_iterator i = lst.begin(); i != lst.end(); i++) {
    ASSERT(count < result.size());
    result[count++] = *i;
  }
  return result;
}
*/

/** n trials with counts.size() possible outcomes are performed.
 * Given a vector of probabilities of observing each outcome,
 * it returns the probability of observing a particular patterns of counts of outcomes.
 * Corresponds to trials "with replacement"; use hypergeometric distribution if "without replacement"
 */
double
logMultinomial(const Vec<int>& counts, const Vec<double>& probs, int n);

/**
 * translates vector of unsigned integer to vector of integer
 */
Vec<int>
uivtoiv(const Vec<unsigned int>& v);

/**
 * translates vector of integer to vector of unsigned integer
 */
Vec<unsigned int>
ivtouiv(const Vec<int>& v);

/* return smaller entity. If equal,return a */
template <class T>
inline
T
minimum(const T& a, const T& b)
{
  if (b < a) {
    return b;
  }
  return a;
}

/** returns j'th column of matrix */
template<class T>
inline
Vec<T>
getColumn(const Vec<Vec<T> >& m, unsigned int j)
{
  Vec<T> result(m.size());
  for (unsigned int i = 0; i < m.size(); ++i) {
    ASSERT(j < m[i].size());
    result[i] = m[i][j];
  }
  return result;
}

/** Multiplies all row and columns with vector entries */
void
modulateMatrix(Vec<Vec<double> >& matrix,
	       const Vec<double>& modulateValues);

/** returns j'th column of matrix */
template<class T>
inline
void
setColumn(Vec<Vec<T> >& m, const Vec<T>& v, unsigned int j)
{
  PRECOND(m.size()==v.size());
  for (unsigned int i = 0; i < m.size(); ++i) {
    ASSERT(j < m[i].size());
    m[i][j] = v[i];
  }
}


/** returns j'th column of matrix */
template<class T>
inline
Vec<Vec<T> >
getYSlice(const Vec<Vec<Vec<T> > >& m, unsigned int y)
{
  Vec<Vec<T> >result(m.size(), Vec<T>(m[0][0].size()));
  for (unsigned int i = 0; i < m.size(); ++i) {
    for (unsigned int j = 0; j < m[0][0].size(); ++j) {
      ASSERT(i < m[i].size());
      result[i][j] = m[i][y][j];
    }
  }
  return result;
}

/** returns j'th column of matrix */
template<class T>
inline
void
setYSlice(Vec<Vec<Vec<T> > >& m, const Vec<Vec<T> >& v, unsigned int j)
{
  PRECOND(m.size()==v.size());
  for (unsigned int i = 0; i < m.size(); ++i) {
    for (unsigned int k = 0; k < m[0][0].size(); ++k) {
      m[i][j][k] = v[i][k];
    }
  }
}


/** returns j'th column of matrix */
template<class T>
inline
Vec<Vec<T> >
getZSlice(const Vec<Vec<Vec<T> > >& m, unsigned int z)
{
  Vec<Vec<T> >result(m.size(), Vec<T>(m[0].size()));
  for (unsigned int i = 0; i < m.size(); ++i) {
    for (unsigned int j = 0; j < m[0].size(); ++j) {
      result[i][j] = m[i][j][z];
    }
  }
  return result;
}

/** returns j'th column of matrix */
template<class T>
inline
void
setZSlice(Vec<Vec<Vec<T> > >& m, const Vec<Vec<T> >& v, unsigned int z)
{
  PRECOND(m.size()==v.size());
  for (unsigned int i = 0; i < m.size(); ++i) {
    for (unsigned int j = 0; j < m[0].size(); ++j) {
      m[i][j][z] = v[i][j];
    }
  }
}


/** returns j'th column of matrix */
template<class T>
inline
Vec<Vec<T> >
getColumns(const Vec<Vec<T> >& m, const Vec<unsigned int> cols)
{
  Vec<Vec<T> > result(m.size(), Vec<T>(cols.size()));
  for (unsigned int i = 0; i < m.size(); ++i) {
    for (unsigned int j = 0; j < cols.size(); ++j) {
      ASSERT(cols[j] < m[i].size());
      result[i][j] = m[i][cols[j]];
    }
  }
  return result;
}

/** returns j'th column of matrix, ignore too short rows */
template<class T>
inline
Vec<T>
getColumnSave(const Vec<Vec<T> >& m, unsigned int j)
{
  Vec<T> result;
  for (unsigned int i = 0; i < m.size(); ++i) {
    if (m[i].size() > j) {
      result.push_back(m[i][j]);
    }
  }
  return result;
}

/** deletes one row of matrix */
template<class T>
inline
void
deleteMatrixRow(Vec<Vec<T> >& mtx,
		unsigned int n)
{
  PRECOND(n < mtx.size());
  mtx.erase(mtx.begin()+n);
}

/** deletes one column of matrix */
template<class T>
inline
void
deleteMatrixCol(Vec<Vec<T> >& mtx,
		unsigned int n)
{
  PRECOND((mtx.size() > 0) && (n < mtx[0].size()));
  for (unsigned int i = 0; i < mtx.size(); ++i) {
    mtx[i].erase(mtx[i].begin()+n);
  }
}

/* return larger entity. If equal return b */
template <class T>
inline
T
maximum(const T& a, const T& b)
{
  if (a < b) {
    return b;
  }
  return a;
}

/** 
 * returns vector in which each element i is the maximum of the i'th element of a and the i'th element of b
 */
template <class T>
inline
Vec<T>
maxVector(const Vec<T>& a, const Vec<T>& b) {
  PRECOND(a.size() == b.size());
  unsigned int n = a.size();
  Vec<T> c(n);
  for (unsigned int i = 0; i < n; ++i) {
    c[i] = maximum(a[i], b[i]);
  }
  return c;
}

/** 
 * returns vector in which each element i is the maximum of the i'th element of a and the i'th element of b
 */
template <class T>
inline
Vec<T>
minVector(const Vec<T>& a, const Vec<T>& b) {
  PRECOND(a.size() == b.size());
  unsigned int n = a.size();
  Vec<T> c(n);
  for (unsigned int i = 0; i < n; ++i) {
    c[i] = minimum(a[i], b[i]);
  }
  return c;
}

Vec<int> uiVecToIVec(const Vec<unsigned int>& v);

Vec<unsigned int> iVecToUiVec(const Vec<int>& v);


/** return true, if matrix is symmetrix */
bool
isSymmetric(const Vec<Vec<double> >& m);

/** symmetrizes matrix */
void
symmetrize(Vec<Vec<double> >& m);

/** my version of maximum function */
template <class T>
inline
T
maxFunc(const T& a, const T& b)
{
  if (a < b) {
    return b;
  }
  return a;
}

/** my version of minimum function */
template <class T>
inline
T
minFunc(const T& a, const T& b)
{
  if (b < a) {
    return b;
  }
  return a;
}

// generate array starting with 0 , incresing like a stair
Vec<unsigned int>
generateStair(unsigned int maxi, unsigned int mini = 0, unsigned int step = 1);

// return permutation of vector
template <class T>
Vec<T>
permutate(const Vec<T>& v, const Vec<unsigned int>& permut)
{
  PRECOND(permut.size() == v.size());
  Vec<T> result(v.size());
  for (unsigned int i = 0; i < v.size(); ++i) {
    ASSERT(permut[i] < v.size());
    result[i] = v[permut[i]];
  }
  return result;
}

// devides orig data into v1 (with size numPart) and v2 (size.orig -numpart)
// orderOrig: specifies order of v1 and v2
// may not contain dublicates!
template<class T>
inline
void
generatePartition(Vec<T>& orig,
		  Vec<T>& v1, 
		  Vec<T>& v2,
		  unsigned int numPart,
		  const Vec<unsigned int>& orderOrig)
{
  PRECOND(numPart <= orig.size()); 
  Vec<unsigned int> order;
  if (orderOrig.size() != orig.size()) {
    order = generateStair(orig.size());
  }
  else {
    order = orderOrig;
  }
  if (numPart == 0) {
    v1.clear();
    v2 = orig;
  }
  else if (numPart >= orig.size()) {
    numPart = orig.size();
    v1 = orig;
    v2.clear();
  }
  else {
    v1 = Vec<T>(numPart);
    v2 = Vec<T>(orig.size() - numPart);
  }
  for (unsigned int i = 0; i < numPart; ++i) {
    ASSERT(order[i] < orig.size());
    v1[i] = orig[order[i]]; 
  }
  for (unsigned int i = numPart; i < orig.size(); ++i) {
    ASSERT(order[i] < orig.size());
    v2[i-numPart] = orig[order[i]]; 
  }
  POSTCOND(v1.size() + v2.size() == orig.size());  
}

// generate 2 partitions from orig
// v2 has size 1
template<class T>
inline
void
generatePartitionLeaveOneOut(Vec<T>& orig,
			     Vec<T>& v1, 
			     Vec<T>& v2,
			     unsigned int numPart)
{
  PRECOND(numPart <= orig.size()); 
  ERROR_IF(orig.size() == 0, "Cannot generate partition of empty set!");
  if (orig.size() == 1) {
    v1 = Vec<T>();
    v2 = orig;
    return;
  }
  v1 = Vec<T>(orig.size() - 1);
  v2 = Vec<T>(1, orig[numPart]);
  for (unsigned int i = 0; i < numPart; ++i) {
    v1[i] = orig[i];
  }
  for (unsigned int i = numPart+1; i < orig.size(); ++i) {
    v1[i-1] = orig[i];
  }

  POSTCOND(v1.size() + v2.size() == orig.size());
}

/*
// devides orig data into v1 (with size numPart) and v2 (size.orig -numpart)
// orderOrig: specifies order of v1 and v2
// may not contain dublicates!
template<class T>
inline
void
generatePartition(Vec<T>& orig,
		  Vec<T>& v1, 
		  Vec<T>& v2,
		  unsigned int numPart,
		  const Vec<unsigned int>& orderOrig)
{
  PRECOND(numPart <= orig.size()); 
  Vec<unsigned int> order;
  if (orderOrig.size() != orig.size()) {
    order = generateStair(orig.size());
  }
  else {
    order = orderOrig;
  }
  if (numPart == 0) {
    v1.clear();
    v2 = orig;
  }
  else if (numPart >= orig.size()) {
    numPart = orig.size();
    v1 = orig;
    v2.clear();
  }
  else {
    v1 = Vec<T>(numPart);
    v2 = Vec<T>(orig.size() - numPart);
  }
  for (unsigned int i = 0; i < numPart; ++i) {
    ASSERT(order[i] < orig.size());
    v1[i] = orig[order[i]]; 
  }
  for (unsigned int i = numPart; i < orig.size(); ++i) {
    ASSERT(order[i] < orig.size());
    v2[i-numPart] = orig[order[i]]; 
  }
  POSTCOND(v1.size() + v2.size() == orig.size());  
}
*/

/* return vector with size numReturn, which has randomly assigned non-redundant bin values between min and max */
Vec<double>
rouletteWheel(double min, double max, unsigned int numBins, unsigned int numReturn);

/**
 * chooses an element indix from vector pVec with probability given in pVec
 */
unsigned int
chooseRouletteWheel(const Vec<double>& pVec); 

/**
 * chooses an element indix from vector pVec with probability given in pVec
 * imagine probabilities like pieces of "pie" 
 * given is also a list of taboo indices.
 */
unsigned int
chooseRouletteWheel(const Vec<double>& pVec, const Vec<unsigned int>& tabooList);

/**
 * chooses an element indix from vector pVec with probability given in pVec
 * imagine probabilities like pieces of "pie" 
 * given is also a list of taboo indices.
 */
unsigned int
chooseRouletteWheel2(const Vec<double>& pVec, const Vec<unsigned int>& tabooList);

/**
 * chooses an element indix from vector pVec with probability given in pVec
 * imagine probabilities like pieces of "pie" 
 * given is also a list of taboo indices.
 */
unsigned int
chooseRouletteWheel3(const Vec<double>& pVec, const set<unsigned int>& tabooList);

/** Thread-safe version based on rand_r
 * chooses an element indix from vector pVec with probability given in pVec
 * imagine probabilities like pieces of "pie" 
 * given is also a list of taboo indices.
 */
unsigned int
chooseRouletteWheel4(const Vec<double>& pVec, const set<unsigned int>& tabooList, unsigned int * seed);

/** Thread-safe version based on rand_r
 * chooses an element indix from vector pVec with probability given in pVec
 * imagine probabilities like pieces of "pie" 
 * given is also a list of taboo indices.
 */
unsigned int
chooseRouletteWheel4(const Vec<double>& pVec, const Vec<unsigned int>& tabooList, unsigned int * seed);


/** Thread safe version of chooseRouletteWheel
 * chooses an element indix from vector pVec with probability given in pVec
 */
unsigned int
chooseRouletteWheel4(const Vec<double>& pVec, unsigned int * seed); 

/**
 * generates non-redundant vecor of unsigned int with num entries between minId and maxId
 */
Vec<unsigned int>
generateRandomIndexSubset(unsigned int num,
                          unsigned int maxId, unsigned int minId);

/**
 * generates vecor of unsigned int with num entries between 0 and num
 * no indices can be double, WITH "replacement", use for bootstrap!
 */
Vec<unsigned int>
generateBootSubset(unsigned int num);

/** c = a + b */
template <class T>
inline
void
vectorAdd(const Vec<T>& a, const Vec<T>& b, Vec<T>& c)
{
  ERROR_IF((a.size() != b.size()),
	   "Adding vectors of different sizes!");
  if (a.size() == 0) {
    c.clear();
    return;
  }
  if (c.size() != a.size()) {
    c = Vec<T>(a.size());
  }
  for (unsigned int i = 0; i < a.size(); ++i) {
    c[i] = a[i] + b[i];
  }
}

/** c = a + b */
template <class T, class S>
inline
Vec<Vec<double> >
matrixAdd(const Vec<Vec<T> >& a, const Vec<Vec<S> >& b)
{
  PRECOND((a.size() == b.size()));
  Vec<Vec<T> > c = a;
  for (unsigned int i = 0; i < a.size(); ++i) {
    for (unsigned int j = 0; j < a[i].size(); ++j) {
      c[i][j] = a[i][j] + b[i][j];
    }
  }
  return c;
}

/** a <- a + b */
template <class T, class S>
inline
void
matrixAdd2(Vec<Vec<T> >& a, const Vec<Vec<S> >& b)
{
  PRECOND((a.size() == b.size()));
  for (unsigned int i = 0; i < a.size(); ++i) {
    for (unsigned int j = 0; j < a[i].size(); ++j) {
      a[i][j] += b[i][j];
    }
  }
}

/** a <- a - b */
template <class T, class S>
inline
void
matrixSub2(Vec<Vec<T> >& a, const Vec<Vec<S> >& b)
{
  PRECOND((a.size() == b.size()));
  for (unsigned int i = 0; i < a.size(); ++i) {
    for (unsigned int j = 0; j < a[i].size(); ++j) {
      a[i][j] -= b[i][j];
    }
  }
}

/** c = a + b */
template <class T>
inline
Vec<Vec<double> >
matrixSquareAdd(const Vec<Vec<T> >& a, const Vec<Vec<T> >& b)
{
  PRECOND((a.size() == b.size()));
  Vec<Vec<T> > c = a;
  for (unsigned int i = 0; i < a.size(); ++i) {
    for (unsigned int j = 0; j < a[i].size(); ++j) {
      c[i][j] = a[i][j] + (b[i][j] * b[i][j]);
    }
  }
  return c;
}

/** cij = max(aij, bij) for each element */
template <class T>
inline
Vec<Vec<double> >
matrixMax(const Vec<Vec<T> >& a, const Vec<Vec<T> >& b)
{
  PRECOND((a.size() == b.size()));
  Vec<Vec<T> > c = a;
  for (unsigned int i = 0; i < a.size(); ++i) {
    for (unsigned int j = 0; j < a[i].size(); ++j) {
      if (a[i][j] >= b[i][j]) {
	c[i][j] = a[i][j];
      }
      else {
	c[i][j] = b[i][j];
      }
    }
  }
  return c;
}

/** c = a + b */
template <class T>
inline
void
vectorAdd(const Vec<T>& a, const T& b, Vec<T>& c)
{
  PRECOND((a.size() == c.size()));
  for (unsigned int i = 0; i < a.size(); ++i) {
    c[i] = a[i] + b;
  }
}

/** c = a - b */
template <class T>
inline
void
vectorSubtract(const Vec<T>& a,
	       const Vec<T>& b, 
	       Vec<T>& c)
{
  ERROR_IF((a.size() != b.size()), 
	   "Internal error in line 471: subtracting vectors of different dimensions!");
  if (a.size() == 0) {
    c.clear();
    return;
  }
  if (c.size() != a.size()) {
    c = Vec<T>(a.size());
  }
  for (unsigned int i = 0; i < a.size(); ++i) {
    c[i] = a[i] - b[i];
  }
}

/** a <- b a  */
void
scalarMul(Vec<double>& a, double b);

/** c = a . b */
double
dotProduct(const Vec<double>& a,
	   const Vec<double>& b);

/** return sum of elements */
double
elementSum(const Vec<double>& v);

/** return sum of elements */
template<class T>
inline
T
elementSum(const Vec<T>& v) {
  T result = 0;
  for (Vec<int>::size_type i = 0; i < v.size(); ++i) {
    result += v[i];
  }
  return result;
}

/** return sum of elements */
template <class T>
inline
T
elementSum(const Vec<Vec<T> >& v)
{
  T sum = 0;
  for (unsigned int i = 0; i < v.size(); ++i) {
    for (unsigned int j = 0; j < v[i].size(); ++j) {
      sum += v[i][j];
    }
  }
  return sum;
}

/** return sum of elements */
template <class T>
inline
T
elementSquareSum(const Vec<Vec<T> >& v)
{
  T sum = 0;
  for (unsigned int i = 0; i < v.size(); ++i) {
    for (unsigned int j = 0; j < v[i].size(); ++j) {
      sum += v[i][j]*v[i][j];
    }
  }
  return sum;
}

/** return sum of elements */
template <class T>
inline
T
elementSum(const Vec<Vec<Vec<T> > >& v)
{
  T sum = 0;
  for (unsigned int i = 0; i < v.size(); ++i) {
    for (unsigned int j = 0; j < v[i].size(); ++j) {
      for (unsigned int k = 0; k < v[i][j].size(); ++k) {
	sum += v[i][j][k];
      }
    }
  }
  return sum;
}

/** return sum of elements */
template <class T>
inline
T
elementSum(const Vec<Vec<Vec<Vec<T> > > >& v)
{
  T sum = 0;
  for (unsigned int i = 0; i < v.size(); ++i) {
    for (unsigned int j = 0; j < v[i].size(); ++j) {
      for (unsigned int k = 0; k < v[i][j].size(); ++k) {
	for (unsigned int m = 0; m < v[i][j][k].size(); ++m) {
	  sum += v[i][j][k][m];
	}
      }
    }
  }
  return sum;
}

/** returns "1.0" for perfectly "sharp" predictions
    (each row with all zeros and one "1"),
    final result is smaller one otherwise
*/
double
contactMatrixQuality(const Vec<Vec<double> >& matrix, unsigned int diagLim);

/** return index of maximum value element */
template<class T>
inline
unsigned int
maxElement(const Vec<T>& v)
{
  PRECOND(v.size() > 0);
  unsigned int bestIndex = 0;
  T bestVal = v[0];
  for (unsigned int i = 1; i < v.size(); ++i) {
    if (v[i] > bestVal) {
      bestVal = v[i];
      bestIndex = i;
    }
  }
  return bestIndex;
}

/** return index of minimum value element */
template<class T>
inline
unsigned int
minElement(const Vec<T>& v)
{
  PRECOND(v.size() > 0);
  unsigned int bestIndex = 0;
  T bestVal = v[0];
  for (unsigned int i = 1; i < v.size(); ++i) {
    if (v[i] < bestVal) {
      bestVal = v[i];
      bestIndex = i;
    }
  }
  return bestIndex;
}

// transform such that sum of values is 1.0
void
probabilityNormalize(Vec<double>& v);

// transform such that sum of values is targetSum
void
probabilityNormalize(Vec<double>& v, double targetSum);

// transform such that sum of values is 1.0
void
probabilityNormalize(Vec<Vec<double> >& v);

// transform such that sum of values is 1.0
void
bayesianNormalize(Vec<double>& v);

// transform such that sum of values is 1.0
void
bayesianNormalize(Vec<Vec<double> >& v);

// compute euclidian norm of vector
template <class T>
inline
double
euclidianNorm(const Vec<T>& v)
{
  T sum = 0;
  for (unsigned int i = 0; i < v.size(); ++i) {
    sum += v[i] * v[i];
  }
  return sqrt(static_cast<double>(sum));
}

// compute euclidian norm of vector
template <class T>
inline
double
euclidianNormSquare(const Vec<T>& v)
{
  T sum = 0;
  for (unsigned int i = 0; i < v.size(); ++i) {
    sum += v[i] * v[i];
  }
  return static_cast<double>(sum);
}


template <class T>
inline
double 
euclidianDistance(const Vec<T>& v1, const Vec<T>& v2)
{
  ERROR_IF(v1.size() != v2.size(), "Different dimensions encountered!");
  ERROR_IF(v1.size() == 0, "Zero dimension encountered!");
  Vec<T> v3(v1.size());
  vectorSubtract(v1, v2, v3);
  return euclidianNorm(v3);
}

template <class T>
inline
double 
euclidianDistanceSquare(const Vec<T>& v1, const Vec<T>& v2)
{
  ERROR_IF(v1.size() != v2.size(), "Different dimensions encountered!");
  ERROR_IF(v1.size() == 0, "Zero dimension encountered!");
  Vec<T> v3(v1.size());
  vectorSubtract(v1, v2, v3);
  return euclidianNormSquare(v3);
}

template <class T>
inline
double 
euclidianNorm(const Vec<Vec<T> >& v1)
{
  ERROR_IF(v1.size() == 0, "Zero dimension encountered!");
  Vec<Vec<T> > v2 = v1;
  return sqrt(elementSquareSum(v2));
}

template <class T>
inline
double 
euclidianDistance(const Vec<Vec<T> >& v1, const Vec<Vec<T> >& v2)
{
  ERROR_IF(v1.size() != v2.size(), "Different dimensions encountered!");
  ERROR_IF(v1.size() == 0, "Zero dimension encountered!");
  Vec<Vec<T> > v3 = v1;
  matrixSub2(v3, v2);
  return sqrt(elementSquareSum(v3));
}

/** square of euclidian norm, however with scaling of dimensions */
template <class T>
inline
double 
euclidianDistanceSquare(const Vec<T>& v1, const Vec<T>& v2, 
			const Vec<double>& scaling)
{
  ERROR_IF(v1.size() != v2.size(), "Different dimensions encountered!");
  ERROR_IF(v1.size() == 0, "Zero dimension encountered!");
  Vec<T> v3(v1.size());
  vectorSubtract(v1, v2, v3);
  for (unsigned int i = 0; i < v3.size(); ++i) {
    v3[i] *= scaling[i];
  }
  return euclidianNormSquare(v3);
}


// compute euclidian norm of vector
inline
double
absNorm(const Vec<double>& v) {
  double result = 0.0;
  for (unsigned int i = 0; i < v.size(); ++i) {
    result += fabs(v[i]);
  }
  return result;
}

// transform such that sum of SQUARED values is 1.0
void
normalizeEuclidian(Vec<double>& v);

// transform such that sum of SQUARED values is 1.0
inline
void
normalizeAbs(Vec<double>& v)
{
  double norm = absNorm(v);
  if (norm <= 0.0) {
    return;
  }
  norm = 1.0 / norm;
  for (unsigned int i = 0; i < v.size(); ++i) {
    v[i] *= norm;
  }
}

/** return mean value of elements */
template<class T>
inline
double
vecMean(const Vec<T>& v, unsigned int start, unsigned int end)
{
  PRECOND((v.size() > 0) && (start < end) && (end <= v.size()));
  double sum = 0.0;
  for (unsigned int i = start; i < end; ++i) {
    sum += static_cast<double>(v[i]);
  }
  return sum / (end - start);
}

/** return mean value of elements */
inline
Vec<double>
vecInverse(const Vec<double>& v)
{
  Vec<double> result(v.size(), 0.0);
  for (unsigned int i = 0; i < v.size(); ++i) {
    if (v[i] != 0.0) {
      result[i] = 1.0 / v[i];
    }
  }
  return result;
}

/** return mean value of elements */
template<class T>
inline
double
vecMean(const Vec<T>& v)
{
  PRECOND(v.size() > 0);
  return vecMean(v, 0, v.size());
}

/* compute mean from sum and sum of squares of set of values */
double
varianceFromSum(double sum, double sumSquares, unsigned int n);

/** return variance (sigma squared) value of elements
  <x**2> - <x>**2 with correction factor */
template<class T>
inline
double
vecVariance(const Vec<T>& v)
{
  PRECOND(v.size() > 0);
  double sum = 0.0;
  double sumSquares = 0.0;
  for (unsigned int i = 0; i < v.size(); ++i) {
    sumSquares += static_cast<double>(v[i])
         * static_cast<double>(v[i]);
    sum += static_cast<double>(v[i]);
    
  }
  double result = varianceFromSum(sum, sumSquares, v.size());
  POSTCOND(result >= 0.0);
  return result;
}

/** return mean value of elements */
template<class T>
inline
double
vecMeanMinusDev(const Vec<T>& v)
{
  PRECOND(v.size() > 0);
  double mean = vecMean(v, 0, v.size());
  double stddev = sqrt(vecVariance(v)/v.size());
  if (fabs(mean) <= stddev) {
    mean = 0.0;
  }
  else if (mean > 0.0) {
    mean -= stddev;
  }
  else {
    mean += stddev;
  }
  return mean;
}


/** 
    transform scores to z-scores: measure is 
    deviation from mean in terms of its standard deviation
*/
void transformToZScores(Vec<double>& v);

/** 
    transform scores to z-scores: measure is 
    deviation from mean in terms of its standard deviation
*/
void transformToZScores(Vec<Vec<double> >& m);

/** 
    transform scores to z-scores: measure is 
    deviation from mean in terms of its standard deviation
*/
void
transformToZScores(Vec<double>& v, double minSigma);

/** 
    transform scores to z-scores: measure is 
    deviation from mean in terms of its standard deviation
    use only subset indices for computing mean and std
*/
void
transformToZScores(Vec<double>& v,
		   const Vec<unsigned int>& subSet);

/** multiply all entries of matrix with this factor */
void
matrixScalarMul(Vec<Vec<double> >& matrix, double val);

/** b = m a */
void
matrixVectorMul(const Vec<Vec<double> >& m,
		const Vec<double>& a,
		Vec<double>& b);

/** b = a(T) b */
void
vectorMatrixMul(const Vec<Vec<double> >& m,
		const Vec<double>& a,
		Vec<double>& b);

/** standard matrix multiplication. c must already have correct dimensions */
void 
matrixMul(const Vec<Vec<double> >& a,
	  const Vec<Vec<double> >& b,
	  Vec<Vec<double> >& c);

/** non-standard element-wise multiplication. All matrices must have same dimensions. */
void 
matrixElementMul(const Vec<Vec<double> >& a,
		 const Vec<Vec<double> >& b,
		 Vec<Vec<double> >& c);

/** transpose matrix even if not square matrix */
template <class T>
inline
Vec<Vec<T> >
matrixTranspose2(const Vec<Vec<T> >& m1)
{
  unsigned int m = m1.size(); 
  ERROR_IF(m == 0, "Matrix with zero elements encountered.");
  unsigned int n = m1[0].size(); 
  Vec<T> newRow(m);
  Vec<Vec<T> > m2(n, newRow);
  for (unsigned int i = 0; i < m1.size(); ++i) {
    for (unsigned int j = 0; j < m1[i].size(); ++j) {
      ASSERT((j < m2.size())&& (i < m2[j].size()));
      m2[j][i] = m1[i][j];
    }
  }
  return m2;
}


/** transpose matrix itself. Works only for square matrices.  */
void
matrixTranspose(Vec<Vec<double> >& m);

/** set matrix elements to one, if i=j, zero otherwise
 */
void
unitMatrix(Vec<Vec<double> >& matrix);

/* return m x n matrix */
template<class T>
inline
Vec<Vec<T> > 
generate2DMatrix(unsigned int m, unsigned int n, const T& val)
{
  Vec<T> row(n, val);
  return Vec<Vec<T> > (m, row);
}

/* return m x n matrix */
template<class T>
inline
Vec<Vec<T> > 
generate3DMatrix(unsigned int n1, unsigned int n2, unsigned int n3, const T& val)
{
  Vec<T> row(n3, val);
  Vec<Vec<T> > field(n2, row);
  return Vec<Vec<T> > (n1, field);
}

/* generate matrix with distances of vectors */
inline
Vec<Vec<double> >
generateDistanceMatrix(const Vec<Vector3D>& v)
{
  Vec<Vec<double> > field = generate2DMatrix(v.size(), v.size(), 0.0);
  for (unsigned i = 0; i < v.size(); ++i) {
    field[i][i] = 0.0;
    for (unsigned j = 0; j < i; ++j) {
      field[i][j] = vecDistance(v[i],v[j]);
      field[j][i] = field[i][j];
    }
  }
  return field;
}

template<class T>
inline
unsigned int 
countElements(const Vec<Vec<T> >& m) {
  unsigned int result = 0;
  for (unsigned int i = 0; i < m.size(); ++i) {
    result += m[i].size();
  }
  return result;
}

/** translates matrix into 1D vector */
template<class T>
inline
Vec<T>
matrix2Vec(const Vec<Vec<T> >& m) {
  unsigned int numElements = countElements(m);
  Vec<T> v(numElements);
  unsigned int pc = 0;
  for (unsigned int i = 0; i < m.size(); ++i) {
    for (unsigned int j = 0; j < m[i].size(); ++j) {
      v[pc++] = m[i][j];
    }
  }
  return v;
}

/** return bin of histogram */
int getBin(double val, double min, double delta);

/** return histogram: vector of bins filled with counts */
Vec<unsigned int> computeHistogram(const Vec<double>& data,
	          double min, double delta, unsigned int numBins);

/** return historgramm: vector of bins filled with y values belonging
 to this bin */
Vec<Vec<double> > computeScatterHistogram(const Vec<double>& datax,
	  const Vec<double>& datay, double min, double delta, 
					  unsigned int numBins);

/** return histogram: vector of bins filled with counts */
Vec<double> computeDoubleHistogram(const Vec<double>& data,
	          double min, double delta, unsigned int numBins);

/** return histogram: vector of bins filled with counts */
Vec<double> 
computeFrequency(const Vec<double>& data,
		 double min, double delta, unsigned int numBins);

/**
 * perform sum (integral) over histogram data
 */
template <class T>
inline
Vec<T>
accumulation(const Vec<T>& v) {
  Vec<T> result(v.size());
  T sum = 0;
  for (unsigned int i = 0; i < v.size(); ++i) {
    sum = sum + v[i];
    result[i] = sum;
  }
  return result;
}

/** reads matrix in format that is also used by program "R" */
inline
Vec<Vec<double> >
readPlainMatrix(istream& is)
{
  Vec<Vec<double > > result;
  while (is) {
    string line = getLine(is);
    Vec<string> words = getTokens(line);
    if (words.size() > 0) {
      Vec<double> row(words.size());
      for (unsigned int i = 0; i < words.size(); ++i) {
	row[i] = stod(words[i]);
      }
      result.push_back(row);
    }
  }
  return result;
}

/** compute uniqe set from original set
    result is also sorted */
template <class T>
inline
Vec<T>
uniqueSet(const Vec<T>& vOrig)
{
  Vec<T> result;
  if (vOrig.size() == 0) {
    return result;
  }
  Vec<T> v = vOrig;
  sort(v.begin(), v.end());
  result.push_back(v[0]);
  for (unsigned int i = 1; i < v.size(); ++i) {
    if (!(v[i] == v[i-1])) {
      result.push_back(v[i]);
    }
  }
  // POSTCOND(!containsDuplicates(result));
  return result;
}

/** compute uniqe set from original set */
bool
containsDuplicates(const Vec<unsigned int>& vOrig);

/** get subset of set */
template <class T>
inline
Vec<T>
getSubset(const Vec<T>& v, const Vec<unsigned int>& m)
{
  PRECOND(!containsDuplicates(m));
  Vec<T> result(m.size());
  for (unsigned int i = 0; i < m.size(); ++i) {
    ASSERT(m[i] < v.size());
    result[i] = v[m[i]];
  }
  return result;
}

/** get subset of set */
template <class T>
inline
Vec<T>
getSubset(const Vec<T>& v, const Vec<int>& m)
{
  Vec<T> result(m.size());
  for (unsigned int i = 0; i < m.size(); ++i) {
    ASSERT(static_cast<unsigned int>(m[i]) < v.size());
    result[i] = v[m[i]];
  }
  return result;
}

/** get subset of set, ignore invalid indices of m */
template <class T>
inline
Vec<T>
getValidSubset(const Vec<T>& v, const Vec<int>& m)
{
  Vec<T> result;
  for (unsigned int i = 0; i < m.size(); ++i) {
    if ((m[i] >= 0) && (m[i] < static_cast<int>(v.size()))) {
      result.push_back(v[m[i]]);
    }
  }
  return result;
}

/** return all emelents between 0 and maxSize, which are not part of m */
template <class T>
inline
Vec<unsigned int>
getAntiSet(unsigned int maxSize, const Vec<unsigned int>& m)
{
  Vec<T> result;
  for (unsigned int i = 0; i < maxSize; ++i) {
    bool found = false;
    for (unsigned int j = 0; j < m.size(); ++j) {
      if (i == m[j]) {
	found = true;
	break;
      }
    }
    if (!found) {
      result.push_back(i);
    }
  }
  return result;
}

/* deprecated, used removeFromSet instead */
template <class T>
inline
Vec<T>
getRemainingSet(const Vec<T>& m, const Vec<T>& subset)
{
  Vec<T> result;
  for (unsigned int i = 0; i < m.size(); ++i) {
    bool found = false;
    for (unsigned int j = 0; j < subset.size(); ++j) {
      if (m[i] == subset[j]) {
	found = true;
	break;
      }
    }
    if (!found) {
      result.push_back(m[i]);
    }
  }
  return result;
}

/** get subset of matrix (subset 2,5,13 will get you 3x3 matrix consisting of those columns and rows of the original matrix) */
template <class T>
inline
Vec<Vec<T> >
getMatrixSubset(const Vec<Vec<T> >& v, const Vec<unsigned int>& m)
{
  Vec<Vec<T> > result = getSubset(v, m);
  for (unsigned int i = 0; i < result.size(); ++i) {
    result[i] = getSubset(result[i], m);
  }
  return result;
}

/** get subset of matrix (subset 2,5,13 will get you 3x3 matrix consisting of those columns and rows of the original matrix) */
template <class T>
inline
Vec<Vec<T> >
getMatrixSubset(const Vec<Vec<T> >& v, const Vec<int>& m)
{
  Vec<Vec<T> > result = getSubset(v, m);
  for (unsigned int i = 0; i < result.size(); ++i) {
    result[i] = getSubset(result[i], m);
  }
  return result;
}

/** get subset of set of v, with all element removed which correspond to the indices of m 
 * @todo  : very slow implementation!
 */
template <class T>
inline
Vec<T>
removeFromSet(const Vec<T>& v, const Vec<unsigned int>& m)
{
  Vec<T> result;
  for (unsigned int i = 0; i < v.size(); ++i) {
    bool found = false;
    for (unsigned int j = 0; j < m.size(); ++j) {
      if (i == m[j]) {
	found = true;
	break;
      }
    }
    if (!found) {
      result.push_back(v[i]);
    }
  }
  return result;
}

/** get subset of set of v, with all element removed which correspond to the indices of m 
 * @todo  : very slow implementation!
 */
template <class T>
inline
Vec<T>
removeElement(const Vec<T>& v, unsigned int m)
{
  return removeFromSet(v, Vec<unsigned int>(1, m));
}

/** get subset of set v, which is NOT part of m*/
template <class T>
inline
Vec<T>
getExcludingSubset(const Vec<T>& v, const Vec<T>& m)
{
  Vec<T> result;
  for (unsigned int i = 0; i < v.size(); ++i) {
    bool found = false;
    for (unsigned int j = 0; j < m.size(); ++j) {
      // static_cast<int>(i)) { // BUGFIX November 2002!
      if (m[j] == v[i]) { 
	found = true;
	break;
      }
    }
    if (!found) {
      result.push_back(v[i]);
    }
  }
  return result;
}

/* compute common subset of two sets.
   not unique list, if input lists are not unique! */
Vec<unsigned int>
commonSubset(const Vec<unsigned int>& lOrig1,
	     const Vec<unsigned int>& lOrig2 );

/* compute unique union of two sets */
Vec<unsigned int>
uniqueUnion(const Vec<unsigned int>& l1,
	    const Vec<unsigned int>& l2);

/** vector with probabilities of different entities.
    values must be between 0 and 1
*/
double
conservation(const Vec<double>& v);

/* return 3D matrix from Euler angles
   rotation around x-axis: w
   rotation around y-axis: a
   rotation around z-axis: k
   taken from: Ronald Azuma 1991 (collinear.pdf). Check again from other source!!!
*/
Matrix3D
computeEulerRotationMatrix(double w, double a, double k);

/* find minimum x, y, z coordinates */
Vector3D
findMinCoordinates(const Vec<Vector3D>& vv);

/* find maximum x, y, z coordinates */
Vector3D
findMaxCoordinates(const Vec<Vector3D>& vv);

/* return index of v, which is furthest to pos */
unsigned int
findMaxDistanceIndex(const Vector3D& pos,
		     const Vec<Vector3D>& v);

/* return index of v, which is closest to pos */
unsigned int
findMinDistanceIndex(const Vector3D& pos,
		     const Vec<Vector3D>& v);

/** return coordinates which are in the middle of
    min and max values for each dimension */
Vector3D
computeCenterOfCoordinates(const Vec<Vector3D>& v);

/** return center of mass. Do not take weights into account! */
Vector3D
computeCenterOfMass(const Vec<Vector3D>& v);

/** return center of mass. */
Vector3D
computeCenterOfMass(const Vec<Vector3D>& v,
		    const Vec<double> weights);

/** see Goldstein, Classical Mechanics Ch. "The inertia tensor and the
    moment of inertia */
Matrix3D
computeGyrationMatrix(const Vec<Vector3D>& v,
		      const Vec<double> weights);

/** see Goldstein, Classical Mechanics Ch. "The inertia tensor and the
    moment of inertia */
void
computeGyrationRadiusGeometry(const Vec<Vector3D>& v,
			      const Vec<double> weights,
			      Vector3D& eigenValues,
			      Matrix3D& eigenVectors);

/** compute root mean square deviation of two sets of coordinate vectors */
double
rms(const Vec<Vector3D>& v1, const Vec<Vector3D>& v2);

/** 
    Move centroid of Vec <Vector3D> to ( 0, 0, 0)
  */
Vector3D moveCenter( Vec <Vector3D> &v1 );

/** compute optimal superposition of coordinates of
    molecules m1 and m2
    m2' = (m2-center(m2)) * rot + v
*/
void
minRmsSuperpose(const Vec<Vector3D>& v1Orig, Vec<Vector3D>& v2, 
		Matrix3D& rot, Vector3D& v);

/** compute optimal superposition of coordinates of
    molecules m1 and m2
    m2' = (m2-oldCenter) * rot + v
*/
void
minRmsSuperpose(const Vec<Vector3D>& v1Orig, Vec<Vector3D>& v2, 
		Matrix3D& rot, Vector3D& c1, Vector3D& c2);

/** compute optimal superposition of coordinates of
    molecules m1 and m2
*/
void
minRmsSuperpose(const Vec<Vector3D>& v1, Vec<Vector3D>& v2);

/** compute optimal superposition of coordinates of
    molecules m1 and m2
*/
inline
double 
minRms(const Vec<Vector3D>& v1, const Vec<Vector3D>& v2){
  Vec<Vector3D> vHelp = v2;
  minRmsSuperpose(v1, vHelp);
  return rms(v1, vHelp);
}


/** return connected patch. Dist is maximum distance between
    adjacent vectors */
Vec<unsigned int>
getConnected(const Vec<Vector3D> v, double dist, unsigned int start);


/** sort vector of elements
    but also return the new rank order as Vec<unsigned int> 
*/
/*
  template <class T>
  inline
  Vec<unsigned int>
  documentedSort(Vec<T>& v)
  {
  return documentedSort2(v); // faster version!!!
  //   Vec<unsigned int> order(v.size());
//   for (unsigned int i = 0; i < order.size(); ++i) {
//     order[i] = i;
//   }
//   bool sorted = false;
//   while (!sorted) {
//     sorted = true;
//     for (unsigned int i = 1; i < v.size(); ++i) {
//       if (v[i] < v[i-1]) {
// 	swap(v[i], v[i-1]);
// 	swap(order[i], order[i-1]);
// 	sorted = false;
//       }
//     }
//   }
//   return order;
}
*/

/**
 * generates loopup table: i'th element of result is position of value of i in v
 */
Vec<unsigned int>
getIndexSet(const Vec<unsigned int>& v);

/**
 * exchanges content of elements n and m of vector 
 */
template <class T>
inline
void
swapElements(Vec<T>& v, unsigned int n, unsigned int m)
{
  PRECOND((n < v.size()) && (m < v.size()));
  T help = v[n];
  v[n] = v[m];
  v[m] = help;
} 

template <class T>
bool
isRectangle(const Vec<Vec<T> >& v) 
{
  if (v.size() == 0) {
    return false;
  }
  unsigned int sOrig = v[0].size();
  for (unsigned int i = 0; i < v.size(); ++i) {
    if (v[i].size() != sOrig) {
      return false;
    }
  }
  return true;
}

/** sort vector of elements of v (sorry, only bubble sort)
    but also return the new rank order as Vec<unsigned int>
    also sorts element of w ACCORDING TO ORDER OF v !
*/
/*
template <class T, class S>
inline
Vec<unsigned int>
parallelSort(Vec<T>& v, Vec<S>& w)
{
  return parallelSort2(v, w); // overwrite with faster version!
    Vec<unsigned int> order(v.size());
    for (unsigned int i = 0; i < order.size(); ++i) {
    order[i] = i;
    }
    bool sorted = false;
    while (!sorted) {
    sorted = true;
    for (unsigned int i = 1; i < v.size(); ++i) {
    if (v[i] < v[i-1]) {
    swap(v[i], v[i-1]);
    swap(order[i], order[i-1]);
    swap(w[i], w[i - 1]);
    sorted = false;
    }
    }
    }
    return order;
}
*/

/** sort vector of elements of v (sorry, only bubble sort)
    but also return the new rank order as Vec<unsigned int>
    also sorts element of w ACCORDING TO ORDER OF v !
*/
/*
template <class T, class S>
inline
Vec<unsigned int>
parallelUnaffectedSort(const Vec<T>& vOrig, Vec<S>& w)
{
  return parallelUnaffectedSort2(vOrig, w); // overwrite with faster version!
  Vec<T> v = vOrig;
  Vec<unsigned int> order(v.size());
  for (unsigned int i = 0; i < order.size(); ++i) {
    order[i] = i;
  }
  bool sorted = false;
  while (!sorted) {
    sorted = true;
    for (unsigned int i = 1; i < v.size(); ++i) {
      if (v[i] < v[i-1]) {
	swap(v[i], v[i-1]);
	swap(order[i], order[i-1]);
	swap(w[i], w[i - 1]);
	sorted = false;
      }
    }
  }
  return order;
}
*/

/** sort vector of elements (fast!)
    but also return the old rank order as Vec<unsigned int> 
*/
template <class T>
inline
Vec<unsigned int>
documentedSort(Vec<T>& v)
{
  PRECOND(v.size() > 0);
  // RankedSolution4<T,unsigned int> t;
  Vec<RankedSolution4<T, unsigned int> > order(v.size());
  for (unsigned int i = 0; i < order.size(); ++i) {
    order[i] = RankedSolution4<T,unsigned int>(v[i], i);
  }
  sort(order.begin(), order.end());
  Vec<unsigned int> resultOrder(v.size());
  for (unsigned int i = 0; i < v.size(); ++i) {
    resultOrder[i] = order[i].second;
    v[i] = order[i].first;
  }
  return resultOrder;
}

/** sort vector of elements (fast!)
    but also return the old rank order as Vec<unsigned int> 
*/
template <class T>
inline
Vec<unsigned int>
documentedOrder(const Vec<T>& v)
{
  PRECOND(v.size() > 0);
  // RankedSolution4<T,unsigned int> t;
  Vec<RankedSolution4<T, unsigned int> > order(v.size());
  for (unsigned int i = 0; i < order.size(); ++i) {
    order[i] = RankedSolution4<T,unsigned int>(v[i], i);
  }
  sort(order.begin(), order.end());
  Vec<unsigned int> resultOrder(v.size());
  for (unsigned int i = 0; i < v.size(); ++i) {
    resultOrder[i] = order[i].second;
    // v[i] = order[i].first;
  }
  return resultOrder;
}


/** sort vector of elements of v 
    but also return the new rank order as Vec<unsigned int>
    also sorts element of w ACCORDING TO ORDER OF v !
*/
template <class T, class S>
inline
Vec<unsigned int>
parallelSort(Vec<T>& v, Vec<S>& w)
{
  PRECOND(v.size() == w.size());
  Vec<unsigned int> order = documentedSort(v);
  // cout << "The input list is: " << v << " and " << w << endl << "order: " << order << endl;
  Vec<S> wNew(w.size());
  for (unsigned int i = 0; i < order.size(); ++i) {
    wNew[i] = w[order[i]];
  }
  w = wNew;
  // cout << "Sorted list: " << v << " parallel: " << w << endl;
  return order;
}

/** sort vector of elements of v (sorry, only bubble sort)
    but also return the new rank order as Vec<unsigned int>
    also sorts element of w ACCORDING TO ORDER OF v !
*/
template <class T, class S>
inline
Vec<unsigned int>
parallelUnaffectedSort(const Vec<T>& vOrig, Vec<S>& w)
{
  PRECOND(vOrig.size() == w.size());
  Vec<T> v = vOrig;
  return parallelSort(v, w);
}

unsigned int findLargestDistanceIndex(const Vector3D& v, 
				      const Vec<Vector3D>& data);

unsigned  int findSmallestDistanceIndex(const Vector3D& v, 
					const Vec<Vector3D>& data);

/* find smallest distance point not belonging to forbidden set */
unsigned int findSmallestDistanceIndex(const Vector3D& v, 
				       const Vec<Vector3D>& data,
				       const Vec<unsigned int>& forbidden);

/* find n closest points */
Vec<unsigned int>
findSmallestDistanceIndices(const Vector3D& v, 
			    const Vec<Vector3D>& data,
			    unsigned int n);

/* find n closest points */
Vec<unsigned int>
findSmallestDistanceIndices(const Vector3D& v, 
			    const Vec<Vector3D>& data,
			    unsigned int n,
			    double minRadius,
			    double& maxRadius);

/* return distance in indices of pair of vectors with shortest distance */
double
findSmallestDistance(const Vec<Vector3D>& v1,
		     const Vec<Vector3D>& v2,
		     unsigned int& n1,
		     unsigned int& n2);

/** return point of m, which are closer than limitDist 
    to subset sub, but do not belong to sub 
*/
Vec<unsigned int>
getClosePoints(const Vec<unsigned int>& sub, const Vec<Vector3D>& m,
	       double limitDist);

/** return point of m2, which are closer than limitDist to m1
*/
Vec<unsigned int>
getClosePoints(const Vec<Vector3D>& m1, const Vec<Vector3D>& m2,
	       double limitDist);

unsigned int
centralPoint(const Vec<Vector3D>& choose, const Vec<Vector3D>& data,
	     double& radius);


/** return minimum element of vector */
template <class T>
inline
T findMin(const Vec<T>& v)
{
  PRECOND(v.size() > 0);
  T best = v[0];
  for (unsigned int i = 0; i < v.size(); ++i) {
    if (v[i] < best) {
      best = v[i];
    }
  }
  return best;
}

/** return index of minimum element of vector */
template <class T>
inline
unsigned int
findMinIndex(const Vec<T>& v)
{
  PRECOND(v.size() > 0);
  unsigned int iBest = 0;
  T best = v[iBest];
  for (unsigned int i = 0; i < v.size(); ++i) {
    if (v[i] < best) {
      best = v[i];
      iBest = i;
    }
  }
  POSTCOND(iBest < v.size());
  return iBest;
}

/** return index of maximum element of vector */
template <class T>
inline
unsigned int
findMaxIndex(const Vec<T>& v)
{
  PRECOND(v.size() > 0);
  unsigned int iBest = 0;
  T best = v[iBest];
  for (unsigned int i = 0; i < v.size(); ++i) {
    if (v[i] > best) {
      best = v[i];
      iBest = i;
    }
  }
  POSTCOND(iBest < v.size());
  return iBest;
}

/** return index of maximum element of vector */
template <class T>
inline
void
findMaxIndices(const Vec<Vec<T> >& v, unsigned int& i1, unsigned int& i2)
{
  PRECOND((v.size() > 0)&&(v[0].size()>0));
  i1 = 0;
  i2 = 0;
  T best = v[i1][i2];
  for (unsigned int i = 0; i < v.size(); ++i) {
    for (unsigned int j = 0; j < v[i].size(); ++j) {
      if (v[i][j] > best) {
	best = v[i][j];
	i1 = i;
	i2 = j;
      }
    }
  }
}

/** return indeces of elements equal to elem */
template <class T>
inline
Vec<unsigned int>
findEqualIndexSet(const Vec<T>& v, const T& elem)
{
  Vec<unsigned int> result;
  for (unsigned int i = 0; i < v.size(); ++i) {
    if (v[i] == elem) {
      result.push_back(i);
    }
  }
  return result;
}

/** return first index of elements equal to elem */
template <class T>
inline
unsigned int
findFirstIndex(const Vec<T>& v, const T& elem)
{
  for (unsigned int i = 0; i < v.size(); ++i) {
    if (v[i] == elem) {
      return i;
    }
  }
  return v.size();
}

// find index, with value closes to x
inline
unsigned int
findClosestIndex(const Vec<double>& v, double x)
{
  PRECOND(v.size() > 0);
  unsigned int idxBest = 0;
  double bestVal = fabs(v[0] - x);
  for (unsigned int i = 1; i < v.size(); ++i) {
    double d = fabs(v[i] - x); 
    if (d < bestVal) {
      idxBest = i;
      bestVal = d; 
    }
  }
  return idxBest;
}
 
/** 
 * add element to vector if it is not yet part of the vector
 */
template<class T>
inline
void
push_back_ifuniq(Vec<T>& v, const T& elem) {
  unsigned int n = findFirstIndex(v, elem);
  if (n >= v.size()) {
    v.push_back(elem);
  }
}

/** 
 * add element to vector if it is not yet part of the vector
 */
template<class T>
inline
void
push_front(Vec<T>& v, const T& elem) {
  v.push_back(elem); // just for allocating space for new element
  if (v.size() == 1) {
    return;
  }
  // shift all elements to one higher index:
  for (unsigned int i = v.size() - 1; i > 0; --i) {
    v[i] = v[i - 1];
  } 
  // now set first element to "elem":
  v[0] = elem;
}

/** see Goldstein, Classical Mechanics Ch. "The inertia tensor and the
    moment of inertia */
/*
void
computeGyrationGeometry(const Vec<Vector3D>& v,
			const Vec<double> weights,
			Vec<double>& eigenValues,
			Vec<Vec<double> >& eigenVectors);
*/

/* return radius of gyration of set of vectors */
double
computeRadiusOfGyration(const Vec<Vector3D>& v);

Vec<Vec<double> >
translateMatrix3DToVec(const Matrix3D& m);

/** internal amino acid counting starts from zero, external from ONE */
template <class T>
void
convert2InternalCounting(Vec<T>& v)
{
  for (unsigned int i = 0; i < v.size(); ++i) {
    if (v[i] > 0) {
      v[i] = v[i] - 1;
    }
    else {
      v[i] = 9999;
    }
  }
}

/** internal amino acid counting starts from zero, external from ONE */
template <class T>
void
convert2ExternalCounting(Vec<T>& v)
{
  for (unsigned int i = 0; i < v.size(); ++i) {
    v[i] = v[i] + 1;
  }
}

// convert internal residues counting (starting from zero)
// to external counting (starting from one)
template <class T>
Vec<T>
externalCounting(const Vec<T>& residues)
{
  Vec<T> result(residues.size());
  for (unsigned int i = 0; i < residues.size(); ++i) {
    result[i] = residues[i] + 1;
  }
  return result;
}

/* generate set of n points with radius r around origin vector, on a ring defined by theta */
Vec<Vector3D>
generateRingPoints(unsigned int n, double r, double theta, const Vector3D& origin);

/* generate set of n points with radius r around origin vector on a ring defined by theta */
Vec<Vector3D>
generateRingPoints(unsigned int n, double r, double theta, const Vector3D& origin, 
		   Vector3D direction);

/** generate set of n points with radius r around origin vector */
Vec<Vector3D>
generateSpherePoints(unsigned int n, double r, const Vector3D& origin);

/* generate surface points of point set */
void
generateSurfacePoints(Vec<Vector3D>& mol, const Vec<Vector3D>& molProt, 
		      double r, unsigned int tessel, bool rawMode);

/* generate surface points of point set
   result: vector of size molProt, each subvector
   containing surface vectors belonging to that atom */
Vec<Vec<Vector3D> >
generateSurfacePoints(const Vec<Vector3D>& molProt, 
		      double r, unsigned int tessel, bool rawMode);

/* generate surface points of point set */
void
generateSurfacePoints(Vec<Vector3D>& mol, Vec<int>& newIdVec,
      const Vec<Vector3D>& molProt, const Vec<int>& idVec, 
      double r, unsigned int tessel, bool rawMode);

/* generate surface points of point set */
void
generateSurfacePoints(Vec<Vector3D>& mol,
     Vec<int>& newIdVec, Vec<Vector3D>& directions,
     const Vec<Vector3D>& molProt, 
     const Vec<int>& idVec,
		      double r, unsigned int tessel, bool rawMode);

/* generate surface points of point set */
void
generateSurfacePointsAccessibleSurface(Vec<Vector3D>& mol,
     Vec<int>& newIdVec, Vec<Vector3D>& directions,
     Vec<double>& accessSurface,				       
     const Vec<Vector3D>& molProt, 
     const Vec<int>& idVec,
     double r, unsigned int tessel, bool rawMode);

/* generate compute accessible surface for point set */
void
computeAccessibleSurface(const Vec<Vector3D>& molProt, 
			  Vec<double>& accessSurface,				       
			 double r, unsigned int tessel);

/* generate simplex of n+1 dimensions */
Vec<Vec<double> > generateSimplex(const Vec<double>& v, double step);

/* generate simplex of n+1 dimensions */
Vec<Vec<double> > generateSimplex(const Vec<double>& v, const Vec<double>& stepSizes);

inline
bool
isInBox(const Vector3D& p, const Vector3D& minVec, const Vector3D& maxVec)
{
  return ( (p.x() >= minVec.x()) && (p.x() <= maxVec.x()) 
	   && (p.y() >= minVec.y()) && (p.y() <= maxVec.y()) 
	   && (p.z() >= minVec.z()) && (p.z() <= maxVec.z()) );
}



// get k nearest neighours of query vector closer than maxDist 
Vec<unsigned int> getKNearestNeighbours(const Vec<Vector3D>& v, 
		unsigned int query, unsigned int maxN, double maxDist);

// get k nearest neighours of query vector closer than maxDist 
Vec<Vec<unsigned int> >
getKNearestNeighbours(const Vec<Vector3D>& v, unsigned int maxN,
		      double maxDist);

/* return non-linear weighting of value x */
double nonLinearWeighting(double x, double maxY, double x0, double scale, 
			  double skew = 0.0);

/* return 2 step non-linear weighting of value x */
double
nonLinearWeightingDouble(double x, double maxY1, double x01, double scale1, 
			 double maxY2, double x02, double scale2);

/* expects N SORTED x-values and N+1 y values defining a step function. */
double
stepFunction(double x, const Vec<double>& xVec, const Vec<double>& yVec);

/* compute true/false positives/negatives of scores, which if greater
   than cutOff, are predicted to belong to class classNum */
double computeMathews(const Vec<double>& scores, 
    const Vec<unsigned int>& trueClass, unsigned int classNum, double cutOff, 
    unsigned int& truePos, unsigned int& falsePos, 
    unsigned int& trueNeg, unsigned int& falseNeg);

/* compute true/false positives/negatives of scores, which if greater
   than cutOff, are predicted to belong to class classNum */
double
computeMathews(	unsigned int truePos, unsigned int falsePos, 
		unsigned int trueNeg, unsigned int falseNeg);


/* compute true/false positives/negatives of scores, which if greater
   than cutOff, are predicted to belong to class classNum */
double
computeMathews(const Vec<Vec<double> > & scores, 
	       const Vec<Vec<unsigned int> > & trueClass, 
	       unsigned int classNum, 
	       double cutOff,
	       unsigned int& truePos, 
	       unsigned int& falsePos, 
	       unsigned int& trueNeg, 
	       unsigned int& falseNeg);

/** write info about distribution */
template<class T>
inline
void
distributionInfo(ostream& os, const Vec<T>& v)
{
  if (v.size() == 0) {
    cout << "Vector is empty." << endl;
  }
  else {
    cout << "Elements: " << v.size() << " mean: " 
	 << vecMean(v) << " std dev: " << sqrt(vecVariance(v))
	 << " std dev(mean): " << sqrt(vecVariance(v)/v.size())
	 << " min: " << *min_element(v.begin(), v.end())
	 << " max: " << *max_element(v.begin(), v.end()) << endl;
  }
}

/** return lengths of 2D array */
template<class T>
inline
Vec<unsigned int>
getLengths(const Vec<Vec<T> >& field)
{
  Vec<unsigned int> result(field.size());
  for (unsigned int i = 0; i < field.size(); ++i) {
    result[i] = field[i].size();
  }
  return result;
}

/** return 1D array from 2D array */
template<class T>
inline
Vec<T>
flatten(const Vec<Vec<T> >& field)
{
  // compute total length:
  unsigned int n = 0;
  for (unsigned int i = 0; i < field.size(); ++i) {
    n += field[i].size();
  }
  Vec<T> result(n);
  unsigned int count = 0;
  for (unsigned int i = 0; i < field.size(); ++i) {
    for (unsigned int j = 0; j < field[i].size(); ++j) {
      ASSERT(count < result.size());
      result[count++] = field[i][j];
    }
  }
  return result;
}

/** return 1D array from 2D array 
 * @param idVec       : i'th element: stores from which row of the original structure the i'th element in result came
 * @param flatIndices : stores new flat indices or original structures
 * */
template<class T>
inline
Vec<T>
flatten(const Vec<Vec<T> >& field, Vec<unsigned int>& idVec,
        Vec<Vec<unsigned int> >& flatIndices)
{
  // compute total length:
  unsigned int n = 0;
  for (unsigned int i = 0; i < field.size(); ++i) {
    n += field[i].size();
  }
  Vec<T> result(n);
  idVec = Vec<unsigned int>(n, 0);
  flatIndices = Vec<Vec<unsigned int> >(field.size());
  unsigned int count = 0;
  for (unsigned int i = 0; i < field.size(); ++i) {
    flatIndices[i] = Vec<unsigned int>(field[i].size(), 0U);
    for (unsigned int j = 0; j < field[i].size(); ++j) {
      ASSERT(count < result.size());
      idVec[count] = i; // store id from which this element originated
      flatIndices[i][j] = count; // store new index
      result[count++] = field[i][j];
    }
  }
  POSTCOND(idVec.size() == result.size());
  POSTCOND(flatIndices.size() == field.size());
  return result;
}

/** return cut array of size n */
template<class T>
inline
Vec<T>
vectorCut(const Vec<T>& v,  unsigned int n)
{
  // cout << "Starting vectorCut!" << endl;
  if (n >= v.size()) {
    return v;
  }
  if (n == 0) {
    return Vec<T>();
  }
  Vec<T> result(n);
  for (unsigned int i = 0; i < n; ++i) {
    result[i] = v[i];
  }
  // cout <<  " ending vector cut!" <<endl;
  return result;
}

/** return number of elements */
template <class T>
inline
unsigned int
vecCount(const Vec<T>& v, const T& x, unsigned int a, unsigned int b)
{
  PRECOND(b <= v.size());
  unsigned int count = 0;
  for (unsigned int i = a; i < b; ++i) {
    if (v[i] == x) {
      ++count;
    }
  }
  return count;
}

/** return number of elements */
template <class T>
inline
unsigned int
matrixCount(const Vec<Vec<T> >& v, const T& x) {

  unsigned int count = 0;
  for (unsigned int i = 0; i < v.size(); ++i) {
    for (unsigned int j = 0; j < v[i].size(); ++j) {
      if (v[i][j] == x) {
	++count;
      }
    }
  }
  return count;
}

/* my version of "winner takes most" ranking */
double
computeWinnerTakesMostScore(const Vec<double>& vOrig, double maxVal);

/** output of histogram x axis */
void
outBins(ostream& os, double min, double delta, unsigned int numBins);

/** Writes matrix to output stream as rectangular space-separated block */
template<class T>
inline
void
writeMatrix(ostream& os, const Vec<Vec<T> > & matrix) {
  for (Vec<Vec<double> >::size_type i = 0; i < matrix.size(); ++i) {
    for (Vec<Vec<double> >::size_type j = 0; j < matrix[i].size(); ++j) {
      os << matrix[i][j];
      if ((j + 1) < matrix[i].size()) {
	os << " ";
      } else {
	os << endl;
      }
    }
  }
}

#ifdef COVARNA_CONCURRENT_VECTOR
/** Writes matrix to output stream as rectangular space-separated block */
template<class T>
inline
void
writeMatrix(ostream& os, const concurrent_vector<concurrent_vector<T> > & matrix) {
  for (concurrent_vector<concurrent_vector<double> >::size_type i = 0; i < matrix.size(); ++i) {
    for (concurrent_vector<concurrent_vector<double> >::size_type j = 0; j < matrix[i].size(); ++j) {
      os << matrix[i][j];
      if ((j + 1) < matrix[i].size()) {
	os << " ";
      } else {
	os << endl;
      }
    }
  }
}
#endif

/* after computation the values between the two indices are
   interpolated */
void
vecInterpolate(Vec<double>& v, unsigned int startidx, unsigned int endidx);

/* fill in missing values if borders by defined values.
   undefined is every v[i] with fabs(v[i] - empty) <= diff.
*/
void vecInterpolateMissing(Vec<double>& v, double empty, double diff);

template<class T>
inline
T
power(const T& a, const T& b)
{
  PRECOND(b >= 0);
  T result = 1;
  for (T i = 0; i < b; ++i) {
    result *= a;
  }
  return result;
}

// compute convolution of vector v with gauss Kernel exp(-x*x/d*d)
Vec<double>
smoothGaussian(const Vec<double>& v, double d);

Vec<double>
smoothGaussian(const Vec<double>& v, double d, unsigned int nMax);

void
smoothGaussian(Vec<Vec<Vec<double> > >& cube, double d, unsigned int nMax);

// increase number to a certain basis.
void
incBaseNumber(unsigned int& n, unsigned int base, bool& overflow);

// increase number to a certain basis. v[0] corresponds to lowest digit!
void
incBaseNumber(Vec<unsigned int>& v, unsigned int base);

// return maximum number which can be expressed by number with so many digits 
// to a certain base
unsigned int
maxBaseNumber(unsigned int digits, unsigned int base);

// sorts such that least populated columns are in front
template<class T>
void
sortBySize(Vec<Vec<T> >& m) {
  Vec<unsigned int> sizes(m.size());
  for (unsigned int i = 0; i < m.size(); ++i) {
    sizes[i] = m[i].size();
  }
  parallelSort(sizes, m);
#ifndef NDEBUG
  for (unsigned int i = 1; i < m.size(); ++i) {
    if (m[i].size() < m[i-1].size()) {
      cout << "Internal error: " << endl << sizes << endl << m << endl;
      ERROR("Internal error in sortBySize!");
    }
  }
#endif
}

/** makes matrix more similar to distance matrix: positive definite,
    and zero in diagonals */
void
distify(Vec<Vec<double> >& m);

inline
int
computeAntiDiagonalId(int nx, int ny, int sz) 
{
  int szm = sz - 1;
  while ((nx > 0) && (ny < szm)) {
    --nx;
    ++ny;
  }
  if (nx == 0) {
    return ny;
  }
  return sz + nx - 1;
}

/** n: id of antidiagonal, nx, ny: left hand start points */
inline
void
computeStartFromAntiDiagonalId(int n, int& nx, int& ny, int sz) 
{
  if (n < sz) {
    nx = 0;
    ny =  n;
  }
  else {
    nx = n + 1 - sz;
    ny =  sz - 1;
  }
}


/** get anti diagonal slice including element
  n < 2*size - 1
  n counts left left hand index if smaller of than size or
  the upper bar if larger than size
*/
template <class T>
inline
void
setAntiDiagonal(Vec<Vec<T> >& mat, const Vec<T>& diag, int n)
{
  int sz = mat.size(); 
  int x, y;
  ERROR_IF((n < 0) || (n >= ((2*sz)-1)),
	   "Illegal diagonal index!");
  computeStartFromAntiDiagonalId(n, x, y, sz);
  int len = diag.size();
  for (int i = 0; i < len; ++i) {
    ASSERT((x < static_cast<int>(mat.size())) 
	   && (y < static_cast<int>(mat[x].size())));
//     cout << "Setting " << x << " " << y << " from " 
// 	 << mat[x][y] << " to " << diag[i] << endl;
    mat[x++][y--] = diag[i];

  }
}

/** get anti diagonal slice including element
  n < 2*size - 1
  n counts left left hand index if smaller of than size or
  the upper bar if larger than size
*/
template <class T>
inline
Vec<T>
getAntiDiagonal(const Vec<Vec<T> >& mat, int n)
{
  int sz = mat.size(); 
  if ((n < 0) || (n >= ((2*sz)-1))) {
    return Vec<T>();
  }
  int len = sz - abs(static_cast<int>(sz - 1 - n));
  if (len == 0) {
    return Vec<T>();
  }
  Vec<T> result(len, 0.0);
  int srow, scol;
  if (n < sz) {
    srow =  n;
    scol = 0;
  }
  else {
    srow =  sz - 1;
    scol = n + 1 - sz;
  }
  int row = srow;
  int col = scol;
  for (int i = 0; i < len; ++i) {
    ASSERT((row < static_cast<int>(mat.size())) && (col < static_cast<int>(mat[row].size())));
    result[i] = mat[row--][col++];
  }
  return result;
}

/** get anti diagonal slice including element nx,ny
  n < 2*size - 1
*/
template <class T>
inline
Vec<T>
getAntiDiagonal(const Vec<Vec<T> >& mat, int nx, int ny)
{
  return getAntiDiagonal(mat, computeAntiDiagonalId(nx, ny, mat.size()));
}

/** get anti diagonal slice including element n,n
    counting: n varies between -size+1 ... to size -1
    0 corresponds to main diagonal. 
    n < 0 corresponds to row 0, col (-n),
    n > 0 corresponds to row n, col (0),
*/
template <class T>
inline
Vec<T>
getDiagonal(const Vec<Vec<T> >& mat, int n)
{
  int sz = mat.size(); 
  if ((n <= -sz) || (n >= sz)) {
    return Vec<T>();
  }
  int srow, scol;
  if (n < 0) {
    srow =  0;
    scol = -n;
  }
  else {
    srow =  n;
    scol = 0;
  }
  int len = sz - (scol + srow);
  if (len == 0) {
    return Vec<T>();
  }
  Vec<T> result(len, 0.0);
  int row = srow;
  int col = scol;
  for (int i = 0; i < len; ++i) {
    ASSERT((row < static_cast<int>(mat.size())) 
	   && (col < static_cast<int>(mat[row].size())));
    result[i] = mat[row++][col++];
  }
  return result;
}

/** returns index n,n from point m,n */
double
diagonalProjection(int row, int col, int size);

/** returns row index of projection to antidiagonal */
double
antiDiagonalProjection(int row, int col, int size);

Vec<double>
getAntiDiagonalAverages(const Vec<Vec<double> >& mat);

Vec<double>
getDiagonalAverages(const Vec<Vec<double> >& mat);

double
correlationCoefficient(const Vec<double>& x, const Vec<double>& y);

/** in each row, leave only the highest scoring element,
 also delete diagonal */
void
winnerTakesAll(Vec<Vec<double> >& matrix, int diagBorder, double cutoff);

/** sets all elements i,i to i,i+diagBorder */
void
fillDiagonal(Vec<Vec<double> >& matrix, int diagBorder, double value);

#endif /* __A_CLASS_H__ */


