#include <SequenceAlignmentIOTools.h>

char
SequenceAlignmentIOTools::getMsfSequenceType(const string& alphabet) {
  char alignmentType = 'N'; // either N for nucleotide of P for protein
  if (alphabet.size() > 10) {
    alignmentType = 'P'; // protein sequence
  }
  return alignmentType;
}

string
SequenceAlignmentIOTools::getMsfName(const SequenceAlignment& ali,
				     SequenceAlignment::size_type n) {
  string name = ali.getName(n);
  if (name.size() > MSF_NAME_MAX) {
    name = name.substr(0, MSF_NAME_MAX);
  }
  name = name + "/1-" + uitos(ali.getLength());
  return name;
}

string
SequenceAlignmentIOTools::getClustalName(const SequenceAlignment& ali,
					 SequenceAlignment::size_type n) {
  string name = ali.getName(n);
  if (name.size() > CLUSTAL_NAME_MAX) {
    name = name.substr(0, CLUSTAL_NAME_MAX);
  }
  return name;
}


string
SequenceAlignmentIOTools::getClustalEmptyName() {
  return string(CLUSTAL_NAME_MAX, ' ');
}

void
SequenceAlignmentIOTools::writeMsfHeader(ostream& os, const SequenceAlignment& ali, char sequenceType) {
  os << "!!NA_MULTIPLE_ALIGNMENT 1.0" << endl;
  os << endl;
  // TODO : check numbers not implemented
  os << "   MSF: " << ali.getLength() << "   Type: " << sequenceType << "    Check: 1234   .." << endl;
  os << endl;
  os << endl;
  for (SequenceAlignment::size_type i = 0; i < ali.size(); ++i) {
    os << "  Name: " << getMsfName(ali, i) << " Len: " << ali.getLength() << "  Check: 1234  Weight: 1.0" << endl;
  }
  os << endl;
  os << endl;
}

void
SequenceAlignmentIOTools::writeClustalHeader(ostream& os, const SequenceAlignment& ali) {
  os << "CLUSTAL alignedit2" << endl;
  os << endl << endl;
}

/** writes string with space character every 10 (stride) characters. */
void
SequenceAlignmentIOTools::writeStride(ostream& os,
				      const string& s, 
				      int stride) {
  for (string::size_type i = 0; i < s.size(); ++i) {
    if ((i > 0) && ((i % stride) == 0)) {
      os << " ";
    }
    os << s[i];
  }
}



void
SequenceAlignmentIOTools::writeMsfSequenceChunk(ostream& os, const SequenceAlignment& ali,
						SequenceAlignment::size_type n,
						SequenceAlignment::size_type start) {
  SequenceAlignment::size_type stop = start + MSF_CHUNK - 1;
  if (stop >= ali.getLength()) {
    stop = ali.getLength()-1;
  }
  SequenceAlignment::size_type len = stop - start + 1;
  string chunk = ali.getSequence(n).substr(start, len);
  os << getMsfName(ali, n) << " ";
  writeStride(os, chunk, MSF_STRIDE);
}

void
SequenceAlignmentIOTools::writeClustalSequenceChunk(ostream& os, const SequenceAlignment& ali,
						    SequenceAlignment::size_type n,
						    SequenceAlignment::size_type start) {
  SequenceAlignment::size_type stop = start + CLUSTAL_CHUNK - 1;
  if (stop >= ali.getLength()) {
    stop = ali.getLength()-1;
  }
  SequenceAlignment::size_type len = stop - start + 1;
  string chunk = ali.getSequence(n).substr(start, len);
  os << getClustalName(ali, n) << " ";
  os << chunk << " " << (stop+1);
}

void
SequenceAlignmentIOTools::writeClustalDummyChunk(ostream& os, const SequenceAlignment& ali,
						    SequenceAlignment::size_type start) {
  SequenceAlignment::size_type stop = start + CLUSTAL_CHUNK - 1;
  if (stop >= ali.getLength()) {
    stop = ali.getLength()-1;
  }
  SequenceAlignment::size_type len = stop - start + 1;
  string chunk(len, ':'); // ali.getSequence(n).substr(start, len);
  os << getClustalEmptyName() << " ";
  os << chunk << " " << stop;
}

void
SequenceAlignmentIOTools::writeMsfBody(ostream& os, const SequenceAlignment& ali) {
   for (SequenceAlignment::size_type pc = 0; pc < ali.getLength(); pc += MSF_CHUNK) {
     for (SequenceAlignment::size_type i = 0; i < ali.size(); ++i) {
       writeMsfSequenceChunk(os, ali, i, pc);
       os << endl;
     }
     os << endl;
   }
}


void
SequenceAlignmentIOTools::writeClustalBody(ostream& os, const SequenceAlignment& ali) {
  for (SequenceAlignment::size_type pc = 0; pc < ali.getLength(); pc += CLUSTAL_CHUNK) {
     for (SequenceAlignment::size_type i = 0; i < ali.size(); ++i) {
       writeClustalSequenceChunk(os, ali, i, pc);
       os << endl;
     }
     writeClustalDummyChunk(os, ali, pc);
     os << endl << endl;
   }
}


void
SequenceAlignmentIOTools::writeFastaSequence(ostream& os,
					   const SequenceAlignment& ali,
					   size_type n,
					   size_type blockLength)
{
  const string& seq = ali.getSequence(n);
  const string& name = ali.getName(n);
  os << ">" << name << endl;
  for (size_type i = 0; i < seq.size(); ++i) {
    if ((i > 0) && ((i % blockLength) == 0)) {
      os << endl;
    }
    os << seq[i];
  }
  os << endl;
}

void
SequenceAlignmentIOTools::writeFasta(ostream& os,
				   const SequenceAlignment& ali,
				   size_type blockLength)
{
  for (SequenceAlignment::size_type i = 0; i < ali.size(); ++i) {
    writeFastaSequence(os, ali, i, blockLength);
  }
}


/** writes IG/Standford format */
void
SequenceAlignmentIOTools::writeStanford(ostream& os,
					const SequenceAlignment& ali)
{
  os << ";stanford format sequence" << endl
     << "sequence-name" << endl;
  os << ali.getLength() << endl;
  for (SequenceAlignment::size_type i = 0; i < ali.size(); ++i) {
    writeFastaSequence(os, ali, i, 80);
  }
  os << "1" << endl; // pathetic format !
}
