
#include <iostream>
#include <debug.h>
#include <Vec.h>
#include <GetArg.h>

using namespace std;

typedef int int_type;

#define VARIABLE_STEP 1
#define FIXED_STEP 2


/* Converts "bedgraph" format into UCSC wiggle format.
 * Example:
browser position chr19:59304200-59310700
browser hide all
#150 base wide bar graph at arbitrarily spaced positions,
#threshold line drawn at y=11.76
#autoScale off viewing range set to [0:25]
#priority = 10 positions this as the first graph
#Note, one-relative coordinate system in use for this format

track type=wiggle_0 name="variableStep" description="variableStep format" visibility=full autoScale=off viewLimits=0.0:25.0 color=50,150,255 yLineMark=11.76 yLineOnOff=on priority=10 variableStep chrom=chr19 span=150
59304701 10.0
59304901 12.5
59305401 15.0
59305601 17.5
59305901 20.0
59306081 17.5
59306301 15.0
59306691 12.5
*/

void
writeVariableStep(ostream& os, 
		  const Vec<int_type>& starts,
		  const Vec<int_type>& ends,
		  const Vec<double>& scores,
		  const string& chromName) {
  cout << "browser position " << chromName << ":" << starts[0] << "-"
       << ends[ends.size()-1] << endl;
  cout << "browser hide all" << endl;
  int span = 1;
  cout << "track type=wiggle_0 name=\"variableStep\" description=\"variableStep format\" visibility=full autoScale=off viewLimits=0.0:25.0 color=50,150,255 yLineMark=11.76 yLineOnOff=on priority=10 variableStep chrom=" << chromName << " span=" << span << endl;
  cout << "variableStep chrom=" << chromName << " span=" << span << endl;
  for (Vec<int_type>::size_type i = 0; i < starts.size(); ++i) {
    if ((i > 0) && (starts[i] != ends[i-1])) {
      // there is a gap; fill with score zero:
      cout << ends[i-1] << " 0" << endl;
    }
    cout << starts[i] << " " << scores[i] << endl;
  }
  // make sure interval is drawn to end:
  cout << (ends[ends.size()-1] - 1) << " " << scores[scores.size()-1] << endl;
}

void
writeFixedStep(ostream& os, 
		  const Vec<int_type>& starts,
		  const Vec<int_type>& ends,
		  const Vec<double>& scores,
		  const string& chromName) {
  cout << "browser position " << chromName << ":" << starts[0] << "-"
       << ends[ends.size()-1] << endl;
  cout << "browser hide all" << endl;
  int span = 1;
  int step = 1;
  cout << "track type=wiggle_0 name=\"fixedStep\" description=\"fixedStep format\" visibility=full autoScale=off viewLimits=0.0:25.0 color=50,150,255 yLineMark=11.76 yLineOnOff=on priority=10 fixedStep chrom=" << chromName << " span=" << span << endl;
  for (Vec<int_type>::size_type i = 0; i < starts.size(); ++i) {
    cout << "fixedStep chrom=" << chromName 
	 << " start=" << (starts[i]+1) 
	 << " step=" << step
	 << " span=" << span << endl;
    for (int_type j = starts[i]; j < ends[i]; ++j) {
      cout << scores[i] << endl;
    }
  }
}

/** Main program */
int main(int argc, char ** argv) {
  int outputFormat = FIXED_STEP;
  string filename;
  Vec<int_type> starts;
  Vec<int_type> ends;
  Vec<double> scores;
  int_type discr = 1;
  int verbose = 0;
  string chromName;
  getArg("i", filename, argc, argv, filename);
  getArg("c", chromName, argc, argv,chromName);
  getArg("d", discr, argc, argv, discr);
  getArg("v", verbose, argc, argv, verbose);
  int_type val1, val2;
  double score;
  string name;

  while (cin) {
    cin >> name >> val1 >> val2 >> score;
    if (cin) {
      if ((chromName.size() == 0) || (name == chromName)) {
	starts.push_back(val1);
	ends.push_back(val2);
	scores.push_back(score);
      }
    }
  }
  if (verbose > 0) {
    cout << "# Start positions: " << starts;
    cout << "# End positions: " << ends;
  }
  switch (outputFormat) {
  case VARIABLE_STEP:
    writeVariableStep(cout, starts, ends, scores, chromName);
    break;
  case FIXED_STEP:
    writeFixedStep(cout, starts, ends, scores, chromName);
    break;
    ERROR("Unknown output format!");
  }
  if (verbose > 0) {
    cout << "# Good bye!" << endl;
  }
  return 0;
}


