# TPM method for combining p values

# factorial function
Fac <- function(n) {
  result <- 1
  if (n > 1) {
   result <- n * Fac(n-1)
  }
  result
}

# logarithm of factorial function
logFac.recursive <- function(n, base=exp(1)) {
  result <- 0
  if (n > 1) {
   result <- log(n, base) + logFac(n-1, base=base)
  }
  result
}


# logarithm of factorial function
logFac <- function(n, base=exp(1)) {
  result <- 0
  if (n > 1) {
   for (i in 2:n) {
    result <- result + log(i, base)
   }
  }
  result
}


test.Fac <- function() {
 stopifnot(Fac(3) == 6)
}

test.logFac <- function(n=16) {
 result <- logFac(n)
 checkEquals(exp(result), Fac(n))
 checkEquals(result, logFac.recursive(n))
}

test.Fac <- function() {
 stopifnot(exp(logFac(3)) == 6)
}

# N choose K function
NchK <- function(n, k) {
 exp(logFac(n) - ((logFac(k)+logFac(n-k))))
}

# Logarithm of N choose K function
LogNchK <- function(n, k, base=exp(1)) {
 logFac(n,base=base) - ((logFac(k, base=base)+logFac(n-k, base=base)))
}

test.NchK <- function() {
 stopifnot(NchK(3,2) == 2)
}

Ind <- function(b) {
 result <- FALSE
 if (b) {
  result <- TRUE
 }
 b
}

# helper function for TPM method
# L : number of p-values
# t: cutoff p-value (often 0.05 or 0.01) 
# w: truncated product value product value
# k: iteration variable
TPM_P_k <- function(w,L,k, t, verbose=FALSE) {
 if (! (w > 0)) {
   print(w)
   stop("Strange product value w encountered!")
 }
 stopifnot(t > 0)
 stopifnot(w > 0)
 stopifnot(L > 0)
 stopifnot(k > 0) 
 ot <- 1-t
 logTerm1 <- LogNchK(L,k)*(ot^(L-k) )
 stopifnot(is.finite(logTerm1))
 term2 <- 0
 tpk <- t^k
 lnT <- log(t)
 lnW <- log(w)   
 if (verbose) {
  cat("Started TPM_P_k with parameters", w,L,k,t, " logTerm1, term2:", logTerm1, term2, "\n")
 }
 if (w <= tpk) { 
  term3 <- (k*lnT) - lnW 
  for (s in 0:(k-1)) {
    if (term3 > 0) {
      term2 <- term2 + exp( ( s * log(term3 ) ) - logFac(s) )
    } else {
      term2 <- term2 + ( ( ( (k*lnT) - lnW )^s) / Fac(s) )
    }     
    if (verbose) {
      cat("Iteration ", s, "term2, term3:", term2, term3, "\n")
    }
  }
  term2 <- term2 * w 
 } else {
   term2 <- tpk
 }
 logTerm2 <- log(term2)
 result <- exp(logTerm1 + logTerm2)
 if (is.na(result)) {
  cat ("Encountered NA result in TPM_P_k with input parameters", w, L, k, t, "logTerm1, term2, logTerm2, tpk, result:", logTerm1, term2, logTerm2, tpk,
        result, "\n")
 }
 stopifnot(!is.na(result))
 result
}

# helper function for TPM method
# helper function for TPM method
# L : number of p-values
# t: cutoff p-value (often 0.05 or 0.01) 
# w: truncated product value product value
# k: iteration variable

TPM_P_k.old <- function(w,L,k, t) {
 ot <- 1-t
 term1 <- NchK(L,k)*(ot^(L-k) )
 term2 <- 0
 tpk <- t^k
 lnT <- log(t)
 lnW <- log(w)   
 for (s in 0:(k-1)) {
   if (w <= tpk) {
    term3 <- (k*lnT) - lnW 
    if (term3 > 0) {
      term2 <- term2 + exp( ( s * log(term3 ) ) - logFac(s) )
    } else {
     term2 <- term2 + ( ( ( (k*lnT) - lnW )^s) / Fac(s) )
    }     
   } else {
     term2 <- term2 + tpk
   }
 }
 term2 <- term2 * w
 result <- term1 * term2
 stopifnot(!is.na(result))
 result
}

# testing TPM_p_k function with large parameters (1070 p-values)
test.TPM_P_k <- function(w=9.756319e-07, L=1070, k=413, t=0.05, verbose=TRUE) {
 result <- TPM_P_k(w=w,L=L,k=k,t=t, verbose=verbose)
 result
}

# helper function for TPM method: compute probability of observing a certain truncated product
# helper function for TPM method
# L : number of p-values
# t: cutoff p-value (often 0.05 or 0.01) 
# w: truncated product value product value
TPM_Pr <- function(w,L, t, verbose=FALSE) {
 result <- 0
 if (w > 0) {
  for (k in 1:L) {
  term <- TPM_P_k(w,L,k,t)
  if (verbose) {
   cat("adding probability for k =", k, term, "\n") 
  }
  result <- result + term
  }
 }
 result
}

# helper function for TPM: compute truncated product
# pVec: vector of p-values
# t: cutoff P-value (often 0.05 or 0.01)
TPM_W <- function(pVec, t=0.05) {
  W = 1
  L <- length(pVec)
  for (i in 1:L) {
   if (pVec[i] <= t) {
     W <- W * pVec[i]
   } else {
    stopifnot(pVec[i] <= 1.0)
   }
  }
  W
}

# TPM method for combining P values in meta-analysis
# see Zaykin et al: Truncated Product Method for Combining p-values. Genet Epidemiol, 2002
computeCombinedPValuesTPM <- function (pVec, t=0.05, verbose=FALSE) {
 L <- length(pVec)
 if (L > 1) {
  w <- TPM_W(pVec, t)
  if (verbose) {
   cat("The truncated product value (w statistic) is", w, "\n")
  }
  if (w > 0) {
   result <- TPM_Pr(w, L, t, verbose=verbose)
  } else {
   result <- 0
  }
  stopifnot(result >= 0)
  if (result > 1.0) {
   result <- 1.0
  }
 } else if (L == 1) { # only one P-value provided
  result <- pVec[1]
 } else if (L == 0) { # no P-values to combine. Set resulting P-value to 1.
  result <- 1
 } else {
   stop("internal error in computeCombinedPValuesTPM")
 }
 result
}

test.computeCombinedPValuesTPM <- function(pVec=rep(0.05, 3), t=0.05) {
 computeCombinedPValuesTPM(pVec, t)
}

# testing TPM_Pr function with large parameters (1070 p-values)
test.TPM_Pr <- function(w=9.756319e-07, L=1070, t=0.05, verbose=TRUE) {
 result <- TPM_Pr(w=w,L=L,t=t, verbose=verbose)
 result
}

# testing TPM_W function with large parameters (1070 p-values)
test.TPM_W <- function(t=0.05, pVec=rep(t, 1070)) {
 result <- TPM_W(pVec,t=t)
 result
}

